<?php

use Carbon\Carbon;
use App\Models\City;
use App\Models\User;
use App\Utils\DPDApi;
use App\Models\Package;
use App\Models\SaleKpi;
use App\Models\Service;
use App\Models\Shipment;
use App\Utils\EtowerApi;
use App\Models\PackageSg;
use App\Utils\AusPostApi;
use App\Utils\ShipsaoApi;
use App\Models\SaleMember;
use App\Utils\DmxSmartApi;
use App\Models\HistoryScan;
use App\Models\SaleLinkFwd;
use Illuminate\Support\Str;
use App\Models\DiscountUser;
use App\Models\SurchargeGoods;
use App\Utils\TopLogisticsApi;
use App\Models\PackageTracking;
use App\Models\FwdCostByService;
use Illuminate\Support\Facades\DB;
use App\Models\HistoryEditShipment;
use App\Models\AssociateWithService;
use App\Models\ShipmentOperatingCost;
use App\Models\DiscountUserWeightLevel;
use App\Models\InitiallizeLabelHistory;
use App\Http\Controllers\ShipmentController;
use App\Models\ServiceSurchargeFee;

function getUrlTracking($tracking_code)
{
    return URL_TRACKING . "?code=" . $tracking_code;
}

function onShowLabel2ByAccount($user)
{
    try {
        $have_show_label_LF2 = false;
        $array_label_LF2 = [
            'Quyen@woeldlink-express.net',
            'ops.sgn.06@asiacargoexpress.com.vn',
        ];
        if (in_array($user['user_name'], $array_label_LF2)) {
            $have_show_label_LF2 = true;
        }

        return $have_show_label_LF2;
    } catch (\Throwable $th) {
        return false;
    }
}

function getTotalWeightCompaniesForSale($sale_id, $filters = null)
{
    try {
        $leader = SaleMember::isShow()
            ->where('user_id', $sale_id)
            ->leftJoin('sales_teams', 'sales_members.sale_team_id', 'sales_teams.sale_team_id')
            ->where('sales_teams.active_flg', ACTIVE)->where('sales_teams.delete_flg', NO_DELETE)
            ->select('leader_id', 'member_kind', 'sales_teams.sale_team_id')->first();

        $is_leader = false;

        if (isset($leader) && $leader->member_kind == SALE_LEADER) {
            $is_leader = true;
        }

        $fwd_costs = [];
        if ($leader) {
            $fwd_costs = FwdCostByService::isShow()->where('leader_id', $leader->leader_id)
                ->select('service_id', 'fwd_id', 'leader_cost', 'member_cost', 'leader_member_cost')
                ->get()->toArray();
        }

        $data = [
            'total_weight' => 0,
            'total_price' => 0,
        ];

        $fwd_link = SaleLinkFwd::isShow()
            ->where('sale_id', $sale_id)->pluck('fwd_id');
        if (count($fwd_link) < 1 && $sale_id != $leader->leader_id) {
            return $data;
        }

        foreach (getShipmentWeight($fwd_link, $filters) as $key => $value) {
            $weight = isset($value['total_charged_weight']) ? $value['total_charged_weight'] : 0;
            if (empty($fwd_costs)) {
                $data['total_price'] += ($is_leader ? 5000 : 3000) * $weight;
            } else {
                $check_isset = collect($fwd_costs)->where('service_id', $value['shipment_service_id'])
                    ->where('fwd_id', $value['user_id'])->first();
                if ($check_isset) {
                    $data['total_price'] += ($is_leader ? $check_isset['leader_cost'] : $check_isset['member_cost']) * $weight;
                } else {
                    $data['total_price'] += ($is_leader ? 5000 : 3000) * $weight;
                }
            }
            $data['total_weight'] += $weight;
        }

        if ($is_leader) {
            $members = SaleMember::isShow()->where('sale_team_id', $leader->sale_team_id)
                ->where('user_id', '!=', $leader->leader_id)->pluck('user_id');

            foreach (getShipmentWeight($members, $filters) as $key => $value) {
                $weight = isset($value['total_charged_weight']) ? $value['total_charged_weight'] : 0;
                if (empty($fwd_costs)) {
                    $data['total_price'] += 2000 * $weight;
                } else {
                    $check_isset = collect($fwd_costs)->where('service_id', $value['shipment_service_id'])
                        ->where('fwd_id', $value['user_id'])->first();
                    if ($check_isset) {
                        $data['total_price'] += $check_isset['leader_member_cost'] * $weight;
                    } else {
                        $data['total_price'] += 2000 * $weight;
                    }
                }
                $data['total_weight'] += isset($value['total_charged_weight']) ? $value['total_charged_weight'] : 0;
            }
        }

        return $data;
    } catch (\Throwable $th) {
        return [
            'total_weight' => 0,
            'total_price' => 0,
        ];;
    }
}

function getDataFwdCost($fwd_id, $service_id, $kind, $weight = 0)
{
    try {

        if (!isset($kind)) {
            $kind = SALE_MEMBER;
        }
        $fwd_costs = FwdCostByService::isShow()->where('fwd_id', $fwd_id)
            ->where('service_id', $service_id)->first();
        if ($fwd_costs) {
            return [
                'sale_cost' => $kind == SALE_LEADER ? $fwd_costs['leader_cost'] : $fwd_costs['member_cost'],
                'weight' => $weight,
                'leader_cost' => $fwd_costs['leader_cost'],
                'member_cost' => $fwd_costs['member_cost'],
                'leader_member_cost' => $fwd_costs['leader_member_cost'],
                'amount_weight' => $kind == SALE_LEADER ? $weight * $fwd_costs['leader_cost'] : $weight * $fwd_costs['member_cost'],
                'amount_weight_member' => $weight * $fwd_costs['leader_member_cost'],
            ];
        }
        return [
            'sale_cost' => $kind == SALE_LEADER ? 5000 : 3000,
            'weight' => $weight,
            'leader_cost' => 5000,
            'member_cost' => 3000,
            'leader_member_cost' => 2000,
            'amount_weight' => $kind == SALE_LEADER ? $weight * 5000 : $weight * 3000,
            'amount_weight_member' => $weight * 2000,
        ];
    } catch (\Throwable $th) {
        return [
            'sale_cost' => $kind == SALE_LEADER ? 5000 : 3000,
            'weight' => $weight,
            'leader_cost' => 5000,
            'member_cost' => 3000,
            'leader_member_cost' => 2000,
            'amount_weight' => $kind == SALE_LEADER ? $weight * 5000 : $weight * 3000,
            'amount_weight_member' => $weight * 2000,
        ];
    }
}

function getShipmentWeight($fwd_ids, $filters = null)
{
    $shipments = Shipment::isShow()->whereIn('user_id', $fwd_ids)->isFilters($filters)
        ->select(
            'shipments.user_id',
            'shipments.shipment_service_id',
            DB::raw('SUM(packages.package_charged_weight) as total_charged_weight')
        )
        ->leftJoin('packages', 'packages.shipment_id', '=', 'shipments.shipment_id')
        ->where('packages.delete_flg', NO_DELETE)
        ->where('packages.package_status', '!=', SHIPMENT_STATUS_RETURNED)
        ->groupBy('packages.shipment_id')->get()->toArray();

    return $shipments;
}


function getTotalWeightCompaniesForSale2($sale_id, $filters = null)
{

    try {
        $date_range = $filters['date_range'] ?? null;

        $companies = SaleLinkFwd::isShow();
        if (!empty($sale_id)) {
            $companies = $companies->where('sale_link_fwds.sale_id', $sale_id);
        }

        $companies = $companies->leftJoin('shipments', function ($join) use ($date_range) {
            $join->on('sale_link_fwds.fwd_id', '=', 'shipments.user_id')
                ->where('shipments.delete_flg', NO_DELETE)
                ->whereNotIn('shipments.shipment_status', [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_RETURNED])
                ->where('shipments.active_flg', ACTIVE);

            if (!empty($date_range['start_date']) && !empty($date_range['end_date'])) {
                $start_date = Carbon::parse($date_range['start_date'])->startOfDay()->toDateTimeLocalString();
                $end_date = Carbon::parse($date_range['end_date'])->endOfDay()->toDateTimeLocalString();
                $join->whereBetween('shipments.created_at', [$start_date, $end_date]);
            }
        })
            ->leftJoin('fwd_costs', function ($join) {
                $join->on('shipments.user_id', '=', 'fwd_costs.fwd_id')
                    ->on('shipments.shipment_service_id', '=', 'fwd_costs.service_id')
                    ->where('fwd_costs.active_flg', ACTIVE)
                    ->where('fwd_costs.delete_flg', NO_DELETE);
            })
            ->leftJoin('packages', function ($join) {
                $join->on('shipments.shipment_id', '=', 'packages.shipment_id')
                    ->where('packages.active_flg', ACTIVE)
                    ->where('packages.delete_flg', NO_DELETE);
            });

        $companies = $companies
            ->leftJoin('sales_members', function ($join) {
                $join->on('sale_link_fwds.sale_id', '=', 'sales_members.user_id')
                    ->where('sales_members.delete_flg', NO_DELETE);
            })
            ->leftJoin('sales_teams', function ($join) {
                $join->on('sales_members.sale_team_id', '=', 'sales_teams.sale_team_id')
                    ->where('sales_teams.delete_flg', NO_DELETE);
            })
            ->leftJoin('users as leader', function ($join) {
                $join->on('sales_teams.leader_id', '=', 'leader.user_id')
                    ->where('leader.delete_flg', NO_DELETE);
            });

        $companies = $companies->select([
            DB::raw('SUM(packages.package_charged_weight * IFNULL(fwd_costs.leader_cost, 5000)) as total_leader_cost'),
            DB::raw('SUM(packages.package_charged_weight * IFNULL(fwd_costs.member_cost, 3000)) as total_member_cost'),
            DB::raw('SUM(packages.package_charged_weight * IFNULL(fwd_costs.leader_member_cost, 2000)) as total_leader_member_cost'),
            DB::raw('SUM(packages.package_charged_weight) as fw_weight')
        ])
            ->groupBy('sale_link_fwds.sale_link_fwd_id')
            ->get()
            ->toArray();


        return [
            "total_weight" => collect($companies)->sum('fw_weight') ?? 0,
            "total_price" => collect($companies)->sum('total_member_cost') ?? 0,
            "total_leader_cost" => collect($companies)->sum('total_leader_cost') ?? 0,
            "total_leader_member_cost" => collect($companies)->sum('total_leader_member_cost') ?? 0,
        ];
    } catch (\Throwable $th) {
        return [
            "total_weight" => 0,
            "total_price" => 0,
            "total_leader_cost" => 0,
            "total_leader_member_cost" => 0,
        ];
    }
}

function getCommistionSale($profit, $range_kpi)
{
    try {
        $sale_profit = $profit - $range_kpi["salary"];
        if ($sale_profit > 0) {
            return ($sale_profit * $range_kpi["ratio_commission"]) / 100;
        } else {
            return 0;
        }
    } catch (\Throwable $th) {
        return 0;
    }
}

function onGetKpiSale($kpi_id, $monthDate)
{
    try {
        $kpi = SaleKpi::isShow()
            ->where('sale_kpi_id', $kpi_id)
            ->where(function ($q) use ($monthDate) {
                $q->where('time_kind', '!=', 1) // Không cần so sánh ngày nếu time_kind khác 1
                    ->orWhere(function ($query) use ($monthDate) {
                        // Chỉ kiểm tra điều kiện ngày nếu time_kind = 1
                        $query->where('time_kind', 1)
                            ->where('time_start', '<=', $monthDate)
                            ->where('time_end', '>=', $monthDate);
                    });
            })
            ->with(['range_kpis'])
            ->first();
        return $kpi;
    } catch (\Throwable $th) {
        return null;
    }
}

function getCommistionSaleTeam($team, $leader)
{
    try {
        $commission = 0;
        foreach ($team as $key => $member) {
            if ($leader["leader_id"] == $member["user_id"]) {
                continue;
            }
            if ($member['leader_have_commission'] == 1) {
                $commission += (getCommistionSale(1, 1) * 0.07);
            }
        }
        return $commission;
    } catch (\Throwable $th) {
        return 0;
    }
}


function getDataSmPackageByDate($user_id, $filters = null)
{
    try {
        $data = [
            'total_shipment' => 0,
            'total_profit' => 0,
            'total_price' => 0,
            'total_amount_customer' => 0,
            'total_package' => 0,
            'total_charged_weight' => 0,
        ];
        $sm = Shipment::where('user_id', $user_id)
            ->isFilters($filters)
            ->isShow()
            ->select(
                'shipments.shipment_id',
                'shipments.shipment_payment_status',
                DB::raw("CASE 
                    WHEN shipments.shipment_payment_status = 0 THEN 0 
                    WHEN shipments.shipment_status IN (0, 3, 4) THEN 0 
                    ELSE shipments.shipment_amount_profit 
                 END as shipment_amount_profit"),
                'shipments.shipment_final_amount',
                'shipments.shipment_amount_total_customer'
            )
            ->get()
            ->toArray();

        $sm = $sm ?? [];

        $data['total_shipment'] = count($sm);
        $data['total_profit'] = collect($sm)->sum('shipment_amount_profit');
        $data['total_price'] = collect($sm)->sum('shipment_final_amount');
        $data['total_amount_customer'] = collect($sm)->sum('shipment_amount_total_customer');

        $shipment_ids = collect($sm)->pluck('shipment_id');
        $pks = Package::isShow()->whereIn('shipment_id', $shipment_ids)
            ->select(
                DB::raw('COUNT(packages.package_id) AS total_package'),
                DB::raw('SUM(packages.package_charged_weight) AS total_charged_weight'),
            )->first();
        $data['total_package'] = $pks['total_package'];
        $data['total_charged_weight'] = $pks['total_charged_weight'];

        // $data = Shipment::selectRaw(
        //     'COUNT(shipments.shipment_id) AS total_shipment,
        //      SUM(shipments.shipment_amount_profit) AS total_profit,
        //      SUM(shipments.shipment_final_amount) AS total_price,
        //      SUM(shipments.shipment_amount_total_customer) AS total_amount_customer,
        //      COUNT(packages.package_id) AS total_package,
        //      SUM(packages.package_charged_weight) AS total_charged_weight'
        // )
        // ->leftJoin('packages', 'shipments.shipment_id', '=', 'packages.shipment_id')
        // ->where('shipments.user_id', $user_id)
        // ->isShow()
        // ->where('packages.delete_flg', NO_DELETE)
        // ->where('packages.active_flg', ACTIVE)
        // ->first();

        if ($data) {
            return $data;
        }
        return [
            'total_shipment' => 0,
            'total_package' => 0,
            'total_profit' => 0,
            'total_price' => 0,
            'total_amount_customer' => 0,
            'total_charged_weight' => 0,
        ];
    } catch (\Throwable $th) {
        return [
            'total_shipment' => 0,
            'total_package' => 0,
            'total_profit' => 0,
            'total_price' => 0,
            'total_amount_customer' => 0,
            'total_charged_weight' => 0,
        ];
    }
}

function onCreateLabel($shipment, $packages, $invoices, $service)
{
    try {
        $area_code = City::where('city_id', $shipment['receiver_city_id'])
            ->leftJoin('states', 'cities.state_id', 'states.state_id')
            ->leftJoin('countries', 'states.country_id', 'countries.country_id')
            ->select('state_code', 'state_name', 'country_code', 'country_name', 'city_name')
            ->first()->toArray();

        $data_label = $shipment;
        $data_label['packages'] = collect($packages)->toArray();
        $data_label['invoices'] = collect($invoices)->toArray();
        $data_label['country_code'] = $area_code['country_code'];
        $data_label['country_name'] = $area_code['country_name'];
        $data_label['state_code'] = $area_code['state_code'];
        $data_label['state_name'] = $area_code['state_name'];
        $data_label['city_code'] = $area_code['city_name'];
        $data_label['service_volumetric_mass'] = $service['service_volumetric_mass'];

        $associate_with_service = AssociateWithService::where('service_id', $service['service_id'])->first();
        if (!isset($associate_with_service['associate_types']) || empty(json_decode($associate_with_service['associate_types'], true))) {
            return [
                'status' => HTTP_OK,
                'OrderId' => null,
                'res_label' => [
                    'status' => HTTP_INTERNAL_SERVER_ERROR
                ]
            ];
        }
        $associate_with_service = $associate_with_service->toArray();
        $associate_with_service['associate_types'] = json_decode($associate_with_service['associate_types'], true);
        $random_api = $associate_with_service['associate_types'][array_rand($associate_with_service['associate_types'])];

        $res_label = [];
        switch ($random_api) {
            case API_LABEL_SHIPSAO:
                $initialisation_label = new ShipsaoApi();
                $res_label = $initialisation_label->createOrder($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $label['labelLink'] = convertBase64ToPdf($res_label['data']['_id'], $res_label['data']['label_base64'], PUBLIC_PATH_SHIPMENT_LABEL);
                    $data_label_creating = [
                        'shipment_id' => $shipment['shipment_id'],
                        'create_by' => $shipment['user_id'],
                        'label_code' => $res_label['data']['_id'],
                        'link' => $label['labelLink'],
                        'reference_no' => $res_label['data']['_id'],
                        'state' => $data_label['state_code'],
                        'service_code' => 'SHIPSAO',
                        'in_directory' => IN_DIRECTORY,
                    ];
                    createLabelEpackage($data_label_creating);
                    $OrderId = $res_label['data']['_id'];
                }
                break;
            case API_LABEL_DMXSMART_USPSGA:
                $initialisation_label = new DmxSmartApi();
                $res_label = $initialisation_label->createShipmentWithServiceCodeOfUPS($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    foreach ($res_label['data']['labelBase64s'] as $label) {
                        $label['labelLink'] = convertBase64ToPdf($label['trackingNo'] ?? Str::random(50), $label['label'], PUBLIC_PATH_SHIPMENT_LABEL);
                        $data_label_creating = [
                            'shipment_id' => $shipment['shipment_id'],
                            'create_by' => $shipment['user_id'],
                            'label_code' => $res_label['data']['shipmentId'],
                            'link' => $label['labelLink'],
                            'reference_no' => $res_label['businessParams']['referenceId'],
                            'state' => $data_label['state_code'],
                            'service_code' => $res_label['businessParams']['serviceCode'],
                            'in_directory' => IN_DIRECTORY,
                        ];
                        createLabelEpackage($data_label_creating);
                        $OrderId = $res_label['businessParams']['referenceId'];
                    }
                }
                break;
            case API_LABEL_DMXSMART_UPSGND:
                $initialisation_label = new DmxSmartApi();
                $res_label = $initialisation_label->createShipmentWithServiceCodeOfUPSGROUND($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    foreach ($res_label['data']['labelBase64s'] as $label) {
                        $label['labelLink'] = convertBase64ToPdf($label['trackingNo'] ?? Str::random(50), $label['label'], PUBLIC_PATH_SHIPMENT_LABEL);
                        $data_label_creating = [
                            'shipment_id' => $shipment['shipment_id'],
                            'create_by' => $shipment['user_id'],
                            'label_code' => $res_label['data']['shipmentId'],
                            'link' => $label['labelLink'],
                            'reference_no' => $res_label['businessParams']['referenceId'],
                            'state' => $data_label['state_code'],
                            'service_code' => $res_label['businessParams']['serviceCode'],
                            'in_directory' => IN_DIRECTORY,
                        ];
                        createLabelEpackage($data_label_creating);
                        $OrderId = $res_label['businessParams']['referenceId'];
                    }
                }
                break;
            case API_LABEL_AUSPOST:
                $initialisation_label = new AusPostApi();
                $res_label = $initialisation_label->createShipments($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    foreach ($res_label['data']['labels'] as $label) {
                        $data_label_creating = [
                            'shipment_id' => $shipment['shipment_id'],
                            'create_by' => $shipment['user_id'],
                            'label_code' => $label['request_id'],
                            'link' => $label['url'],
                            'reference_no' => $label['shipments'][0]['shipment_id'],
                            'state' => $data_label['state_code'],
                            'post_code' => $shipment['receiver_postal_code'],
                            'service_code' => "AUSPOST",
                            'in_directory' => NOT_DIRECTORY,
                        ];
                        createLabelEpackage($data_label_creating);
                        $OrderId = $label['shipments'][0]['shipment_id'];
                    }
                }
                break;
            // case API_LABEL_ETOWER:
            //     $etower = new EtowerApi();
            //     $res_label = $etower->createShippingOrder($data_label);

            //     if ($res_label['status'] == HTTP_OK) {
            //         $OrderId = $res_label['data'][0]['orderId'];
            //         $data = [
            //             "orderIds" => [$res_label['data'][0]['orderId']]
            //         ];

            //         $labels = $etower->printLabels($data);
            //         $labels = $labels['data'];
            //         foreach ($labels as $label) {
            //             $label['labelLink'] = convertBase64ToPdf($label['trackingNo'], $label['labelContent'], PUBLIC_PATH_SHIPMENT_LABEL);
            //             $data_label_creating = [
            //                 'shipment_id' => $shipment['shipment_id'],
            //                 'create_by' => $shipment['user_id'],
            //                 'label_code' => $OrderId,
            //                 'link' => $label['labelLink'],
            //                 'reference_no' => $res_label['data'][0]['referenceNo'],
            //                 'state' => $data_label['state_code'],
            //                 'post_code' => $shipment['receiver_postal_code'],
            //                 'service_code' => 'UBI.CN2CA.CPC',
            //                 'in_directory' => IN_DIRECTORY,
            //             ];
            //             createLabelEpackage($data_label_creating);
            //         }
            //     }
            //     break;
            case API_LABEL_WORLDTECH_HERMES:
                $shipmentController = new ShipmentController();
                $res_label = $shipmentController->convertDataShipmentToLabel('UKHERMES48H', $data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data'][0]['OrderId'];
                }
                break;
            case API_LABEL_WORLDTECH_NZ_POST:
                $shipmentController = new ShipmentController();
                $res_label = $shipmentController->convertDataShipmentToLabel('AP-NZPOST', $data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data'][0]['OrderId'];
                }
                break;
            case API_LABEL_WORLDTECH_DHL:
                $shipmentController = new ShipmentController();
                $res_label = $shipmentController->convertDataShipmentToLabel('AP-DE-DHL-INTER', $data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data'][0]['OrderId'];
                }
                break;
            case API_LABEL_DPD_UK_HERMES:
                $dpd = new DPDApi();
                $res_label = $dpd->createShipmentWithDpdUk($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $res_label['data']['pdf'] = convertStringToFileTxt($shipment['shipment_code'], $res_label['data']['html'], PUBLIC_PATH_SHIPMENT_LABEL);
                    $data_label_creating = [
                        'shipment_id' => $shipment['shipment_id'],
                        'create_by' => $shipment['user_id'],
                        'label_code' => $res_label['data']['shipmentid'],
                        'link' => $res_label['data']['pdf'],
                        'reference_no' => $res_label['data']['reference_no'],
                        'state' => $data_label['state_code'],
                        'service_code' => 'DPD UK',
                        'in_directory' => IN_DIRECTORY,
                    ];

                    createLabelEpackage($data_label_creating);
                    $OrderId = $res_label['data']['shipmentid'];
                }
                break;
            case API_LABEL_DPD_NL_HERMES:
                $dpd = new DPDApi();
                $res_label = $dpd->createShipmentWithDpdNl($data_label);
                if ($res_label['status'] == HTTP_OK) {
                    $data_label_creating = [
                        'shipment_id' => $shipment['shipment_id'],
                        'create_by' => $shipment['user_id'],
                        'label_code' =>  $res_label['data']['data']['sk_no'],
                        'link' => $res_label['data']['data']['label_link'],
                        'reference_no' => $res_label['data']['reference_no'],
                        'state' => $data_label['state_code'],
                        'service_code' => 'DPD NL',
                        'in_directory' => NOT_DIRECTORY,
                    ];

                    createLabelEpackage($data_label_creating);
                    $OrderId = $res_label['data']['reference_no'];
                }
                break;
            case API_LABEL_UBI_AU2AU_PRIORITY:
                $etower = new EtowerApi();
                $res_label = $etower->createShippingOrder($data_label, 'UBI.AU2AU.PRIORITY');

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data'][0]['orderId'];
                    $data = [
                        "orderIds" => [$res_label['data'][0]['orderId']]
                    ];

                    $labels = $etower->printLabels($data);
                    $labels = $labels['data'];
                    foreach ($labels as $label) {
                        $label['labelLink'] = convertBase64ToPdf($label['trackingNo'], $label['labelContent'], PUBLIC_PATH_SHIPMENT_LABEL);
                        $data_label_creating = [
                            'shipment_id' => $shipment['shipment_id'],
                            'create_by' => $shipment['user_id'],
                            'label_code' => $OrderId,
                            'link' => $label['labelLink'],
                            'reference_no' => $res_label['data'][0]['referenceNo'],
                            'state' => $data_label['state_code'],
                            'post_code' => $shipment['receiver_postal_code'],
                            'service_code' => 'UBI.AU2AU.PRIORITY',
                            'in_directory' => IN_DIRECTORY,
                        ];
                        createLabelEpackage($data_label_creating);
                    }
                }
                break;
            case API_LABEL_UBI_AU2AU_IPEC:
                $etower = new EtowerApi();
                $res_label = $etower->createShippingOrder($data_label, 'UBI.AU2AU.IPEC');

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data'][0]['orderId'];
                    $data = [
                        "orderIds" => [$res_label['data'][0]['orderId']]
                    ];

                    $labels = $etower->printLabels($data);
                    $labels = $labels['data'];
                    foreach ($labels as $label) {
                        $label['labelLink'] = convertBase64ToPdf($label['trackingNo'], $label['labelContent'], PUBLIC_PATH_SHIPMENT_LABEL);
                        $data_label_creating = [
                            'shipment_id' => $shipment['shipment_id'],
                            'create_by' => $shipment['user_id'],
                            'label_code' => $OrderId,
                            'link' => $label['labelLink'],
                            'reference_no' => $res_label['data'][0]['referenceNo'],
                            'state' => $data_label['state_code'],
                            'post_code' => $shipment['receiver_postal_code'],
                            'service_code' => 'UBI.AU2AU.IPEC',
                            'in_directory' => IN_DIRECTORY,
                        ];
                        createLabelEpackage($data_label_creating);
                    }
                }

                if ($res_label['status'] != HTTP_OK) {
                    $initiallize_label_history_data = [
                        'shipment_id' => $shipment['shipment_id'],
                        'user_id' => $shipment['user_id'],
                        'result_call_api' => json_encode($res_label['data']),
                        'initiallize_label_history_kind' => $random_api
                    ];
                    initiallizeLabelHistory($initiallize_label_history_data);

                    return [
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'OrderId' => null,
                        'res_label' => null
                    ];
                }
                break;
            case API_TOP_LOGISTICS_API:
                $top_logistics = new TopLogisticsApi();
                $res_label = $top_logistics->createShipment($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data']['custref'];
                    $data = [
                        "cust_ref" => $res_label['data']['custref'],
                        'base64' => true
                    ];

                    $label = $top_logistics->getCourierLabel($data);
                    $label['labelLink'] = convertBase64ToPdf($OrderId, $label['data'], PUBLIC_PATH_SHIPMENT_LABEL);

                    $data_label_creating = [
                        'shipment_id' => $shipment['shipment_id'],
                        'create_by' => $shipment['user_id'],
                        'label_code' => $OrderId,
                        'link' => $label['labelLink'],
                        'reference_no' => $res_label['data']['lref'],
                        'state' => $data_label['state_code'],
                        'post_code' => $shipment['receiver_postal_code'],
                        'service_code' => 'TOP-LOG-AU',
                        'in_directory' => IN_DIRECTORY,
                    ];
                    createLabelEpackage($data_label_creating);
                }
                break;
            case API_TOP_LOGISTICS_API:
                $top_logistics = new TopLogisticsApi();
                $res_label = $top_logistics->createShipment($data_label);

                if ($res_label['status'] == HTTP_OK) {
                    $OrderId = $res_label['data']['custref'];
                    $data = [
                        "cust_ref" => $res_label['data']['custref'],
                        'base64' => true
                    ];

                    $label = $top_logistics->getCourierLabel($data);
                    $label['labelLink'] = convertBase64ToPdf($OrderId, $label['data'], PUBLIC_PATH_SHIPMENT_LABEL);

                    $data_label_creating = [
                        'shipment_id' => $shipment['shipment_id'],
                        'create_by' => $shipment['user_id'],
                        'label_code' => $OrderId,
                        'link' => $label['labelLink'],
                        'reference_no' => $res_label['data']['lref'],
                        'state' => $data_label['state_code'],
                        'post_code' => $shipment['receiver_postal_code'],
                        'service_code' => 'TOP-LOG-AU',
                        'in_directory' => IN_DIRECTORY,
                    ];
                    createLabelEpackage($data_label_creating);
                }
                break;
        }

        if ($res_label['status'] != HTTP_OK) {
            $initiallize_label_history_data = [
                'shipment_id' => $shipment['shipment_id'],
                'user_id' => $shipment['user_id'],
                'result_call_api' => json_encode($res_label['data']),
                'initiallize_label_history_kind' => $random_api
            ];
            initiallizeLabelHistory($initiallize_label_history_data);

            return [
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'OrderId' => null,
                'res_label' => null
            ];
        }
        return [
            'status' => HTTP_OK,
            'OrderId' => isset($OrderId) ? $OrderId : null,
            'res_label' => $res_label
        ];
    } catch (\Throwable $th) {
        return [
            'status' => HTTP_INTERNAL_SERVER_ERROR,
            'OrderId' => null,
            'res_label' => null
        ];
    }
}

function initiallizeLabelHistory($data)
{
    InitiallizeLabelHistory::create($data);
}
function getShipmentHawbCode()
{
    try {
        $shipment = Shipment::isShow()->select('shipment_hawb_code')->orderByDesc('shipment_hawb_code')->first();
        if ($shipment && isset($shipment->shipment_hawb_code)) {
            return $shipment->shipment_hawb_code + 2;
        } else {
            return '2024010900';
        }
    } catch (\Throwable $th) {
        return null;
    }
}

function getShipmentCodes()
{
    try {
        $shipment = Shipment::isShow()->select('shipment_hawb_code')->orderByDesc('shipment_hawb_code')->first();
        if ($shipment && isset($shipment->shipment_hawb_code)) {
            return $shipment->shipment_hawb_code + 2;
        } else {
            return '2024010900';
        }
    } catch (\Throwable $th) {
        return '2024010900';
    }
}


function getCodePackage()
{
    try {
        $package = Package::isShow()
            ->select('package_hawb_code', 'package_code', 'package_tracking_code')
            ->orderByDesc('package_code')->first();
        if ($package) {
            // preg_match('/\d+$/', $package->package_code, $matches);
            // $last_numeric_part = $matches[0] ?? '0';
            // $new_numeric_part = (int) $last_numeric_part + 2;
            return [
                "package_code" =>  $package->package_code + 2,
                "package_hawb_code" => $package->package_hawb_code + 2,
                // "package_tracking_code" =>  $package->package_code + 1,
            ];
        } else {
            return [
                "package_code" =>  "320240109001",
                "package_hawb_code" =>  "320240109001",
            ];
        }
    } catch (\Throwable $th) {
        return [
            "package_code" =>  "320240109001",
            "package_hawb_code" =>  "320240109001",
        ];
    }
}


function updateStatusShipment($shipment_id)
{
    try {
        $shipment = Shipment::find($shipment_id);
        if ($shipment['shipment_status'] == SHIPMENT_STATUS_HOLD) {
            return [
                'status' => HTTP_OK,
                'message' => "Đơn hàng đã được check hold",
                'shipment' => $shipment
            ];
        }
        $packages = Package::isShow()->where('shipment_id', $shipment_id)
            ->select('package_id', 'package_status')->get()->toArray();

        $count_package = count($packages);

        $count_created = collect($packages)->where('package_status', SHIPMENT_STATUS_CREATE)->count();
        $count_imported = collect($packages)->where('package_status', SHIPMENT_STATUS_IMPORTED)->count();
        $count_exported = collect($packages)->where('package_status', SHIPMENT_STATUS_EXPORED)->count();
        $count_returned = collect($packages)->where('package_status', SHIPMENT_STATUS_RETURNED)->count();
        $count_hold = collect($packages)->where('package_status', SHIPMENT_STATUS_HOLD)->count();

        if ($count_created == $count_package) {
            $shipment_status = SHIPMENT_STATUS_CREATE;
        } else if ($count_imported == $count_package) {
            $shipment_status = SHIPMENT_STATUS_IMPORTED;
        } else if ($count_exported == $count_package) {
            $shipment_status = SHIPMENT_STATUS_EXPORED;
        } else if ($count_returned == $count_package) {
            $shipment_status = SHIPMENT_STATUS_RETURNED;
        } else if ($count_hold == $count_package) {
            $shipment_status = SHIPMENT_STATUS_HOLD;
        } else if ($count_created > 0) {
            $shipment_status = SHIPMENT_STATUS_CREATE;
        } else if ($count_imported > 0 && $count_created == 0) {
            $shipment_status = SHIPMENT_STATUS_IMPORTED;
            if ($shipment['shipment_status'] == SHIPMENT_STATUS_CREATE && onCheckPositionAccount(SETTING_FWD_ACCOUNT, $shipment['user']['position_id'])) {
                $last_import = HistoryScan::isShow()->where('history_scan.status', SHIPMENT_STATUS_IMPORTED)
                    ->leftJoin('packages', 'history_scan.package_id', '=', 'packages.package_id')
                    ->where('packages.shipment_id', $shipment_id)
                    ->orderByDesc('history_scan.created_at')
                    ->groupBy('history_scan.history_scan_id')
                    ->select('history_scan.created_at')->first();
                $completed_date = isset($last_import['created_at']) ? Carbon::parse($last_import['created_at'])->setTimezone('Asia/Ho_Chi_Minh') : Carbon::now()->setTimezone('Asia/Ho_Chi_Minh');
            }
        } else if ($count_exported > 0 && $count_created == 0 && $count_imported == 0) {
            $shipment_status = SHIPMENT_STATUS_EXPORED;
        }
        if (isset($completed_date)) {
            Shipment::find($shipment_id)->update([
                'shipment_status' => $shipment_status,
                'completed_date' => $completed_date
            ]);
            $result_save_amount = onSaveAmountShipment($shipment['shipment_id']);
            if ($result_save_amount['status'] === HTTP_INTERNAL_SERVER_ERROR) {
                DB::rollback();
                return response()->json($result_save_amount);
            }
            // tính giá trị cuối cùng của đơn và lợi nhuận 
            onUpdateFinalAmountShipment($shipment['shipment_id']);
        } else {
            Shipment::find($shipment_id)->update(['shipment_status' => $shipment_status]);
        }
        return [
            'status' => HTTP_OK,
            'message' => "Cập nhật trạng thái shipment thành công",
            'shipment' => $shipment
        ];
    } catch (\Throwable $th) {
        return [
            'status' => HTTP_UNKNOWN_ERROR,
            'message' => "Cập nhật trạng thái shipment thất bại"
        ];
    }
}

function onUpdateCompletedDate($shipment_id)
{
    try {
        $shipment = Shipment::find($shipment_id);
        if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, $shipment['user']['position_id'])) {
            $last_import = HistoryScan::isShow()->where('history_scan.status', SHIPMENT_STATUS_IMPORTED)
                ->leftJoin('packages', 'history_scan.package_id', '=', 'packages.package_id')
                ->where('packages.shipment_id', $shipment_id)
                ->orderByDesc('history_scan.created_at')
                ->groupBy('history_scan.history_scan_id')
                ->select('history_scan.created_at')->first();
            $completed_date = isset($last_import['created_at']) ? Carbon::parse($last_import['created_at'])->setTimezone('Asia/Ho_Chi_Minh') : Carbon::now()->setTimezone('Asia/Ho_Chi_Minh');
            Shipment::find($shipment['shipment_id'])->update([
                'completed_date' => $completed_date,
            ]);
        }
    } catch (\Throwable $th) {
        //throw $th;
    }
}

function onCheckAccountantStatus($accountant_status)
{
    try {
        if (in_array($accountant_status, [1, 2]) && !isGetAllData()) {
            return [
                "status" => false,
                "message" => "Đơn hàng đã được kế toán xác nhận, bạn không đủ quyền để sửa"
            ];
        }
        return [
            "status" => true
        ];
    } catch (\Throwable $th) {
        return [
            "status" => false,
            "message" => 'Xảy ra lỗi không xác định!'
        ];
    }
}

function queryExportExcel()
{
    $query = Shipment::isShow()
        ->leftJoin('packages', function ($join) {
            $join->on('shipments.shipment_id', '=', 'packages.shipment_id')
                ->where('packages.delete_flg', NO_DELETE)
                ->where('packages.active_flg', ACTIVE);
        })
        ->leftJoin('users', 'shipments.user_id', 'users.user_id')
        ->leftJoin('debits', 'shipments.shipment_debit_id', 'debits.debit_id')
        ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
        ->leftJoin('countries', 'shipments.receiver_country_id', 'countries.country_id')
        ->select(
            'shipments.shipment_id',
            'users.position_id',
            'users.user_company_name',
            'users.user_contact_name',
            'users.user_debit_type',
            'shipments.shipment_amount_total_customer',
            'shipments.shipment_amount_surcharge',
            'shipments.shipment_service_id',
            'shipments.shipment_code',
            'shipments.shipment_debit_id',
            'shipments.shipment_file_proof_of_payment',

            // Data export
            'users.user_code', //
            'shipments.created_at',
            'shipments.sender_contact_name',
            'services.service_name',
            // AWB
            // HAWB
            // TRACKING_NUMBER
            'shipments.receiver_contact_name',
            'countries.country_name', // DEST ?? country
            // TYPE
            DB::raw('COUNT(packages.package_id) as package_count'), // count package
            DB::raw('SUM(packages.package_charged_weight) as charged_weight'), // Charge_Weight
            DB::raw('
                    (
                        SUM(shipment_domestic_charges) +
                        SUM(shipment_amount_surcharge) +
                        SUM(shipment_collection_fee) +
                        SUM(shipment_amount_insurance) +
                        SUM(shipment_amount_vat) +
                        SUM(shipment_amount_original) -
                        SUM(shipment_amount_discount) +
                        SUM(shipment_amount_operating_costs)
                    ) as total_value
                '), // giá trị shipment - vat
            // Tiền phụ thu hàng hóa
            'shipment_amount_vat',
            'shipment_final_amount',
            'shipment_amount_fsc',
            // Danh sách phụ thu
            DB::raw('SUM(packages.package_weight) as gross_weight'),
            'debit_no',
            'debit_advance_date',
            'shipments.shipment_status',
            DB::raw("
                CASE 
                    WHEN debits.debit_type = 1 THEN 'DAY'
                    WHEN debits.debit_type = 2 THEN 'WEEK'
                    WHEN debits.debit_type = 3 THEN '2 WEEK'
                    WHEN debits.debit_type = 4 THEN 'MONTH'
                    ELSE 'Không xác định'
                END as debit_type_label
            "),
            DB::raw("
                CASE 
                    WHEN shipments.shipment_payment_status = 0 THEN 'Chưa thanh toán'
                    WHEN shipments.shipment_payment_status = 1 THEN 'Đã thanh toán'
                    WHEN shipments.shipment_payment_status = 2 THEN 'Tạm ứng'
                    ELSE ''
                END as shipment_payment_status_label
            "),
            // MAWB
        );

    return $query;
}


function getMawbByShipment($shipment_id)
{
    try {
        $mawb = Package::isShow()
            ->leftJoin('shipments_tracktry', function ($join) {
                $join->on('packages.sm_tracktry_id', 'shipments_tracktry.sm_tracktry_id')
                    ->where('shipments_tracktry.delete_flg', NO_DELETE)
                    ->where('shipments_tracktry.active_flg', ACTIVE);
            })->where('shipment_id', $shipment_id)
            ->select(
                'shipments_tracktry.sm_tracktry_id',
                'shipments_tracktry.awb_code',
                'shipments_tracktry.dest',
            )
            ->groupBy('sm_tracktry_id')->get()->toArray();
        return $mawb;
    } catch (\Throwable $th) {
        return [];
    }
};

function getDescriptionPackageByShipment($shipment_id, $type)
{
    try {
        $packages = Package::isShow()->where('shipment_id', $shipment_id);
        switch ($type) {
            case 'hawb':
                $packages = $packages->pluck('package_hawb_code')->toArray();
                return $packages;

            case 'tracking':
                $packages = $packages->pluck('package_tracking_code')->toArray();
                if (empty($packages)) {
                    return [];
                } else {
                    // return array_unique($packages);
                    return $packages;
                }
            default:
                # code...
                return [];
        }
    } catch (\Throwable $th) {
        return [];
    }
}

function getDescriptionGoodsByShipment($shipment_id)
{
    try {
        $package_ids = Package::isShow()->where('shipment_id', $shipment_id)->get()->pluck('package_id');
        $package_sgs = PackageSg::isShow()->whereIn('package_surchage_goods.package_id', $package_ids)
            ->leftJoin('surcharge_goods', 'package_surchage_goods.surcharge_goods_id', 'surcharge_goods.surcharge_goods_id')
            ->select('package_surchage_goods.*', 'surcharge_goods.surcharge_goods_name')
            ->groupBy('package_surchage_good_id')->get()->toArray();


        return $package_sgs;
    } catch (\Throwable $th) {
        return [];
    }
}

function onGetAmountDiscount($service_id, $user_id, $weight)
{
    try {
        $date = Carbon::now()->format('Y-m-d H:i:s');
        $discount_amount = 0;
        $discount = DiscountUser::isShow()
            ->where('user_id', $user_id)
            ->where('service_id', $service_id)
            ->where(function ($query) use ($date) {
                $query->where('discount_start_date', '<=', $date)
                    ->where('discount_end_date', '>=', $date);
            })
            ->first();
        if ($discount) {
            $discount_weight = DiscountUserWeightLevel::where('discount_user_id', $discount->discount_user_id)
                ->where('discount_user_weight', '>', $weight)
                ->orderBy('discount_user_weight', 'desc')->first();

            $discount_amount = $discount_weight->discount_user_amount ?? 0;
        }
        return $discount_amount;
    } catch (\Throwable $th) {
        return 0;
    }
}

function onCreatePackageTrackingStatus($package_code, $address, $note, $date = null, $status = null, $active = ACTIVE)
{

    try {
        PackageTracking::create([
            'package_code'     => $package_code,
            'package_tracking_date'     => $date ?? Carbon::now()->format('Y-m-d H:i:s'),
            'package_tracking_address' => $address,
            'package_tracking_note'    => $note,
            'package_tracking_status'    => $status ?? 0,
            'active_flg'    => $active,
        ]);
    } catch (\Throwable $th) {
    }
}
function onActivePackageTrackingPending()
{
    try {
        PackageTracking::where('active_flg', INACTIVE)
            ->where('package_tracking_note', 'Origin customs processing')
            ->whereColumn('package_tracking_date', '>', Carbon::raw('DATE_ADD(created_at, INTERVAL 30 MINUTE)'))
            ->update(['active_flg', ACTIVE]);
    } catch (\Throwable $th) {
        //throw $th;
    }
}

function getTotalPackage($packages, $service_volumetric_mass)
{
    $total_weight = 0;
    try {
        foreach ($packages as $key => $package) {
            $converted_weight = $package['package_length'] * $package['package_width'] * $package['package_height'] / $service_volumetric_mass;
            $charged_weight = $package['package_weight'] < $converted_weight ? $converted_weight : $package['package_weight'];
            $total_weight += $package['package_quantity'] * $charged_weight;
        }
        return $total_weight;
    } catch (\Throwable $th) {
        return 0;
    }
}

function getCompanyLinkSaleWithWeight($sale_id, $filters)
{
    try {
        $date_range = isset($filters['date_range']) ? $filters['date_range'] : null;
        $data = SaleLinkFwd::isShow()
            ->where('sale_link_fwds.sale_id', $sale_id)
            ->leftJoin('users', 'sale_link_fwds.fwd_id', '=', 'users.user_id')
            ->leftJoin('shipments', function ($join) use ($date_range) {
                $join->on('sale_link_fwds.fwd_id', '=', 'shipments.user_id')
                    ->where('shipments.delete_flg', NO_DELETE)
                    ->where('shipments.active_flg', ACTIVE);
                if (isset($date_range['start_date']) && isset($date_range['end_date'])) {
                    $start_date = Carbon::parse($date_range['start_date'])->startOfDay()->toDateTimeLocalString();
                    $end_date = Carbon::parse($date_range['end_date'])->endOfDay()->toDateTimeLocalString();
                    $join->whereBetween('shipments.created_at', [$start_date, $end_date]);
                }
            })
            ->leftJoin('packages', function ($join) {
                $join->on('shipments.shipment_id', '=', 'packages.shipment_id')
                    ->whereNotIn('shipment_status', [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_RETURNED])
                    ->where('packages.delete_flg', NO_DELETE)
                    ->where('packages.active_flg', ACTIVE);
            })
            ->select(
                'sale_link_fwds.sale_link_fwd_id',
                'sale_link_fwds.fwd_id',
                'users.user_company_name',
                'users.user_contact_name',
                'users.user_code',
                DB::raw('SUM(packages.package_charged_weight) as total_weight')
            )->groupBy('sale_link_fwds.fwd_id')->get();
        return $data;
    } catch (\Throwable $th) {
        return [];
    }
}

function createBarCode($code, $width = 2, $height = 70)
{
    try {
        $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
        $convert = $generator->getBarcode($code, $generator::TYPE_CODE_128, $width, $height);
        return $convert['barcode'] ?? $convert;
    } catch (\Throwable $th) {
        return null;
    }
}

function onUpdatePackage($shipment, $package)
{
    try {
        $service = Service::isActive(ACTIVE)->isDelete(NO_DELETE)->findOrFail($shipment['shipment_service_id'])->toArray();
        $total_weight = Package::isShow()->where('shipment_id', $shipment['shipment_id'])->sum('package_charged_weight');

        $convert_weight = $package['package_length'] * $package['package_width'] * $package['package_height'] / $service['service_volumetric_mass'];
        $package['package_converted_weight'] = formatNumberCeil($convert_weight, $package['package_weight']);

        $charged_weight = $package['package_weight'] < $package['package_converted_weight'] ? $package['package_converted_weight'] : $package['package_weight'];
        $package['package_charged_weight'] = formatNumberCeil($charged_weight, $total_weight, $shipment['shipment_service_promotion_id']);

        return $package;
    } catch (\Throwable $th) {
        return null;
    }
}

function onUpdateFinalAmountShipment($shipment_id, $shipment = null)
{
    try {
        if (!$shipment) {
            $shipment = Shipment::findOrFail($shipment_id);
        }
        $shipment->shipment_amount_operating_costs = ShipmentOperatingCost::where('shipment_id', $shipment_id)
            ->selectRaw('SUM(shipment_operating_cost_amount * shipment_operating_cost_quantity) as total_cost')
            ->value('total_cost');

        $amount_service_surcharge = 0;
        $surcharge = hasSurchargeFeeService(null, $shipment);
        if ($surcharge) {
            $amount_service_surcharge = $shipment->shipment_amount_residential + $shipment->shipment_amount_insurance;
        }

        $shipment_total_amount =
            $shipment->shipment_domestic_charges +
            $shipment->shipment_amount_surcharge +
            $shipment->shipment_collection_fee +
            // $shipment->shipment_amount_fsc +
            $shipment->shipment_amount_peak +
            $amount_service_surcharge +
            $shipment->shipment_amount_vat +
            $shipment->shipment_amount_original +
            $shipment->shipment_amount_operating_costs -
            $shipment->shipment_amount_discount;

        // lấy phí tiền của tất cả surcharge goods khi dc scan imported
        $user = User::find($shipment['user_id']);
        if (onCheckPositionAccount(SETTING_SALE_ACCOUNT, $user['position_id'])) {
            $shipment->shipment_final_amount = $shipment_total_amount;
        } else {
            $price_pg_surcharge = getSurchargeGoodPackages($shipment_id)['price'];
            $shipment->shipment_final_amount = $shipment_total_amount + ($price_pg_surcharge ?? 0);
        }

        // Tính lợi nhuận
        if ($shipment['service']['promotion_flg'] == IS_EPACKET) {
            $shipment->shipment_amount_profit = 0;
        } else {
            $shipment->shipment_amount_profit = $shipment->shipment_amount_total_customer - $shipment->shipment_final_amount;
        }

        $shipment->save();
        return $shipment;
    } catch (\Throwable $th) {
        //throw $th;
    }
}


function getDataSurchargeGoodPackages($package_id)
{
    try {
        return PackageSg::isShow()->where('package_id', $package_id)
            ->leftJoin('surcharge_goods', 'package_surchage_goods.surcharge_goods_id', 'surcharge_goods.surcharge_goods_id')
            ->select(
                'package_surchage_goods.package_surchage_good_id',
                'package_surchage_goods.price',
                'package_surchage_goods.count',
                'package_surchage_goods.external_surcharge_name',
                'surcharge_goods.surcharge_goods_name',
            )->get();
    } catch (\Throwable $th) {
        return [];
    }
}

function onCreateHistoryShipment($shipment_id, $user_id, $note)
{
    HistoryEditShipment::create([
        'shipment_id' => $shipment_id,
        'edit_by' => $user_id,
        'edit_at' => Carbon::now()->format('Y-m-d H:i:s'),
        'note' => $note
    ]);
}

function onCheckIsEpacket($service_id, $promotion_flg = null)
{
    try {
        if (isset($promotion_flg)) {
            return $promotion_flg == IS_EPACKET;
        } else {
            $service = Service::find($service_id);
            return $service['promotion_flg'] == IS_EPACKET;
        }
    } catch (\Throwable $th) {
        return false;
    }
}


function onCreateSurchargeGoods($surchage_goods, $package_id)
{
    try {
        $total_amount = 0;
        foreach ($surchage_goods as $key => $value) {
            $packageSg = isset($value['surcharge_goods_id']) ? SurchargeGoods::find($value['surcharge_goods_id']) : null;
            if ($packageSg) {
                $package_sg_created = [
                    "package_id" => $package_id,
                    "surcharge_goods_id" => $packageSg['surcharge_goods_id'],
                    "external_surcharge_name" =>  $packageSg['surcharge_goods_name'],
                    "surcharge_type" => 0,
                    "count" => $value['count'],
                    "price" => $packageSg['surcharge_goods_price'],
                    "external_surcharge_unit_price" => $packageSg['surcharge_goods_price']
                ];
            } else {
                $package_sg_created = [
                    "package_id" => $package_id,
                    "surcharge_goods_id" => "",
                    "external_surcharge_name" => $value['external_surcharge_name'] ?? "Hàng hóa ngoài hệ",
                    "surcharge_type" => 1,
                    "count" => $value['count'],
                    "price" => $value['price'],
                    "external_surcharge_unit_price" => $value['price']
                ];
            }
            $total_amount += $package_sg_created['price'];
            PackageSg::create($package_sg_created);
        }
        return [
            'status' => HTTP_OK,
            'total_amount' => $total_amount,
            'message' => ''
        ];
    } catch (\Throwable $th) {
        return [
            'status' => HTTP_BAD_REQUEST,
            'total_amount' => 0,
            'message' => 'Lỗi không xác định khi khởi tạo hàng hóa phụ thu'
        ];
    }
}

function onCheckValidStatusPackage($package_status, $status)
{
    try {
        $check_warining = HTTP_OK;
        $text = 'Thành công';
        switch ($status) {
            case  SHIPMENT_STATUS_CREATE:
                # code...
                break;
            case SHIPMENT_STATUS_IMPORTED:
                if ($package_status == SHIPMENT_STATUS_IMPORTED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng này đã được scan imported rồi';
                    break;
                }

                if ($package_status == SHIPMENT_STATUS_RETURNED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng này đã bị trả về';
                    break;
                }

                if ($package_status == SHIPMENT_STATUS_EXPORED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng này đã được scan exported';
                    break;
                }

                break;

            case SHIPMENT_STATUS_EXPORED:

                if ($package_status == SHIPMENT_STATUS_RETURNED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng đã được đã bị trả về';
                    break;
                }

                if ($package_status == SHIPMENT_STATUS_CREATE) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng chưa được scan imported';
                    break;
                }

                if ($package_status == SHIPMENT_STATUS_EXPORED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng đã được xuất rồi';
                    break;
                }

                break;

            case SHIPMENT_STATUS_RETURNED:
                if ($package_status == SHIPMENT_STATUS_EXPORED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng đã được xuất không thể trả hàng';
                    break;
                }

                break;

            case SHIPMENT_STATUS_TRANSIT_HN:
                if ($package_status != SHIPMENT_STATUS_EXPORED) {
                    $check_warining = HTTP_BAD_REQUEST;
                    $text = 'Kiện hàng chưa được scan xuất';
                    break;
                }
                break;

            default:
                $check_warining = HTTP_BAD_REQUEST;
                $text = 'Trạng thái cập nhật không tồn tại';
                break;
        }

        return [
            'status' => $check_warining,
            'message' => $text
        ];
    } catch (\Throwable $th) {
        return [
            'status' => HTTP_BAD_REQUEST,
            'message' => 'Lỗi không xác định'
        ];
    }
}

function getPackagesAndMawbByShipment($shipment_id)
{
    try {
        $packages = Package::isShow()
            ->where('packages.shipment_id', $shipment_id)
            ->leftJoin('shipments_tracktry', 'packages.sm_tracktry_id', 'shipments_tracktry.sm_tracktry_id')
            ->select('packages.*', 'shipments_tracktry.awb_code')->get();

        return $packages;
    } catch (\Throwable $th) {
        return [];
    }
}

function getCommistionShipment($total_profit, $salary, $shipment_profit, $kpi)
{
    try {
        return ($total_profit - $salary) / ($total_profit) * $shipment_profit * ($kpi / 100);
    } catch (\Throwable $th) {
        return 0;
    }
}

function getDataSurchargeGoodPackagesByShipment($shipment_id)
{
    try {
        $package_ids = Package::isShow()->where('shipment_id', $shipment_id)->pluck('package_id')->all();
        return PackageSg::isShow()
            ->leftJoin('surcharge_goods', 'package_surchage_goods.surcharge_goods_id', 'surcharge_goods.surcharge_goods_id')
            ->whereIn('package_surchage_goods.package_id', $package_ids)
            ->select(
                'package_surchage_goods.package_surchage_good_id',
                'package_surchage_goods.price',
                'package_surchage_goods.count',
                'package_surchage_goods.external_surcharge_name',
                'surcharge_goods.surcharge_goods_name',
            )->get();
    } catch (\Throwable $th) {
        return [];
    }
}

function getPriceOnePackage($package_id)
{
    try {
        $package = Package::find($package_id);

        $shipment = Shipment::isShow()->where('shipment_id', $package['shipment_id'])->first();
        $packages_weight = Package::isShow()
            ->where('package_status', '!=', SHIPMENT_STATUS_RETURNED)
            ->where('shipment_id', $shipment['shipment_id'])->sum('package_charged_weight');
        $price_service = $shipment['shipment_amount_service'] / $packages_weight;

        return $price_service * $package['package_charged_weight'];
    } catch (\Throwable $th) {
        return 0;
    }
}

function getFirstScanImportShipment($shipment_id)
{
    try {
        $scan = HistoryScan::isShow()
            ->leftJoin('packages', 'history_scan.package_id', 'packages.package_id')
            ->where('packages.delete_flg', NO_DELETE)->where('packages.active_flg', ACTIVE)
            ->where('packages.shipment_id', $shipment_id)
            ->where('history_scan.status', SHIPMENT_STATUS_IMPORTED)
            ->select('history_scan.created_at', 'packages.package_id', 'packages.package_code')
            ->orderBy('history_scan.created_at', 'desc')->first();
        if (!$scan) {
            return [
                'created_at' => 'Kiện hàng chưa được imported'
            ];
        }
        return [
            'scan' => $scan,
            'created_at' => formatDate($scan['created_at'])
        ];
    } catch (\Throwable $th) {
        return [
            'created_at' => 'Lỗi'
        ];
    }
}

function getPakageServiceEpacket($promotion_flg, $package)
{
    try {
        if (!isset($promotion_flg)) {
            $shipment = Shipment::where('shipment_id', $package['shipment_id'])
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->select('promotion_flg')->first();

            $promotion_flg = $shipment['promotion_flg'];
        }
        $check_imported = $promotion_flg == IS_EPACKET && $package['package_status'] != SHIPMENT_STATUS_CREATE;

        return [
            // "package_length" => $check_imported ? $package['package_length_actual'] : $package['package_length'],
            // "package_width" => $check_imported ? $package['package_width_actual'] : $package['package_width'],
            // "package_height" => $check_imported ? $package['package_height_actual'] : $package['package_height'],
            // "package_weight" => $check_imported ? $package['package_weight_actual'] : $package['package_weight'],
            // "package_converted_weight" => $check_imported ? $package['package_converted_weight_actual'] : $package['package_converted_weight'],
            // "package_charged_weight" => $check_imported ? $package['package_charged_weight_actual'] : $package['package_charged_weight'],

            "package_length" => $package['package_length'],
            "package_width" => $package['package_width'],
            "package_height" => $package['package_height'],
            "package_weight" => $package['package_weight'],
            "package_converted_weight" => $package['package_converted_weight'],
            "package_charged_weight" => $package['package_charged_weight'],
        ];
    } catch (\Throwable $th) {
        return [
            "package_length" => 0,
            "package_width" => 0,
            "package_height" => 0,
            "package_weight" => 0,
            "package_converted_weight" => 0,
            "package_charged_weight" => 0,
        ];
    }
}

function getPackagesByMawb($sm_tracktry_id)
{
    try {
        return Package::isShow()
            ->where('packages.sm_tracktry_id', $sm_tracktry_id)
            ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
            ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
            ->select(
                'packages.package_id',
                'packages.sm_tracktry_id',
                'packages.package_code',
                'packages.bag_code',
                DB::raw('COUNT(DISTINCT package_id) AS count_package'),
                DB::raw('COUNT(DISTINCT bag_code) AS count_bag'),
                // DB::raw(
                //     '
                // SUM(CASE 
                //         WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                //         THEN packages.package_weight_actual 
                //         ELSE packages.package_weight 
                //     END) AS gross_weight'
                // ),
                // DB::raw(
                //     '
                // SUM( 
                //     CASE 
                //         WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                //         THEN packages.package_charged_weight_actual 
                //         ELSE packages.package_charged_weight 
                //     END
                // ) AS charged_weight'
                // ),
                DB::raw(
                    'SUM(packages.package_weight) AS gross_weight'
                ),
                DB::raw(
                    'SUM(packages.package_charged_weight) AS charged_weight'
                )
            )
            ->groupBy('packages.sm_tracktry_id')
            ->first();
    } catch (\Throwable $th) {
        return null;
    }
}

function checkHaveProfitShipment($shipment_status)
{
    try {
        $arr_false = [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_RETURNED];

        if (in_array($shipment_status, $arr_false)) {
            return false;
        }
        return true;
    } catch (\Throwable $th) {
        return false;
    }
}


function onReturnStateShipment($shipment)
{
    try {
        if (isset($shipment['receiver_state_name']) && $shipment['receiver_state_name'] != '') {
            return $shipment['receiver_state_name'];
        }
        return $shipment['state']['state_name'];
    } catch (\Throwable $th) {
        return '';
    }
}

function hasSurchargeFeeService($shipment_id, $shipment = null)
{
    try {
        if (!isset($shipment)) {
            $shipment = Shipment::isShow()->find($shipment_id);
        }
        $packages = $shipment['packages'];
        $weight = collect($packages)->sum('package_charged_weight');
        $chargareFee = ServiceSurchargeFee::isShow()->where('service_id', $shipment['shipment_service_id'])
            ->where(function ($query) use ($weight) {
                $query->where(function ($q) use ($weight) {
                    $q->where('condition_type', 1) // Điều kiện trên kg
                        ->where('condition_value', '<=', $weight);
                })
                    ->orWhere(function ($q) use ($weight) {
                        $q->where('condition_type', 0) // Điều kiện dưới kg
                            ->where('condition_value', '>=', $weight);
                    });
            })
            ->first();
        return $chargareFee;
    } catch (\Throwable $th) {
        return null;
    }
}
