<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Carbon\Carbon;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable
{
    protected $table = 'users';
    protected $primaryKey = 'user_id';
    protected $guarded = [];
    use HasFactory, Notifiable, HasApiTokens;


    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    public function wallet()
    {
        return $this->belongsTo(Wallet::class, 'user_id', 'user_id');
    }

    public function saleLinkFwd()
    {
        return $this->belongsTo(SaleLinkFwd::class, 'user_id', 'fwd_id')
        ->where('sale_link_fwds.active_flg', ACTIVE)->where('sale_link_fwds.delete_flg', NO_DELETE);
    }

    public function setPasswordAttribute($password)
    {
        $this->attributes['password'] = bcrypt($password);
    }

    public function position()
    {
        return $this->belongsTo(Position::class, 'position_id', 'position_id')->isActive(ACTIVE)->isDelete(NO_DELETE);
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branch_id', 'branch_id')->isActive(ACTIVE)->isDelete(NO_DELETE);
    }


    public function scopeIsFilters($query, $filters = [])
    {
        if (!empty($filters)) {
            if (
                (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
            ) {
                $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                $query->whereBetween("users.created_at", [$start_date, $end_date]);
            }
            if (isset($filters['keywords']) && $filters['keywords'] != "") {
                $query->where(function ($query) use ($filters) {
                    return $query->where('users.user_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_api_key', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_company_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_phone', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_address', 'like', '%' . $filters['keywords'] . '%')

                        ->orWhereIn('users.user_kpi_id', function ($query) use ($filters) {
                            $query->select('sale_kpi_id')
                                ->from('sale_kpis')
                                ->where('kpi_name', 'like', '%' . $filters['keywords'] . '%');
                        });
                });
            }
            if (isset($filters['active_flg']) && $filters['active_flg'] != "") {
                $query->where('users.active_flg', $filters['active_flg']);
            }
            if (isset($filters['position_id']) && $filters['position_id'] != "") {
                $query->where('users.position_id', $filters['position_id']);
            }
            if (isset($filters['branch_id']) && $filters['branch_id'] != "") {
                $query->where('users.branch_id', $filters['branch_id']);
            }
        }
        return $query;
    }

    public function scopeIsDelete($query, $delete_flg)
    {
        return $query->where('users.delete_flg', $delete_flg);
    }

    public function scopeIsActive($query, $active_flg)
    {
        return $query->where('users.active_flg', $active_flg);
    }

    public function scopeIsShow($query)
    {
        return $query->where('users.active_flg', ACTIVE)->where('users.delete_flg', NO_DELETE);
    }

    public function leader()
    {
        return $this->belongsTo(SaleTeam::class, 'user_id', 'leader_id')->isShow();
    }

    public function team()
    {
        return $this->belongsTo(SaleMember::class, 'user_id', 'user_id')->isShow()
            ->leftJoin('sales_teams', 'sales_members.sale_team_id', 'sales_teams.sale_team_id')
            ->leftJoin('users', 'sales_teams.leader_id', 'users.user_id');
    }

    public function scopeIsPosition($query, $position_id)
    {
        return $query->where('users.position_id', $position_id);
    }

    public function scopeIsBranch($query, $branch_id)
    {
        return $query->where('users.branch_id', $branch_id);
    }

    public function scopeIsPaginate($query, $paginate)
    {
        $limit = !empty($paginate['limit']) ? $paginate['limit'] : 10;
        $page = !empty($paginate['page']) ? $paginate['page'] : 1;

        return $query->paginate($limit, ['*'], 'page', $page);
    }

    public function getUserDebitTypeNameAttribute()
    {
        try {
            $debits = config('constans.constans.debits');
            if ($this->user_debit_type == 0) {
                return 'Chưa cập nhật công nợ';
            }
            return $debits[$this->user_debit_type];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function getUserAddress1NameAttribute()
    {
        try {
            return collect(config('constans.address'))->pluck('name')->toArray()[$this->user_address_1];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function getUserAddress2NameAttribute()
    {
        try {
            return collect(config('constans.address.' . $this->user_address_1 . '.districts'))->pluck('name')->toArray()[$this->user_address_2];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function getUserAddress3NameAttribute()
    {
        try {
            return collect(config('constans.address.' . $this->user_address_1 . '.districts.' . $this->user_address_2 . '.wards'))->pluck('name')->toArray()[$this->user_address_3];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function priceList()
    {
        return $this->belongsTo(PriceListUser::class, 'user_id', 'user_id')->isShow()
            ->leftJoin('price_list', 'price_list_user.price_list_id', 'price_list.price_list_id')
            ->select('price_list.*');
    }
}
