<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Transfer extends Model
{
    protected $table = 'transfer';
    protected $primaryKey = 'transfer_id';
    protected $guarded = [];
    use HasFactory;

    public function transfer_shipments()
    {
        return $this->hasMany(TransferShipment::class, 'transfer_id', 'transfer_id')
            ->leftJoin('shipments', 'transfer_shipment.shipment_id', '=', 'shipments.shipment_id')
            ->select('transfer_shipment.*', 'shipments.shipment_code');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'user_id');
    }

    public function reviewer()
    {
        return $this->belongsTo(User::class, 'transfer_reviewer_id', 'user_id');
    }

    public function scopeIsActive($query, $active_flg = ACTIVE)
    {
        return $query->where('transfer.active_flg', $active_flg);
    }

    public function scopeIsDelete($query, $delete_flg = NO_DELETE)
    {
        return $query->where('transfer.delete_flg', $delete_flg);
    }

    public function scopeIsFilters($query, $filters = [])
    {
        $query->leftJoin('users', 'transfer.user_id', '=', 'users.user_id')
            ->leftJoin('users as reviewer', 'transfer.transfer_reviewer_id', '=', 'reviewer.user_id')
            ->leftJoin('transfer_shipment', 'transfer.transfer_id', 'transfer_shipment.transfer_id')
            ->leftJoin('shipments', 'transfer_shipment.shipment_id', 'shipments.shipment_id');
        if (!empty($filters)) {
            if (
                (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
            ) {
                $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                $query->whereBetween("transfer.created_at", [$start_date, $end_date]);
            }

            if (isset($filters['keywords']) && $filters['keywords'] != "") {
                $query->where(function ($query) use ($filters) {
                    return $query->where('users.user_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_phone', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('reviewer.user_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('reviewer.user_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('reviewer.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('reviewer.user_phone', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('shipments.shipment_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('transfer.receiver_phone', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('transfer.receiver_name', 'like', '%' . $filters['keywords'] . '%');
                });
            }

            if (isset($filters['transfer_status']) && $filters['transfer_status'] != "") {
                $query->where('transfer.transfer_status', $filters['transfer_status']);
            }
        }
        $query->groupBy('transfer.transfer_id');
        return $query->select('transfer.*');
    }

    public function getTransferStatusNameAttribute()
    {
        $arr = [
            INACTIVE => 'Chưa xác nhận',
            ACTIVE => 'Đã xác nhận',
        ];
        return $arr[$this->transfer_status];
    }
}
