<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Shipment extends Model
{
    protected $table = 'shipments';
    protected $primaryKey = 'shipment_id';
    protected $guarded = [];
    use HasFactory;

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'user_id');
    }

    public function service()
    {
        return $this->belongsTo(Service::class, 'shipment_service_id', 'service_id');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'shipment_branch_id', 'branch_id');
    }

    public function country()
    {
        return $this->belongsTo(Country::class, 'receiver_country_id', 'country_id');
    }

    public function state()
    {
        return $this->belongsTo(State::class, 'receiver_state_id', 'state_id');
    }

    public function city()
    {
        return $this->belongsTo(City::class, 'receiver_city_id', 'city_id');
    }

    public function debit()
    {
        return $this->belongsTo(Debit::class, 'shipment_debit_id', 'debit_id');
    }

    public function label()
    {
        return $this->belongsTo(LabelNz::class, 'shipment_id', 'shipment_id');
    }

    public function costs()
    {
        return $this->belongsTo(FwdCostByService::class, 'shipment_service_id', 'service_id')
            ->where('fwd_id', $this->user_id);
    }

    public function packages()
    {
        return $this->hasMany(Package::class, 'shipment_id', 'shipment_id');
    }

    public function invoices()
    {
        return $this->hasMany(Invoice::class, 'shipment_id', 'shipment_id');
    }

    public function shipmentOperatingCosts()
    {
        return $this->hasMany(ShipmentOperatingCost::class, 'shipment_id', 'shipment_id');
    }


    public function scopeIsShow($query)
    {
        return $query->where('shipments.delete_flg', NO_DELETE)->where('shipments.active_flg', ACTIVE);
    }

    public function scopeIsDelete($query, $delete_flg)
    {
        return $query->where('shipments.delete_flg', $delete_flg);
    }

    public function scopeIsActive($query, $active_flg)
    {
        return $query->where('shipments.active_flg', $active_flg);
    }

    public function scopeIsFiltersInDashboard($query, $filters = [])
    {
        if (!empty($filters)) {
            if (
                (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
            ) {
                $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                $query->whereBetween("shipments.created_at", [$start_date, $end_date]);
            }

            if (isset($filters['shipment_branch_id']) && $filters['shipment_branch_id'] != "") {
                $query->where("shipments.shipment_branch_id", $filters['shipment_branch_id']);
            }

            if (isset($filters['accountant_status']) && $filters['accountant_status'] != "") {
                $query->where("shipments.accountant_status", $filters['accountant_status']);
            }

            if (isset($filters['shipment_service_id']) && $filters['shipment_service_id'] != "") {
                $query->where("shipments.shipment_service_id", $filters['shipment_service_id']);
            }
            if (isset($filters['checked_payment_status']) && $filters['checked_payment_status'] != "") {
                $query->where("shipments.checked_payment_status", $filters['checked_payment_status']);
            }

            if (isset($filters['position_id']) && $filters['position_id'] != "") {
                $query->where("users.position_id", $filters['position_id']);
            }

            if (isset($filters['shipment_status']) && $filters['shipment_status'] != "") {
                $query->where("shipments.shipment_status", $filters['shipment_status']);
            }
            if (isset($filters['user_id']) && $filters['user_id'] != "") {
                $query = $query->where('shipments.user_id', $filters['user_id']);
            }
            if (isset($filters['shipment_payment_status']) && $filters['shipment_payment_status'] != "") {
                $query = $query->where('shipments.shipment_payment_status', $filters['shipment_payment_status']);
            }

            if (isset($filters['keywords']) && $filters['keywords'] != "") {
                $query->where(function ($query) use ($filters) {
                    $filter_by = isset($filters['filter_by']) ? $filters['filter_by'] : 'shipment_code';
                    return $query->where("$filter_by", 'like', '%' . $filters['keywords'] . '%');
                });
            }

            if (isset($filters['shipment_payment_method']) && $filters['shipment_payment_method'] != "") {
                $query = $query->where('shipments.shipment_payment_method', $filters['shipment_payment_method']);
            }
            if (isset($filters['shipment_paid_by']) && $filters['shipment_paid_by'] != "") {
                $query = $query->where('shipments.shipment_paid_by', $filters['shipment_paid_by']);
            }
        }
        return $query;
    }

    public function scopeIsFilters($query, $filters = [])
    {
        $query->leftJoin('packages', 'packages.shipment_id', '=', 'shipments.shipment_id');
        if (!empty($filters)) {
            if (
                (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
            ) {
                $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();

                $query->whereBetween("shipments.created_at", [$start_date, $end_date]);
            }

            if (isset($filters['shipment_branch_id']) && $filters['shipment_branch_id'] != "") {
                $query->where("shipments.shipment_branch_id", $filters['shipment_branch_id']);
            }

            if (isset($filters['accountant_status']) && $filters['accountant_status'] != "") {
                $query->where("shipments.accountant_status", $filters['accountant_status']);
            }

            if (!empty($filters['shipment_service_id'])) {
                $filters['shipment_service_id'] = is_array($filters['shipment_service_id']) ? $filters['shipment_service_id'] : [$filters['shipment_service_id']];
                $query->whereIn("shipments.shipment_service_id", $filters['shipment_service_id']);
            }

            if (isset($filters['position_id']) && $filters['position_id'] != "") {
                $query->where("users.position_id", $filters['position_id']);
            }

            if (isset($filters['shipment_status']) && $filters['shipment_status'] != "") {
                $query->where("shipments.shipment_status", $filters['shipment_status']);
            }
            if (isset($filters['checked_payment_status']) && $filters['checked_payment_status'] != "") {
                $query->where("shipments.checked_payment_status", $filters['checked_payment_status']);
            }
            if (isset($filters['user_id']) && $filters['user_id'] != "") {
                $query = $query->where('shipments.user_id', $filters['user_id']);
            }
            if (isset($filters['shipment_payment_status']) && $filters['shipment_payment_status'] != "") {
                $query = $query->where('shipments.shipment_payment_status', $filters['shipment_payment_status']);
            }

            if (isset($filters['keywords']) && $filters['keywords'] != "") {
                if (isset($filters['filter_by']) && $filters['filter_by'] != "") {
                    $query->where(function ($query) use ($filters) {
                        $filter_by = isset($filters['filter_by']) ? $filters['filter_by'] : 'shipment_code';
                        return $query->where("$filter_by", 'like', '%' . $filters['keywords'] . '%');
                    });
                } else {
                    $query->where(function ($query) use ($filters) {
                        return $query->where('shipments.shipment_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('packages.package_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('shipments.receiver_contact_name', 'like', '%' . $filters['keywords'] . '%');
                    });
                }
            }

            if (isset($filters['shipment_payment_method']) && $filters['shipment_payment_method'] != "") {
                $query = $query->where('shipments.shipment_payment_method', $filters['shipment_payment_method']);
            }
            if (isset($filters['shipment_paid_by']) && $filters['shipment_paid_by'] != "") {
                $query = $query->where('shipments.shipment_paid_by', $filters['shipment_paid_by']);
            }
        }
        $query->select('shipments.*')->groupBy('shipments.shipment_id');
        return $query;
    }

    public function scopeIsFiltersConstantsKey($query, $filters)
    {
        // Dò theo ngày tháng
        if (
            (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
            && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
        ) {
            $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
            $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
            $query->whereBetween("shipments.created_at", [$start_date, $end_date]);
        }
        // Dò theo chi nhánh
        if (isset($filters['shipment_branch_id']) && $filters['shipment_branch_id'] != "") {
            $query->where("shipments.shipment_branch_id", $filters['shipment_branch_id']);
        }
        // Dò theo trạng thái kế toán (Đã xác nhận/ chưa xác nhận)
        if (isset($filters['accountant_status']) && $filters['accountant_status'] != "") {
            $query->where("shipments.accountant_status", $filters['accountant_status']);
        }
        // Dò theo dịch vụ
        if (!empty($filters['shipment_service_id'])) {
            $filters['shipment_service_id'] = is_array($filters['shipment_service_id']) ? $filters['shipment_service_id'] : [$filters['shipment_service_id']];
            $query->whereIn("shipments.shipment_service_id", $filters['shipment_service_id']);
        }
        // Dò theo trạng thái của gói hàng (created, imported,...)
        if (isset($filters['shipment_status']) && $filters['shipment_status'] != "") {
            $query->where("shipments.shipment_status", $filters['shipment_status']);
        }
        // Dò theo tài khoản sở hữu đơn
        if (isset($filters['user_id']) && $filters['user_id'] != "") {
            $query = $query->where('shipments.user_id', $filters['user_id']);
        }
        // Dò theo trạng thái thanh toán
        if (isset($filters['shipment_payment_status']) && $filters['shipment_payment_status'] != "") {
            $query = $query->where('shipments.shipment_payment_status', $filters['shipment_payment_status']);
        }
        // Dò theo phương thức thanh toán
        if (isset($filters['shipment_payment_method']) && $filters['shipment_payment_method'] != "") {
            $query = $query->where('shipments.shipment_payment_method', $filters['shipment_payment_method']);
        }
        // Dò theo loại thanh toán
        if (isset($filters['shipment_paid_by']) && $filters['shipment_paid_by'] != "") {
            $query = $query->where('shipments.shipment_paid_by', $filters['shipment_paid_by']);
        }

        return $query;
    }

    public function scopeIsPaginate($query, $paginate)
    {
        $limit = isset($paginate['limit']) ? $paginate['limit'] : 10;
        $page = isset($paginate['page']) ? $paginate['page'] : 1;
        return $query->paginate($limit, ['*'], 'page', $page);
    }


    public function getShipmentSignatureNameAttribute()
    {
        try {
            $arr = [
                SHIPMENT_SIGNATURE_NO => 'Không',
                SHIPMENT_SIGNATURE_YES => 'Có',
            ];
            return $arr[$this->shipment_signature_flg];
        } catch (\Throwable $th) {
            return "";
        }
    }

    public function getShipmentExportAsNameAttribute()
    {
        try {
            $arr = config('constans.constans.invoice_exports_as');
            return $arr[$this->shipment_export_as];
        } catch (\Throwable $th) {
            return "";
        }
    }

    public function getShipmentStatusNameAttribute()
    {
        try {
            $arr = config('constans.constans.shipment_status');
            return $arr[$this->shipment_status] ?? "";
        } catch (\Throwable $th) {
            return "";
        }
    }

    public function getSenderCityNameAttribute()
    {
        try {
            return collect(config('constans.address'))->pluck('name')->toArray()[$this->sender_city];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function getSenderDistrictNameAttribute()
    {
        try {
            return getAreaVietName($this->sender_city)['districts'][$this->sender_district];
            // return collect(config('constans.address.' . $this->sender_city . '.districts'))->pluck('name')->toArray()[$this->sender_district];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function getSenderWardNameAttribute()
    {
        try {
            return getAreaVietName($this->sender_city, $this->sender_district)['wards'][$this->sender_ward];
            // return collect(config('constans.address.' . $this->sender_city . '.districts.' . $this->sender_district . '.wards'))->pluck('name')->toArray()[$this->sender_ward];
        } catch (\Throwable $th) {
            return '';
        }
    }

    public function getShipmentPaymentMethodNameAttribute()
    {
        try {
            $arr = config('constans.constans.payment_status_name');
            return $arr[$this->shipment_payment_method] ?? 'Không xác định';
        } catch (\Throwable $th) {
            return "Lỗi";
        }
    }

    public function checked_payment()
    {
        return $this->belongsTo(CheckedPaymentCash::class, 'shipment_code', 'code');
    }
    


    
}
