<?php

namespace App\Http\Controllers;

use App\Models\City;
use App\Models\User;
use App\Models\State;
use App\Models\Branch;
use App\Models\Country;
use App\Models\Package;
use App\Models\Service;
use App\Models\Shipment;
use App\Models\Transfer;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\TransferShipment;
use Illuminate\Support\Facades\DB;

class TransferController extends Controller
{
    public function displayTransferList()
    {
        return view('transfer.list');
    }

    public function getTransferList(Request $request)
    {
        try {
            $request_data = $request->all();
            $filters = $request_data['filters'] ?? [];
            $filters = is_array($filters) ? $filters : json_decode($filters, true);

            $is_api = $request->is('api/*');
            $transfer = Transfer::with(['user', 'reviewer', 'transfer_shipments'])
                ->isFilters($filters)
                ->isDelete(NO_DELETE)
                ->orderByDesc('created_at');

            if (!isGetAllData()) {
                $transfer = $transfer->where('transfer.user_id', request()->user()->user_id);
            }

            $transfer = $transfer->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);
            $html = view('transfer.pagination', compact('transfer'))->render();

            return response()->json([
                'status' => HTTP_OK,
                'html' => $is_api ? $transfer : $html,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function displayTransferCreate()
    {
        try {
            $html = view('transfer.create')->render();

            return response()->json([
                'status' => HTTP_OK,
                'html' => $html,
            ]);
        } catch (\Exception $e) {
            return showMessageError();
        }
    }

    public function displayTransferEdit(Request $request)
    {
        try {
            $request_data = $request->all();
            $transfer = Transfer::with(['user', 'reviewer']);

            if (!isGetAllData()) {
                $transfer = $transfer->where('transfer.user_id', request()->user()->user_id);
            }

            $transfer = $transfer->findOrFail($request_data['transfer_id'])->toArray();

            $transfer_shipment_codes = TransferShipment::where('transfer_id', $transfer['transfer_id'])
                ->leftJoin('shipments', 'transfer_shipment.shipment_id', '=', 'shipments.shipment_id')
                ->select('shipments.shipment_code')
                ->get()
                ->toArray();

            $transfer['transfer_images'] = readFiles($transfer['transfer_images']);

            $html = view('transfer.edit', compact('transfer'))->render();

            return response()->json([
                'status' => HTTP_OK,
                'html' => $html,
                'transfer_images' => $transfer['transfer_images'],
                'transfer_shipment_codes' => $transfer_shipment_codes,
            ]);
        } catch (\Exception $e) {
            return showMessageError();
        }
    }

    public function handleTransferUpdate(Request $request)
    {
        $request->validate([
            'transfer_images' => 'required',
            'transfer_content' => 'required|not_regex:/^<p>&nbsp;<\/p>$/',
            'transfer_shipment_codes.*.shipment_code' => 'exists:shipments,shipment_code',
        ], [
            'transfer_images.required' => 'Vui lòng chọn ảnh',
            'transfer_content.required' => 'Vui lòng nhập nội dung',
            'transfer_content.not_regex' => 'Vui lòng nhập nội dung',
            'transfer_shipment_codes.*.shipment_code.exists' => 'Không tồn tại shipment code',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $transfer_images = [];

            foreach ($request_data['transfer_images'] as $image) {
                $image = convertImagesBase64ToDirectory([$image], PUBLIC_PATH_TRANSFER_IMAGE);
                $transfer_images[] = $image[0];
            }

            $data_update = [
                'transfer_images' => json_encode($transfer_images),
                'transfer_content' => $request_data['transfer_content'],
                'receiver_name' => $request_data['receiver_name'] ?? null,
                'receiver_address' => $request_data['receiver_address'] ?? null,
                'receiver_phone' => $request_data['receiver_phone'] ?? null,
            ];

            if (isset($request_data['transfer_id'])) {
                $transfer = Transfer::find($request_data['transfer_id']);
                if (!$transfer) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy khai hàng');
                }
                Transfer::find($request_data['transfer_id'])->update($data_update);
            } else {
                $data_update['user_id'] = request()->user()->user_id;
                $transfer = Transfer::create($data_update);
            }

            TransferShipment::where('transfer_id', $transfer['transfer_id'])->delete();
            if (!empty($request_data['transfer_shipment_codes'])) {
                $shipment_codes = collect($request_data['transfer_shipment_codes'])->flatten()->toArray();
                $shipment_ids = Shipment::whereIn('shipment_code', $shipment_codes)->pluck('shipment_id')->toArray();

                foreach ($shipment_ids as $shipment_id) {
                    TransferShipment::create([
                        'transfer_id' => $transfer['transfer_id'],
                        'shipment_id' => $shipment_id,
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Cập nhật khai hàng thành công',
                    'icon' => 'success'
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return showMessageError();
        }
    }

    private function extractTransferDataFromHtml($html)
    {
        $dom = new \DOMDocument();
        @$dom->loadHTML(mb_convert_encoding((string) $html, 'HTML-ENTITIES', 'UTF-8'));
        $xpath = new \DOMXPath($dom);

        // Tìm tất cả các đoạn HTML cần xử lý
        $sections_query = "//p[contains(text(), 'Dịch vụ:') or strong[contains(text(), 'Dịch vụ:')]]";
        $sections = $xpath->query($sections_query);

        $data = [];

        foreach ($sections as $section) {
            $result = [];

            // Dịch vụ
            $result['service'] = trim(str_replace('Dịch vụ:', '', $section->textContent ?? ''));

            // Tạo đơn nháp với lô hàng này
            $create_draft_query = "following-sibling::p[contains(text(), 'Tạo đơn nháp với lô hàng này') or strong[contains(text(), 'Tạo đơn nháp với lô hàng này')]]";
            $result['create_draft'] = trim(str_replace('Tạo đơn nháp với lô hàng này(Có/Không):', '', $xpath->query($create_draft_query, $section)->item(0)->textContent ?? ''));

            // Mã tài khoản
            $account_code_query = "following-sibling::p[contains(text(), 'Mã tài khoản:') or strong[contains(text(), 'Mã tài khoản:')]]";
            $result['account_code'] = trim(str_replace('Mã tài khoản:', '', $xpath->query($account_code_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Tên người gửi
            $sender_name_query = "following-sibling::p[contains(text(), 'Tên người gửi:') or strong[contains(text(), 'Tên người gửi:')]]";
            $result['sender_name'] = trim(str_replace('Tên người gửi:', '', $xpath->query($sender_name_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Địa chỉ người gửi
            $sender_address_query = "following-sibling::p[contains(text(), 'Địa chỉ người gửi(Địa chỉ, Xã, Huyện, Tỉnh):') or strong[contains(text(), 'Địa chỉ người gửi(Địa chỉ, Xã, Huyện, Tỉnh):')]]";
            $result['sender_address'] = trim(str_replace('Địa chỉ người gửi(Địa chỉ, Xã, Huyện, Tỉnh):', '', $xpath->query($sender_address_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Số điện thoại người gửi
            $sender_phone_query = "following-sibling::p[contains(text(), 'Số điện thoại người gửi:') or strong[contains(text(), 'Số điện thoại người gửi:')]]";
            $result['sender_phone'] = trim(str_replace('Số điện thoại người gửi:', '', $xpath->query($sender_phone_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Chi nhánh
            $branch_query = "following-sibling::p[contains(text(), 'Chi nhánh(HCM/HN/DN):') or strong[contains(text(), 'Chi nhánh(HCM/HN/DN):')]]";
            $result['branch'] = trim(str_replace('Chi nhánh(HCM/HN/DN):', '', $xpath->query($branch_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Kiện hàng
            $result['packages'] = [];
            $current_node = $section->nextSibling;
            while ($current_node && $current_node->nodeName === 'p') {
                $text = $current_node->textContent;
                if (strpos($text, 'Số kiện:') !== false) {
                    $package = [];
                    $package['package_quantity'] = trim(str_replace('Số kiện:', '', $text));
                    $current_node = $current_node->nextSibling;
                    $package['package_length'] = trim(str_replace('Chiều dài:', '', $current_node->textContent ?? ''));
                    $current_node = $current_node->nextSibling;
                    $package['package_width'] = trim(str_replace('Chiều rộng:', '', $current_node->textContent ?? ''));
                    $current_node = $current_node->nextSibling;
                    $package['package_height'] = trim(str_replace('Chiều cao:', '', $current_node->textContent ?? ''));
                    $current_node = $current_node->nextSibling;
                    $package['package_weight'] = trim(str_replace('Số ký:', '', $current_node->textContent ?? ''));
                    $result['packages'][] = $package;
                } elseif (strpos($text, 'Tên hàng hóa:') !== false) {
                    break;
                }
                $current_node = $current_node->nextSibling;
            }

            // Tên hàng hóa
            $goods_name_query = "following-sibling::p[contains(text(), 'Tên hàng hóa:') or strong[contains(text(), 'Tên hàng hóa:')]]";
            $result['goods_name'] = trim(str_replace('Tên hàng hóa:', '', $xpath->query($goods_name_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Giá trị hàng hóa
            $value_query = "following-sibling::p[contains(text(), 'Giá trị hàng hóa:') or strong[contains(text(), 'Giá trị hàng hóa:')]]";
            $result['value'] = trim(str_replace('Giá trị hàng hóa:', '', $xpath->query($value_query, $section->nextSibling)->item(0)->textContent ?? ''));

            // Thu khách
            $collect_amount_query = "following-sibling::p[contains(text(), 'Thu khách:') or strong[contains(text(), 'Thu khách:')]]";
            $result['collect_amount'] = trim(str_replace('Thu khách:', '', $xpath->query($collect_amount_query, $section->nextSibling)->item(0)->textContent ?? 0));

            // Loại thanh toán
            $payment_type_query = "following-sibling::p[contains(text(), 'Loại thanh toán(NỢ/SAU):') or strong[contains(text(), 'Loại thanh toán(NỢ/SAU):')]]";
            $result['payment_type'] = trim(str_replace('Loại thanh toán(NỢ/SAU):', '', $xpath->query($payment_type_query, $section->nextSibling)->item(0)->textContent ?? 'SAU'));

            $data[] = $result;
        }

        return $data;
    }

    private function handleCreateShipments($request_data)
    {
        try {
            // Crawl dữ liệu từ nội dung
            $crawled_shipments = $this->extractTransferDataFromHtml($request_data['transfer_content']);

            if (empty($crawled_shipments)) {
                return [
                    'status' => HTTP_OK,
                    'message' => [
                        'title' => 'Thành công',
                        'text' => 'Không có dữ liệu để tạo đơn hàng',
                        'icon' => 'success'
                    ]
                ];
            }

            foreach ($crawled_shipments as $shipment_tmp) {
                $shipment = [];

                // Nếu không tạo đơn nháp thì bỏ qua
                if (Str::upper($shipment_tmp['create_draft']) != 'CÓ')
                    continue;

                $explode_receiver_address = explode(',', $request_data['receiver_address']);
                $country_code = trim($explode_receiver_address[4]) ?? '';
                $country = Country::where('country_code', 'like', '%' . $country_code . '%')->firstOrFail();
                $state_code = trim($explode_receiver_address[2]) ?? '';
                $state = State::where('state_code', 'like', '%' . $state_code . '%')
                    ->where('country_id', $country['country_id'])
                    ->first();
                $city_name = trim($explode_receiver_address[1]) ?? '';
                $city = City::where('city_name', 'like', '%' . $city_name . '%')
                    ->where('country_id', $country['country_id'])
                    ->first();

                $user = User::where('user_code', $shipment_tmp['account_code'])->firstOrFail();

                $branch = Branch::where('branch_name', $shipment_tmp['branch'])->firstOrFail();

                $service = Service::leftJoin('country_and_service_links', 'services.service_id', '=', 'country_and_service_links.service_id')
                    ->where('services.service_name', $shipment_tmp['service'])
                    ->where('country_and_service_links.country_id', $country['country_id'])
                    ->select('services.*')
                    ->firstOrFail();

                $explode_sender_address = explode(',', $shipment_tmp['sender_address']);
                $city_name = trim($explode_sender_address[3]) ?? '';
                $district_name = trim($explode_sender_address[2]) ?? '';
                $ward_name = trim($explode_sender_address[1]) ?? '';

                $cities = collect(config('constans.area'))->pluck('name')->toArray();
                $filtered_city = array_filter($cities, function ($city) use ($city_name) {
                    return Str::upper($city) == Str::upper($city_name);
                });
                $filtered_city = !empty($filtered_city) ? collect(array_keys($filtered_city))->first() : 0;

                $districts = collect(config('constans.area.' . $filtered_city . '.districts'))->pluck('name')->toArray();
                $filtered_district = array_filter($districts, function ($district) use ($district_name) {
                    return Str::upper($district) == Str::upper($district_name);
                });
                $filtered_district = !empty($filtered_district) ? collect(array_keys($filtered_district))->first() : 0;

                $wards = collect(config('constans.area.' . $filtered_city . '.districts.' . $filtered_district . '.wards'))->pluck('name')->toArray();
                $filtered_ward = array_filter($wards, function ($ward) use ($ward_name) {
                    return Str::upper($ward) == Str::upper($ward_name);
                });
                $filtered_ward = !empty($filtered_ward) ? collect(array_keys($filtered_ward))->first() : 0;

                $user['user_address_1'] = $filtered_city;
                $user['user_address_2'] = $filtered_district;
                $user['user_address_3'] = $filtered_ward;

                $packages = $shipment_tmp['packages'];

                $shipment['shipment_code'] = initializationShipmentCode();
                $shipment['user_id'] = $user['user_id'];
                $shipment['shipment_service_id'] = $service['service_id'];
                $shipment['shipment_signature_flg'] = SHIPMENT_SIGNATURE_NO;
                $shipment['shipment_branch_id'] = $branch['branch_id'];
                $shipment['shipment_status'] = SHIPMENT_STATUS_CREATE_BILL;
                $shipment['shipment_goods_name'] = $shipment_tmp['goods_name'] ?? '';
                $shipment['shipment_value'] = $shipment_tmp['value'] ?? 0;
                $shipment['shipment_paid_by'] = Str::upper($shipment_tmp['payment_type']) == 'NỢ' ? SHIPMENT_PAID_BY_REMAINING : SHIPMENT_PAID_BY_WALLET;

                $shipment['sender_company_name'] = $user['user_company_name'];
                $shipment['sender_contact_name'] = $shipment_tmp['sender_name'];
                $shipment['sender_telephone'] = $shipment_tmp['sender_phone'];
                $shipment['sender_city'] = $user['user_address_1'] ?? 0;
                $shipment['sender_district'] = $user['user_address_2'] ?? 0;
                $shipment['sender_ward'] = $user['user_address_3'] ?? 0;
                $shipment['sender_address'] = $shipment_tmp['sender_address'];

                $shipment['receiver_company_name'] = $request_data['receiver_name'];
                $shipment['receiver_contact_name'] = $request_data['receiver_name'];
                $shipment['receiver_telephone'] = $request_data['receiver_phone'];
                $shipment['receiver_country_id'] = $country['country_id'];
                $shipment['receiver_state_id'] = $state['state_id'] ?? 0;
                $shipment['receiver_state_name'] = $state['state_name'] ?? '';
                $shipment['receiver_city_id'] = $city['city_id'] ?? 0;
                $shipment['receiver_postal_code'] = $explode_receiver_address[3] ?? '';
                $shipment['receiver_address_1'] = $explode_receiver_address[0] ?? '';
                $shipment['receiver_sms_phone'] = $request_data['receiver_phone'];

                $shipment['shipment_amount_total_customer'] = str_replace(['.', ','], '', $shipment_tmp['collect_amount']);

                $shipment['import_approval'] = INACTIVE;
                $shipment['shipment_payment_status'] = PENDING;
                $shipment['accountant_status'] = PENDING;

                $shipment_created = Shipment::create($shipment);

                // Tạo mới gói hàng
                $this->onCreatePackages($packages, $shipment_created);

                // Cập nhật giá trị số tiền vận chuyển
                $result_save_amount = onSaveAmountShipment($shipment_created['shipment_id']);

                if (!onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user['position_id'])) {
                    // Cập nhật giá trị chi phí vận hành
                    onUpdateOperatingCostsAmount($shipment_created['shipment_id']);
                }

                // Tính giá trị cuối cùng của đơn và lợi nhuận 
                onUpdateFinalAmountShipment($shipment_created['shipment_id']);
            }

            return [
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Tạo đơn hàng thành công',
                    'icon' => 'success'
                ]
            ];
        } catch (\Throwable $th) {
            return [
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'message' => [
                    'title' => 'Thất bại',
                    'text' => 'Tạo đơn hàng thất bại',
                    'icon' => 'error'
                ]
            ];
        }
    }

    /**
     * Tạo mới gói hàng
     * @param mixed $packages
     * @param mixed $shipment
     * @return void
     */
    private function onCreatePackages($packages, $shipment)
    {
        $service = $shipment['service'];
        $branch = $shipment['branch'];
        if ($branch['branch_name'] == 'HCM') {
            $branch_name = 'HO CHI MINH, VN';
        } else if ($branch['branch_name'] == 'HN') {
            $branch_name = 'HA NOI, VN';
        } else if ($branch['branch_name'] == 'DN') {
            $branch_name = 'DA NANG, VN';
        } else {
            $branch_name = 'HO CHI MINH, VN';
        }
        $total_weight = getTotalPackage($packages, $service['service_volumetric_mass']);
        foreach ($packages as $package) {
            $package['shipment_id'] = $shipment['shipment_id'];
            $package['package_converted_weight'] = $package['package_length'] * $package['package_width'] * $package['package_height'] / $service['service_volumetric_mass'];
            $package['package_charged_weight'] = $package['package_weight'] < $package['package_converted_weight'] ? $package['package_converted_weight'] : $package['package_weight'];
            $package['package_charged_weight'] = formatNumberCeil($package['package_charged_weight'], $total_weight, $service['promotion_flg']);
            $package['package_description'] = '';
            $package_quantity = $package['package_quantity'];

            for ($i = 1; $i <= $package_quantity; $i++) {
                $package_code = getCodePackage();
                $package['package_code'] = $package_code['package_code'];
                $package['package_hawb_code'] = $package_code['package_code'];
                $package['package_quantity'] = 1;

                // Tính giá trị của gói hàng
                $package['package_price'] = computedPricePackage($package['package_charged_weight'], $shipment);

                Package::create($package);

                onCreatePackageTrackingStatus($package['package_code'], $branch_name, 'LABEL CREATED');
            }
        }
    }

    public function handleTransferDelete(Request $request)
    {
        DB::beginTransaction();
        try {
            $request_data = $request->all();

            $transfer = Transfer::where('transfer_id', $request_data['transfer_id']);

            if (!isGetAllData()) {
                $transfer = $transfer->where('user_id', request()->user()->user_id);
            }

            $transfer->update([
                'delete_flg' => DELETED,
            ]);

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Xóa khai hàng thành công',
                    'icon' => 'success'
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleTransferApprove(Request $request)
    {
        DB::beginTransaction();
        try {
            $request_data = $request->all();

            $transfer = Transfer::where('transfer_id', $request_data['transfer_id']);

            if (!isGetAllData()) {
                $transfer = $transfer->where('user_id', request()->user()->user_id);
            }

            $transfer->update([
                'transfer_reviewer_id' => request()->user()->user_id,
                'transfer_status' => ACTIVE,
            ]);

            $transfer = Transfer::where('transfer_id', $request_data['transfer_id'])->first();
            $result = $this->handleCreateShipments($transfer);

            DB::commit();

            $text = ' và ' . $result['message']['text'];
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => "Duyệt khai hàng thành công $text",
                    'icon' => 'success'
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
