<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Ticket;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\TicketMessage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class TicketController extends Controller
{
    public function displayTickets()
    {
        $total_tickets_pending = Ticket::where('ticket_status', TICKET_STATUS_PENDING)->count();
        $total_tickets_processing = Ticket::where('ticket_status', TICKET_STATUS_PROCESSING)->count();
        return view('supports.tickets.main', compact('total_tickets_pending', 'total_tickets_processing'));
    }

    public function getSetupTicket()
    {
        try {
            $path = 'public/config/system-settings.json';
            $position_ids = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_TICKET)['position_ids'];
            $is_use_ticket = in_array(auth()->user()['position_id'], isset($position_ids) && $position_ids != "" ? $position_ids : []);
            if ($is_use_ticket) {
                $tiket_kinds = config('constans.constans.tiket_kinds');
            }

            return response()->json([
                'status' => 200,
                'is_use_ticket' => $is_use_ticket,
                'tiket_kinds' => $tiket_kinds,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getTicketsList(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;

            $arr_status = [
                TICKET_STATUS_PENDING => 'Chưa xử lý',
                TICKET_STATUS_PROCESSING => 'Đang xử lý',
                TICKET_STATUS_DONE => 'Đã xong',
            ];

            $tickets = Ticket::with(['user', 'ticketMessage'])
                ->where('tickets.ticket_status', $request_data['ticket_status'])
                ->where('tickets.active_flg', ACTIVE)
                ->where('tickets.delete_flg', NO_DELETE)
                ->orderBy('tickets.created_at', 'DESC')
                ->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            $total_tickets_pending = Ticket::where('ticket_status', TICKET_STATUS_PENDING)->count();
            $total_tickets_processing = Ticket::where('ticket_status', TICKET_STATUS_PROCESSING)->count();

            $html = view('supports.tickets.pagination', compact('tickets', 'arr_status'))->render();
            return response()->json([
                'status' => 200,
                'total_tickets_pending' => $total_tickets_pending,
                'total_tickets_processing' => $total_tickets_processing,
                'html' => $html,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getTicketsByUser(Request $request)
    {
        try {
            $request_data = $request->all();

            $tickets = [];
            $arr_status = [
                TICKET_STATUS_PENDING => 'pending',
                TICKET_STATUS_PROCESSING => 'processing',
                TICKET_STATUS_DONE => 'done',
            ];

            foreach ($arr_status as $key_status => $status) {
                $request_data['ticket_status'] = $key_status;
                $tickets[$status] = $this->getTicketsByUserAndStatus($request_data);
            }

            return response()->json([
                'status' => 200,
                'data' => $tickets,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getTicketsByUserAndStatus($request_data)
    {
        try {
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? request()->user() : auth()->user();

            $tickets = Ticket::where('user_id', $user['user_id'])
                ->where('ticket_status', $request_data['ticket_status'])
                ->orderBy('created_at', 'DESC')
                ->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            if ($is_api) {
                return $tickets;
            }

            return view('supports.tickets.me.pagination', compact('tickets'))->render();
        } catch (\Throwable $th) {
            return [];
        }
    }

    public function handleAddTicket(Request $request)
    {
        $request->validate([
            'ticket_kind' => 'required:in:1,2,0'
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? $request->user() : auth()->user();

            if (!isset($request_data['ticket_title']) || strlen($request_data['ticket_title']) < 1 || strlen($request_data['ticket_title']) >= 255) {
                return response()->json([
                    'status' => 500,
                    'message' => [
                        'title' => 'Cảnh báo',
                        'text' => strlen($request_data['ticket_title']) >= 255 ? 'Tiêu đề không được vượt quá 255 lí tự' : 'Không được để trống tiêu đề.',
                        'icon' => 'warning',
                    ],
                ]);
            }
            if (!isset($request_data['ticket_message_content']) || strlen($request_data['ticket_message_content']) < 1) {
                return response()->json([
                    'status' => 500,
                    'message' => [
                        'title' => 'Cảnh báo',
                        'text' => 'Không được để trống nội dung.',
                        'icon' => 'warning',
                    ],
                ]);
            }
            $ticket = Ticket::create([
                'user_id' => $user['user_id'],
                'ticket_kind' => $request_data['ticket_kind'],
                'ticket_transaction_code' => $request_data['ticket_transaction_code'],
                'ticket_title' => $request_data['ticket_title'],
            ]);

            TicketMessage::create([
                'ticket_id' => $ticket['ticket_id'],
                'sender_id' => $user['user_id'],
                'ticket_message_content' => $request_data['ticket_message_content'],
                'ticket_type' => TICKET_TYPE_TEXT,
            ]);

            for ($i = 0; $i < 3; $i++) {
                if ($request->hasFile('file_' . $i) || isset($request_data['file_' . $i])) {
                    $ticket_message = new TicketMessage();
                    $ticket_message->ticket_id = $ticket->ticket_id;
                    $ticket_message->sender_id = $user->user_id;
                    $ticket_message->ticket_type = TICKET_TYPE_FILE;
                }
                if ($request->hasFile('file_' . $i)) {
                    $upload_path = public_path(PUBLIC_PATH_TICKET);
                    if (!File::isDirectory($upload_path)) {
                        File::makeDirectory($upload_path, 0777, true, true);
                    }
                    $file_name = Carbon::now()->format('d-m-Y') . '-' . $request->file('file_' . $i)->getClientOriginalName();
                    $request->file('file_' . $i)->move($upload_path, $file_name);
                    $file_path = PUBLIC_PATH_TICKET . '/' . $file_name;
                    $ticket_message->ticket_message_content = $file_path;
                } else if (isset($request_data['file_' . $i]) && strlen($request_data['file_' . $i]) > 200) {
                    $file_name = Carbon::now()->format('d-m-Y') . '-' . Str::random(30);
                    $ticket_message->ticket_message_content = convertImagesBase64ToDirectory([$request_data['file_' . $i]], PUBLIC_PATH_TICKET, $file_name, $request_data['extension_' . $i])[0];
                }
                if ($request->hasFile('file_' . $i) || isset($request_data['file_' . $i])) {
                    $ticket_message->save();
                }
            }

            DB::commit();
            return response()->json([
                'status' => 200,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Tạo ticket thành công.',
                    'icon' => 'success',
                ],
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function getDetailTicket(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? $request->user() : auth()->user();

            $ticket = Ticket::with([
                'ticketMessages' => function ($query) {
                    $query->orderBy('created_at', 'ASC');
                }
            ])->where('ticket_id', $request_data['ticket_id'])->first();

            if (!$ticket) {
                return response()->json([
                    'status' => 500,
                    'message' => [
                        'title' => 'Cảnh báo',
                        'text' => 'Không tìm thấy ticket.',
                        'icon' => 'warning',
                    ],
                ]);
            }

            if ($is_api) {
                return response()->json([
                    'status' => 200,
                    'data' => $ticket,
                ]);
            }

            $html = view('supports.tickets.me.detail', compact('ticket'))->render();
            return response()->json([
                'status' => 200,
                'html' => $html,
                'ticket_status' => $ticket['ticket_status'],
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleAddTicketMessage(Request $request)
    {
        $request->validate([
            'ticket_id' => 'required',
            'ticket_message_content' => 'required',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? $request->user() : auth()->user();
            $is_answer = isset($request_data['is_answer']) ? json_decode($request_data['is_answer']) : false;

            Ticket::whereIn('ticket_status', [TICKET_STATUS_PENDING, TICKET_STATUS_PROCESSING])->findOrFail($request_data['ticket_id']);

            $ticket_message = new TicketMessage();
            $ticket_message->ticket_id = $request_data['ticket_id'];
            $ticket_message->sender_id = $is_answer ? null : $user['user_id'];
            $ticket_message->answer_id = $is_answer ? $user['user_id'] : null;

            if ($request->hasFile('ticket_message_content')) {
                $upload_path = public_path(PUBLIC_PATH_TICKET);
                if (!File::isDirectory($upload_path)) {
                    File::makeDirectory($upload_path, 0777, true, true);
                }
                $file_name = Carbon::now()->format('d-m-Y') . '-' . $request->file('ticket_message_content')->getClientOriginalName();
                $request->file('ticket_message_content')->move($upload_path, $file_name);
                $file_path = PUBLIC_PATH_TICKET . '/' . $file_name;
                $ticket_message->ticket_message_content = $file_path;
                $ticket_message->ticket_type = TICKET_TYPE_FILE;
            } else if (isset($request_data['ticket_message_content']) && strlen($request_data['ticket_message_content']) > 200 && isset($request_data['ticket_message_content_extension'])) {
                $file_name = Carbon::now()->format('d-m-Y') . '-' . Str::random(30);
                $ticket_message->ticket_message_content = convertImagesBase64ToDirectory([$request_data['ticket_message_content']], PUBLIC_PATH_TICKET, $file_name, $request_data['ticket_message_content_extension'])[0];
                $ticket_message->ticket_type = TICKET_TYPE_FILE;
            }else{
                $ticket_message->ticket_message_content = $request_data['ticket_message_content'];
                $ticket_message->ticket_type = TICKET_TYPE_TEXT;
            }

            $ticket_message->save();

            DB::commit();
            return response()->json([
                'status' => 200,
                'ticket_message_content' => $ticket_message['ticket_message_content'],
                'created_at' => formatFullDate($ticket_message['created_at']),
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function showSingleChatTicket(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;

            $ticket = Ticket::with([
                'user',
                'ticketMessages' => function ($query) {
                    $query->orderBy('created_at', 'ASC');
                },
                'shipments' => function ($query) {
                    $query->take(10)
                        ->select('shipment_code', 'user_id')
                        ->orderBy('created_at', 'DESC');
                }
            ])->where('ticket_id', $request_data['ticket_id'])->first();

            if (!$ticket) {
                return response()->json([
                    'status' => 500,
                    'message' => [
                        'title' => 'Cảnh báo',
                        'text' => 'Không tìm thấy ticket.',
                        'icon' => 'warning',
                    ],
                ]);
            }

            if ($is_api) {
                return response()->json([
                    'status' => 200,
                    'data' => $ticket,
                ]);
            }

            $html = view('supports.tickets.single-chat', compact('ticket'))->render();
            return response()->json([
                'status' => 200,
                'html' => $html
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleUpdateTicketStatus(Request $request)
    {
        $request->validate([
            'ticket_id' => 'required',
            'ticket_status' => 'required',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();

            $ticket = Ticket::findOrFail($request_data['ticket_id']);
            $ticket->ticket_status = $request_data['ticket_status'];
            $ticket->save();

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Cập nhật trạng thái ticket thành công.',
                    'icon' => 'success',
                ],
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleUpdateTicketNote(Request $request)
    {
        $request->validate([
            'ticket_id' => 'required',
            'ticket_note' => 'required',
        ], [
            'ticket_note.required' => 'Không được để trống ghi chú.',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();

            $ticket = Ticket::findOrFail($request_data['ticket_id']);
            $ticket->ticket_note = $request_data['ticket_note'];
            $ticket->save();

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Cập nhật ghi chú ticket thành công.',
                    'icon' => 'success',
                ],
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}