<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Debit;
use App\Models\Shipment;
use App\Models\Statement;
use Illuminate\Http\Request;
use App\Models\StatementItem;
use App\Exports\ExportStatement;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx\Rels;

class StatementController extends Controller
{
    public function displayIndex()
    {
        // return viewBaoTri();
        $position_id = onGetPositionIdFWDAccount();
        $fwds = User::isShow()->where('position_id', $position_id)->get();
        return view('accounting-management.statement.index', compact('fwds'));
    }

    public function showListStatement(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $filters = $request_data['filters'] ?? null;
            $statements = Statement::isShow()
                ->leftJoin('statement_items', 'statements.statement_id', 'statement_items.statement_id')
                ->leftJoin('debits', 'statement_items.debit_id', 'debits.debit_id')
                ->leftJoin('shipments', 'statement_items.debit_id', 'shipments.shipment_debit_id')
                ->leftJoin('users as customer', 'statements.customer_id', 'customer.user_id')
                ->leftJoin('users as creator', 'statements.ceator_id', 'creator.user_id');

            $statements = $statements->select(
                'customer.user_company_name as customer_company',
                'customer.position_id as customer_position_id',
                'customer.user_code as customer_code',
                'creator.user_contact_name as creator_name',
                DB::raw('SUM(shipment_final_amount+ shipment_amount_fsc) as total_price'),
                DB::raw('SUM(CASE WHEN shipment_payment_status = 0 THEN (shipment_final_amount + shipment_amount_fsc) ELSE 0 END) as payment_price'),
                'statements.*',
                DB::raw("
                    CASE 
                        WHEN statements.statement_status = 0 THEN 'Chờ thanh toán'
                        WHEN statements.statement_status = 1 THEN 'Đã thanh toán'
                        ELSE 'Không xác định'
                    END as status_label
                ")

            );

            if ($filters) {
                $date_range = $filters['date_range'] ?? null;
                $start_date = isset($date_range['start_date'])
                    ? Carbon::parse($date_range['start_date'])->format('Y-m-d H:i:s')
                    : null;

                $end_date = isset($date_range['end_date'])
                    ? Carbon::parse($date_range['end_date'])->endOfDay()->format('Y-m-d H:i:s')
                    : null;

                if ($start_date && $end_date) {
                    $statements = $statements->whereBetween("statements.created_at", [$start_date, $end_date]);;
                }

                if (isset($filters['keywords']) && $filters['keywords'] != '') {
                    $keywords = $filters['keywords'];
                    $statements = $statements->where(function ($query) use ($keywords) {
                        $query = $query->where('customer.user_code', 'like', '%' . $keywords . '%')
                            ->orWhere('customer.user_contact_name', 'like', '%' . $keywords . '%')
                            ->orWhere('customer.user_company_name', 'like', '%' . $keywords . '%')
                            ->orWhere('statements.statement_code', 'like', '%' . $keywords . '%')
                            ->orWhere('creator.user_contact_name', 'like', '%' . $keywords . '%');
                    });
                }

                if (isset($filters['statement_status']) && $filters['statement_status'] != '') {
                    $statements = $statements->where('statements.statement_status', $filters['statement_status']);
                }
                if (isset($filters['checked_payment_status']) && $filters['checked_payment_status'] != "") {
                    $statements->where("statements.checked_payment_status", $filters['checked_payment_status']);
                }
                if (isset($filters['statement_payment_method']) && in_array($filters['statement_payment_method'], [0, 1, 2, 3])) {
                    $statements = $statements->where('statements.statement_payment_method', $filters['statement_payment_method']);
                }
            }

            $statements = $statements->orderBy('statements.created_at', 'desc')
                ->groupBy('statements.statement_id')
                ->isPaginate($request->paginater);

            if ($is_api) {
                # code...
            } else {
                $html = view('accounting-management.statement.ajax.table-statement', compact('statements'))->render();

                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleCreateStatement(Request $request)
    {
        $this->validate($request, [
            'statement_start' => 'required|date',
            'statement_end' => 'required|date',
            'fwd_id' => 'required|numeric',
        ]);
        try {
            $request_data = $request->all();
            $fwd = User::isShow()->find($request_data['fwd_id']);
            if (!$fwd) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy tài khoản fwd');
            }
            $statement_start = Carbon::parse($request_data['statement_start'])->startOfDay()->format('Y-m-d H:i:s');
            $statement_end = Carbon::parse($request_data['statement_end'])->endOfDay()->format('Y-m-d H:i:s');
            if (Carbon::parse($statement_end)->lessThanOrEqualTo(Carbon::parse($statement_start))) {
                return returnResponse(HTTP_BAD_REQUEST, 'Thời gian kết thúc phải lớn hơn thời gian bắt đầu');
            }
            $debits = Debit::isShow()->where('customer_id', $fwd['user_id'])
                ->whereBetween('created_at', [$statement_start, $statement_end])
                ->select('debit_id')->pluck('debit_id');

            if (count($debits) < 1) {
                return returnResponse(HTTP_NOT_FOUND, 'Không có debit nào được tạo trong khoảng thời gian được chọn', null, 'warning');
            }

            DB::beginTransaction();
            $statement = Statement::create([
                'statement_code' => initializationCode('statements', 'statement_id', 'STM'),
                'customer_id' => $request_data['fwd_id'],
                'ceator_id' => auth()->id()
            ]);

            foreach ($debits as $key => $value) {
                StatementItem::create([
                    'statement_id' => $statement['statement_id'],
                    'debit_id' => $value,
                ]);
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Khởi tạo statement thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleCancelStatement(Request $request)
    {
        $this->validate($request, ['statement_code' => 'required']);
        try {
            $request_data = $request->all();
            if (!isGetAllData()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn không có đủ quyền hạn', null, 'warning');
            }
            $statement = Statement::isShow()->where('statement_code', $request->get('statement_code'))->first();
            if (!$statement) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy statement', null, 'warning');
            }
            if ($statement['statement_status'] == SUCCESS) {
                return returnResponse(HTTP_BAD_REQUEST, 'Đơn đã được thanh toán, không thể xóa');
            }
            DB::beginTransaction();
            Statement::find($statement['statement_id'])->update(['delete_flg' => DELETED]);
            // Lịch sử hoạt động tài khoản
            $data_logs = [
                'old_data' => $statement,
                'new_data' => $request_data
            ];
            onCreateAccountActivityLogs('Quản lý hệ thống', 'Hủy lệnh statement mã ' . $statement['statement_code'], $data_logs);
            DB::commit();
            return returnResponse(HTTP_OK, 'Đã hủy lệnh thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function displayStatementDetail(Request $request)
    {
        try {
            $statement_code = $request->query('statement_code');

            $statement = Statement::isShow()
                ->where('statement_code', $statement_code)->first();
            if (!$statement) {
                return viewError404();
            }
            $debits = StatementItem::isShow()->where('statement_items.statement_id', $statement['statement_id'])
                ->leftJoin('debits', 'statement_items.debit_id', 'debits.debit_id')
                ->leftJoin('shipments', 'statement_items.debit_id', 'shipments.shipment_debit_id')
                ->select(
                    DB::raw('SUM(shipment_final_amount + shipment_amount_fsc) as total_price'),
                    DB::raw('SUM(CASE WHEN shipment_payment_status = 0 THEN (shipment_final_amount + shipment_amount_fsc) ELSE 0 END) as payment_price'),
                    'debits.*',
                    DB::raw("
                        CASE 
                            WHEN debits.debit_status = 0 THEN 'Chờ thanh toán'
                            WHEN debits.debit_status = 1 THEN 'Đã thanh toán'
                            WHEN debits.debit_status = 2 THEN 'Quá hạn'
                            WHEN debits.debit_status = 3 THEN 'Đang chờ duyệt'
                            ELSE 'Không xác định'
                        END as status_label
                    "),
                )->groupBy('debits.debit_id')->orderBy('debits.created_at', 'desc')->get();

            $payment_amount = $debits->sum('payment_price');
            return view('accounting-management.statement.detail', compact('statement', 'debits'));
        } catch (\Throwable $th) {
            return viewError404();
        }
    }

    public function onPaymentStatement(Request $request)
    {
        $this->validate($request, [
            'statement_code' => 'required',
            'statement_images' => 'required',
            // 'statement_note' => 'required',
            'statement_amount' => 'required|numeric|min:0',
            'statement_payment_method' => 'required|numeric|min:0|max:3',
        ]);
        try {
            $request_data = $request->all();
            $user = $request->user();
            $statement_code = $request_data['statement_code'] ?? null;
            $statement = Statement::isShow()->where('statement_code', $statement_code)->first();
            if (!$statement) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy statement');
            }

            $statement_images = [];
            if (!empty($request_data['statement_images'])) {
                $request_data['statement_images'] = is_string($request_data['statement_images']) ? json_decode($request_data['statement_images'], true) : $request_data['statement_images'];

                $statement_images = convertImagesBase64ToDirectory($request_data['statement_images'], PUBLIC_PATH_STATEMENT_IMAGE);
            }

            DB::beginTransaction();
            Statement::find($statement['statement_id'])
                ->update([
                    'statement_status' => SUCCESS,
                    'statement_payment_method' => $request_data['statement_payment_method'] ?? BANK,
                    'statement_amount' => $request_data['statement_amount'],
                    'statement_note' => $request_data['statement_note'] ?? '',
                    'statement_images' => $statement_images ?? json_encode($statement_images)
                ]);
            $date_pay = Carbon::now()->format('Y-m-d H:i:s');
            $check = onCreateCheckedPaymentCash(
                $statement['statement_code'],
                $user['user_id'],
                $statement['customer_id'],
                2,
                $request_data['statement_amount'],
                $date_pay,
                $request_data['statement_payment_method'] ?? BANK,
                empty($statement_images) ? '' : json_encode($statement_images),
                $request_data['statement_amount_bank'] ?? null,
                $request_data['statement_amount_cash'] ?? null,
            );

            if ($check['status'] != 200) {
                DB::rollBack();
                return returnResponse($check['status'], $check['message']);
            }

            $debit_ids = StatementItem::isShow()
                ->where('statement_id', $statement['statement_id'])
                ->select('debit_id')->pluck('debit_id');
            if (!empty($debit_ids)) {
                $debit_ids = Debit::isShow()->whereIn('debit_id', $debit_ids)
                    ->where('debit_status', '!=', 1)->select('debit_id')->pluck('debit_id');

                Debit::isShow()->whereIn('debit_id', $debit_ids)
                    ->update([
                        'debit_status' => SUCCESS,
                        'debit_note' => "Thanh toán qua statement[$statement_code]",
                        'debit_payment_date' => $date_pay,
                        'payment_method' => $request_data['statement_payment_method'] ?? BANK,
                        'debit_images' => $statement_images ?? json_encode($statement_images)
                    ]);

                Shipment::isShow()->whereIn('shipment_debit_id', $debit_ids)
                    ->where('shipment_payment_status', '!=', 1)
                    ->update([
                        'shipment_payment_status' => SUCCESS,
                        'shipment_payment_date' => $date_pay,
                        'shipment_payment_des' => "Thanh toán qua statement[$statement_code]",
                        'shipment_payment_method' => $request_data['statement_payment_method'] ?? BANK,
                        'accountant_status' => SUCCESS,
                    ]);

                // $debit_advance_amount = Debit::isShow()->whereIn('debit_id', $debit_ids)
                // ->select('debit_advance_amount')->sum('debit_advance_amount');
                // if ($debit_advance_amount > 0) {
                //     updateUserRemainingLimit2($statement['customer_id'], $debit_advance_amount, MINUS, 'Thanh toán công nợ qua statement mã ' . $statement['statement_code']);
                // }
                onUpdateUserRemainingLimit($statement['customer_id']);
            }

            DB::commit();
            return returnResponse(HTTP_OK, 'Thanh toán thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            foreach ($statement_images as $key => $image) {
                if (File::exists($image)) {
                    File::delete($image);
                }
            }
            return showMessageError();
        }
    }

    public function exportBillStateMent(Request $request)
    {
        $data = [];
        $statement_code = $request->query('statement_code');
        $statement = Statement::isShow()->where('statement_code', $statement_code)->first();

        if (!$statement) {
            return response()->json([
                'status' => HTTP_NOT_FOUND,
                'message' => [
                    'icon' => 'error',
                    'title' => 'Thông báo',
                    'text' => 'Không tìm thấy statement'
                ]
            ], HTTP_NOT_FOUND);
        }
        $export_name = $statement_code . '.xlsx';

        $debits = StatementItem::isShow()->where('statement_items.statement_id', $statement['statement_id'])
            ->leftJoin('debits', 'statement_items.debit_id', 'debits.debit_id')
            ->leftJoin('shipments', 'statement_items.debit_id', 'shipments.shipment_debit_id')
            ->select(
                DB::raw('SUM(shipment_final_amount + shipment_amount_fsc) as total_price'),
                DB::raw('SUM(CASE WHEN shipment_payment_status = 0 THEN (shipment_final_amount + shipment_amount_fsc) ELSE 0 END) as payment_price'),
                'debits.*',
                DB::raw("
                        CASE 
                            WHEN debits.debit_status = 0 THEN 'Chờ thanh toán'
                            WHEN debits.debit_status = 1 THEN 'Đã thanh toán'
                            WHEN debits.debit_status = 2 THEN 'Quá hạn'
                            WHEN debits.debit_status = 3 THEN 'Đang chờ duyệt'
                            ELSE 'Không xác định'
                        END as status_label
                    "),
            )->groupBy('debits.debit_id')->orderBy('debits.created_at', 'desc')->get();

        $data['statement'] = $statement;
        $data['creater'] = User::find($statement['ceator_id']);
        $data['customer'] = User::find($statement['customer_id']);
        $data['debits'] = $debits;


        return Excel::download(new ExportStatement($data), $export_name);
    }
}
