<?php

namespace App\Http\Controllers;

use App\Utils\ESMS;
use App\Models\SmsHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SmsManagerController extends Controller
{
    public function displayIndex()
    {
        return view('sms-manager.index');
    }

    public function showHistorySms(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        try {
            $check_bill_code = isset($request_data['bill_code']);
            $sms_histories = SmsHistory::isShow();
            if ($check_bill_code) {
                $sms_histories = $sms_histories->whereNotNull('sms_histories.bill_code');
            }
            $sms_histories = $sms_histories->leftJoin('users', 'sms_histories.send_by', 'users.user_id')
                ->select(
                    'sms_histories.*',
                    'user_contact_name',
                    DB::raw("
                        CASE 
                            WHEN sms_histories.status = 0 THEN 'Gửi thất bại'
                            WHEN sms_histories.status = 1 THEN 'Gửi thành công'
                            ELSE 'Không xác định'
                        END as status_label
                    ")
                );

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['filter_by'])) {
                    $sms_histories = $sms_histories->where($filters['filter_by'], 'like', '%' . $filters['keywords'] . '%');
                } else {
                    $sms_histories = $sms_histories->where(function ($query) use ($filters) {
                        $query->where("users.user_contact_name", 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere("sms_histories.bill_code", 'like', '%' . $filters['keywords'] . '%');
                    });
                }
            }

            $sms_histories = $sms_histories
            ->orderBy('sms_histories.created_at', 'desc')
            ->isPaginate(isset($request_data['paginate']) ? $request_data['paginate'] : null);

            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "sms_histories" => $sms_histories->toArray()['data'],
                ]);
            } else {
                $html = view('sms-manager.ajax.tables-sms', compact('sms_histories', 'check_bill_code'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function displaySend()
    {
        return view('sms-manager.send-sms');
    }

    public function handleSendSmsHolidayTest(Request $request)
    {
        $request->validate([
            'holiday_name' => 'required|min:1|max:20',
            'congratulation' => 'required|min:1|max:80',
            'phone_test' => 'required|min:10|max:12',
        ]);
        try {
            $request_data = $request->all();
            $holiday_name = $request_data['holiday_name'];
            $congratulation = $request_data['congratulation'];
            $phone_test = $request_data['phone_test'];

            $content = "KIKI CO.LTD xin tran trong cam on Quy khach da tin tuong san pham va dich vu cua chung toi. Nhan dip $holiday_name , KIKI CO.LTD kinh chuc $congratulation";

            $esms = new ESMS();
            $response = $esms->sendSmsOtpOrCustomerCare($phone_test, $content);

            $request_data['status'] = isset($response['CodeResult']) ? $response['CodeResult'] == 100 ? HTTP_OK : $response['CodeResult'] : HTTP_INTERNAL_SERVER_ERROR;
            $request_data['message'] = $response['status'] ? 'Gửi tin nhắn thành công.' : 'Gửi tin nhắn thất bại.';
            return response()->json([
                'status' => $request_data['status'],
                'message' => [
                    'title' => 'Thông báo',
                    'text' => $request_data['message'],
                    'icon' => $request_data['status'] == HTTP_OK ? 'success' : 'error',
                ]
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleSendSmsHoliday(Request $request)
    {
        $request->validate([
            'holiday_name' => 'required|min:1|max:20',
            'congratulation' => 'required|min:1|max:80',
            'phones' => 'required',
        ]);
        try {
            $request_data = $request->all();
            $holiday_name = $request_data['holiday_name'];
            $congratulation = $request_data['congratulation'];
            $phones = $request_data['phones'];

            $content = "KIKI CO.LTD xin tran trong cam on Quy khach da tin tuong san pham va dich vu cua chung toi. Nhan dip $holiday_name , KIKI CO.LTD kinh chuc $congratulation";
            $esms = new ESMS();
            foreach ($phones as $phone) {
                if (isset($phone)) {
                    $response = $esms->sendSmsOtpOrCustomerCare($phone, $content);
                    $sms_data = [
                        "phone" => $phone,
                        "content" => $content,
                        "send_by" => auth()->id(),
                        "status" => isset($response['CodeResult']) && $response['CodeResult'] == 100 ? 1 : 0,
                        "status_content" => isset($response['CodeResult']) ? config('constans.content-other.status_sms.' . $response['CodeResult']) : "Xảy ra lỗi khi gọi đến esms",
                        "sms_history_kind" => SMS_HISTORY_HOLIDAY,
                    ];
                    SmsHistory::create($sms_data);
                }
            }

            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thông báo',
                    'text' => 'Gửi tin nhắn thành công.',
                    'icon' => 'success',
                ]
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getUsersSendSmsHoliday(Request $request)
    {
        try {
            $request_data = $request->all();
            $filters = isset($request_data['filters']) ? $request_data['filters'] : [];

            $subQuery = SmsHistory::isShow()->selectRaw('MAX(sms_history_id) as latest_sms_history_id')
                ->where('status', 1)
                ->leftJoin('packages', 'sms_histories.bill_code', 'packages.package_hawb_code')
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->groupBy('shipments.receiver_contact_name', 'shipments.receiver_telephone');

            $users = SmsHistory::isShow()->isFilters($filters)->where('sms_histories.status', 1)
                ->joinSub($subQuery, 'latest_sms', function ($join) {
                    $join->on('sms_histories.sms_history_id', '=', 'latest_sms.latest_sms_history_id');
                })
                ->leftJoin('packages', 'sms_histories.bill_code', 'packages.package_hawb_code')
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('users', 'shipments.user_id', 'users.user_id')
                ->whereNotNull('shipments.receiver_telephone')
                ->select(
                    'sms_histories.sms_history_id',
                    'shipments.receiver_contact_name as name',
                    'shipments.receiver_telephone as phone',
                    'sms_histories.created_at'
                )
                ->orderBy('sms_histories.created_at', 'desc')
                ->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            $html = view('sms-manager.ajax.send-sms-holiday-pagination', compact('users'))->render();

            return response()->json([
                "status" => HTTP_OK,
                "html" => $html
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function checkedAllUsersSendSmsHoliday(){
        try {
            $prefixes = [
                '086', '096', '097', '098', // Viettel
                '032', '033', '034', '035', '036', '037', '038', '039', // Viettel
                '089', '090', '093', // Mobifone
                '070', '079', '077', '076', '078', // Mobifone
                '088', '091', '094', // Vinaphone
                '083', '084', '085', '081', '082', // Vinaphone
                '092', '056', '058', // Vietnamobile
                '099', '059', // Gmobile
            ];

            $subQuery = SmsHistory::isShow()->selectRaw('MAX(sms_history_id) as latest_sms_history_id')
                ->where('status', 1)
                ->leftJoin('packages', 'sms_histories.bill_code', 'packages.package_hawb_code')
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->groupBy('shipments.receiver_contact_name', 'shipments.receiver_telephone');

            $users = SmsHistory::isShow()->where('sms_histories.status', 1)
                ->joinSub($subQuery, 'latest_sms', function ($join) {
                    $join->on('sms_histories.sms_history_id', '=', 'latest_sms.latest_sms_history_id');
                })
                ->leftJoin('packages', 'sms_histories.bill_code', 'packages.package_hawb_code')
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('users', 'shipments.user_id', 'users.user_id')
                ->whereNotNull('shipments.receiver_telephone')
                ->where(function ($query) use ($prefixes) {
                    foreach ($prefixes as $prefix) {
                        $query->orWhere('shipments.receiver_telephone', 'like', $prefix . '%');
                    }
                })
                ->select(
                    'shipments.receiver_telephone as phone'
                )
                ->orderBy('sms_histories.created_at', 'desc')
                ->get();

            $phones = $users->pluck('phone')->toArray();

            return response()->json([
                "status" => HTTP_OK,
                "phones" => $phones
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }
}
