<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\City;
use App\Models\User;
use App\Models\State;
use App\Models\Branch;
use App\Models\Wallet;
use App\Models\Country;
use App\Models\Invoice;
use App\Models\Package;
use App\Models\Service;
use App\Models\Receiver;
use App\Models\Shipment;
use App\Utils\EtowerApi;
use App\Models\PackageSg;
use App\Utils\LabelNzApi;
use App\Utils\DmxSmartApi;
use App\Exports\ExportFile;
use App\Models\OrderPickup;
use App\Models\SaleLinkFwd;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\OperatingCost;
use App\Models\WeightConversion;
use App\Models\WalletFluctuation;
use Illuminate\Support\Facades\DB;
use App\Models\HistoryEditShipment;
use App\Models\HistoryScan;
use Illuminate\Support\Facades\File;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\ShipmentOperatingCost;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use App\Models\LabelNz as ModelsLabelNz;

class ShipmentController extends Controller
{

    public function displayShipmentCreate()
    {
        // updateDebitStatus(); // Cập nhật trạng thái 

        $user = auth()->user();
        $address = address($user['user_address_1'], $user['user_address_2']);
        $receivers = Receiver::isUser(auth()->id())->isActive(ACTIVE)->isDelete(NO_DELETE)->select('receiver_id', 'receiver_contact_name')->get()->toArray();
        $countries = Country::isActive(ACTIVE)->isDelete(NO_DELETE)->select('country_id', 'country_name')->get()->toArray();
        $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->select('branch_id', 'branch_name')->get()->toArray();
        $package_types = config('constans.constans.package_types');
        $invoice_exports_as = config('constans.constans.invoice_exports_as');
        $invoice_units = config('constans.constans.invoice_units');
        $sales = null;
        if (onCheckPositionAccount(SETTING_DOCUMENT_ACCOUNT, $user['position_id'])) {
            $sales = User::isShow()->where('position_id', getPositionAccountSetting(SETTING_SALE_ACCOUNT));
            $sales = $sales->select('users.user_id', 'users.user_code', 'users.user_contact_name')->get();
        };

        return view('shipment.create', compact('user', 'sales', 'address', 'receivers', 'countries', 'branchs', 'package_types', 'invoice_exports_as', 'invoice_units'));
    }
    public function showServiceType(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $service_types = config('constans.constans.service_types');
            unset($service_types[1]);

            return response()->json([
                "status" => HTTP_OK,
                'service_types' => $service_types,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    /**
     * Xử lý tạo shipment
     * @param \Illuminate\Http\Request $request
     * @return mixed|\Illuminate\Http\JsonResponse
     */
    public function handleShipmentCreate(Request $request)
    {
        // Gọi hàm validate dữ liệu khi tạo shipment
        $this->onValidateUpdateShipment($request);
        DB::beginTransaction();
        try {
            // Lấy dữ liệu từ request
            $request_data = $request->all();
            // Lấy dữ liệu user
            if ($request->user) {
                $user = $request->user;
            } else {
                $user = $request->user();
            }

            // Lấy dữ liệu shipment từ request
            $shipment = $request_data['shipment'];
            // Lấy dữ liệu gói hàng từ request
            $packages = $request_data['shipment']['packages'];
            // Lấy dữ liệu hóa đơn từ request
            $invoices = !empty($request_data['shipment']['invoices']) ? $request_data['shipment']['invoices'] : [];

            // Kiểm tra xem người dùng có đồng ý với điều khoản sử dụng dịch vụ không
            if (!isset($shipment['agree_terms_use_service']) || !json_decode($shipment['agree_terms_use_service'])) {
                return returnResponse(HTTP_UNPROCESSABLE_ENTITY, 'Bạn cần đồng ý với điều khoản của chúng tôi', null, 'warning');
            }
            $user_sale = null;
            $shipment['user_id'] = $user['user_id'];
            // Nếu có sale_id thì kiểm tra xem sale có tồn tại không
            if (isset($shipment['sale_id'])) {
                $user_sale = User::find($shipment['sale_id']);
                if (!isset($user_sale)) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy tài khoản sale');
                }
                $shipment['user_id'] = $user_sale['user_id'];
                $shipment['document_id'] = $user['user_id'];
            }

            // Kiểm tra xem người dùng có muốn lưu thông tin người nhận không
            $save_receiver_flg = isset($shipment['save_receiver_flg']) ? json_decode($shipment['save_receiver_flg']) : false;

            $save_receiver_id = null;
            // Nếu save_receiver_flg = true thì lưu thông tin người nhận
            $receiver = [];
            if ($save_receiver_flg) {
                if (onCheckPositionAccounts(SETTING_DOCUMENT_ACCOUNT, $user['position_id'])) {
                    $save_receiver_id = $shipment['document_id'];
                }
                // Cập nhật hoặc tạo mới thông tin người nhận
                $receiver = $this->onUpdateOrCreateReceiver($shipment, $user_sale, $save_receiver_id, $user);
            }

            // Khởi tạo mã shipment
            $shipment['shipment_code'] = initializationShipmentCode();
            // Khởi tạo lô hàng
            $shipment_created = $this->onCreateShipment($shipment, $user_sale, $user, $receiver, $save_receiver_flg, $save_receiver_id);

            $count_check_e_package = collect($packages)->where('package_quantity', '>', 1)->count();
            // Nếu dịch vụ là ePacket và tổng số gói hàng lớn hơn 1 thì hiện thông báo lỗi
            if ($shipment_created['service']['promotion_flg'] == IS_EPACKET && (count($packages) > 1 || $count_check_e_package > 0)) {
                return response()->json([
                    'status' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => [
                        'icon' => 'error',
                        'title' => 'Thông báo',
                        'text' => 'Dịch vụ ePacket chỉ cho phép tạo 1 gói hàng'
                    ]
                ], HTTP_INTERNAL_SERVER_ERROR);
            }

            // Thêm gói hàng vào shipment
            $this->onCreatePackages($packages, $shipment_created);

            // Thêm hóa đơn vào shipment
            $this->onCreateInvoices($invoices, $shipment_created);

            // Cập nhật giá trị số tiền vận chuyển
            $result_save_amount = onSaveAmountShipment($shipment_created['shipment_id']);
            // Nếu lưu số tiền vận chuyển thất bại thì hiện thông báo lỗi
            if ($result_save_amount['status'] === HTTP_INTERNAL_SERVER_ERROR) {
                return response()->json($result_save_amount);
            }

            if (!onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user['position_id'])) {
                // Cập nhật giá trị chi phí vận hành
                onUpdateOperatingCostsAmount($shipment_created['shipment_id']);
            }

            // Tính giá trị cuối cùng của đơn và lợi nhuận 
            onUpdateFinalAmountShipment($shipment_created['shipment_id']);

            // Khởi tạo order_pickup
            if (isset($request_data['order_pickup']) && json_decode($request_data['order_pickup'])) {
                // Xử lý tạo order pickup
                if (empty($shipment_created['serder_latitude']) || empty($shipment_created['sender_longitude'])) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Bạn cần phải xác định vị trí trên bản đồ để có thể tạo order pickup', null, 'warning');
                }
                $this->onCreateOrderPickup($shipment_created, $packages, $user, $request);
            }

            // Nếu dịch vụ là ePacket thì tính tiền ở ví
            if ($shipment_created['service']['promotion_flg'] == IS_EPACKET) {
                // Phân bổ tiền khi cập nhật shipment
                $result = allocateFundsWhenUpdatingShipment($shipment_created['shipment_id']);
                if ($result['status'] != HTTP_OK) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Bạn không đủ tiền trong ví để thanh toán, vui lòng nạp thêm tiền vào ví'
                        ]
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Cập nhật trạng thái thanh toán của shipment
                $shipment_created->shipment_payment_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment_created->accountant_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment_created->shipment_payment_method = WALLET;
                $shipment_created->save();
            } else if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user['position_id']) && $user['user_debit_type'] != DEBIT_TYPE_DAY) {
                $shipment_created->accountant_status = SUCCESS;
                $shipment_created->save();
            }

            onCreateHistoryShipment(
                $shipment_created['shipment_id'],
                $user['user_id'],
                'Khởi tạo shipment'
            );
            $hawbs = [];
            foreach ($shipment_created['packages'] as $package) {
                array_push(
                    $hawbs,
                    $package['package_code']
                );
            }

            DB::commit();

            return response()->json([
                'status' => HTTP_OK,
                'message' => 'Khởi tạo shipment thành công',
                'redirect_url' => route('shipments.detail.show', ['shipment_code' => $shipment_created['shipment_code']]),
                'data' => [
                    'ID bill' => $shipment_created['shipment_code'],
                    'Branch' => $shipment_created['branch']['branch_name'],
                    'Service' => $shipment_created['service']['service_name'],
                    'Country' => $shipment_created['country']['country_name'],
                    'Hawbs' => $hawbs
                ]
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    /**
     * Bắt điều kiện validate khi tạo shipment
     * @param mixed $request
     * @return void
     */
    private function onValidateUpdateShipment($request)
    {
        $request->validate([
            'shipment.sender_company_name' => 'required|min:1|max:255',
            'shipment.sender_contact_name' => 'required|min:1|max:255',
            'shipment.sender_telephone' => 'required',
            'shipment.sender_city' => 'required|numeric',
            'shipment.sender_district' => 'required|numeric',
            'shipment.sender_ward' => 'required|numeric',
            'shipment.sender_address' => 'required|min:1|max:255',

            'shipment.receiver_contact_name' => 'required|min:1|max:255',
            'shipment.receiver_telephone' => 'required',
            'shipment.receiver_country_id' => 'required|numeric|exists:countries,country_id',
            'shipment.receiver_city_id' => 'required|numeric|exists:cities,city_id',
            'shipment.receiver_postal_code' => 'required|min:1|max:255',
            'shipment.receiver_state_name' => 'required',
            'shipment.receiver_address_1' => 'required|min:1|max:225',
            'shipment.receiver_address_2' => 'max:225',
            'shipment.receiver_address_3' => 'max:225',

            'shipment.shipment_service_id' => 'required|numeric|exists:services,service_id',
            'shipment.shipment_branch_id' => 'required|numeric|exists:branchs,branch_id',
            'shipment.agree_terms_use_service' => 'required',

            'shipment.shipment_goods_name' => 'required|min:1|max:255',
            'shipment.shipment_value' => 'required|numeric',
            'shipment.shipment_export_as' => 'required|numeric|in:0,1',

            'shipment.packages' => 'required',
            'shipment.packages.*.package_quantity' => 'required|numeric|min:1|max:10',
            'shipment.packages.*.package_type' => 'required|numeric|int:0,1,2',
            'shipment.packages.*.package_length' => 'required|numeric|min:0',
            'shipment.packages.*.package_width' => 'required|numeric|min:0',
            'shipment.packages.*.package_height' => 'required|numeric|min:0',
            'shipment.packages.*.package_weight' => 'required|numeric|min:0',

            'shipment.invoices.*.invoice_goods_details' => 'required|min:1|max:500',
            'shipment.invoices.*.invoice_quantity' => 'required|numeric|min:0',
            'shipment.invoices.*.invoice_unit' => 'required|numeric|in:0,1,2,3,4',
            'shipment.invoices.*.invoice_price' => 'required|numeric|min:0',
            'shipment.invoices.*.invoice_total_price' => 'required|numeric|min:0',
        ], [
            'shipment.sender_company_name.required' => 'Tên công ty người gửi không được để trống',
            'shipment.sender_contact_name.required' => 'Tên người gửi không được để trống',
            'shipment.sender_telephone.required' => 'Số điện thoại người gửi không được để trống',
            'shipment.sender_city.required' => 'Thành phố người gửi không được để trống',
            'shipment.sender_district.required' => 'Quận người gửi không được để trống',
            'shipment.sender_ward.required' => 'Phường người gửi không được để trống',
            'shipment.sender_address.required' => 'Địa chỉ người gửi không được để trống',

            'shipment.receiver_contact_name.required' => 'Tên người nhận không được để trống',
            'shipment.receiver_telephone.required' => 'Số điện thoại người nhận không được để trống',
            'shipment.receiver_country_id.required' => 'Quốc gia người nhận không được để trống',
            'shipment.receiver_state_name.required' => 'Khu vực người nhận không được để trống',
            'shipment.receiver_city_id.required' => 'Thành phố người nhận không được để trống',
            'shipment.receiver_postal_code.required' => 'Mã bưu chính người nhận không được để trống',
            'shipment.receiver_address_1.required' => 'Địa chỉ người nhận không được để trống',

            'shipment.shipment_service_id.required' => 'Dịch vụ không được để trống',
            'shipment.shipment_branch_id.required' => 'Chi nhánh không được để trống',
            'shipment.agree_terms_use_service.required' => 'Bạn cần đồng ý với điều khoản sử dụng dịch vụ',

            'shipment.shipment_goods_name.required' => 'Tên hàng hóa không được để trống',
            'shipment.shipment_value.required' => 'Giá trị hàng hóa không được để trống',
            'shipment.shipment_export_as.required' => 'Hình thức xuất không được để trống',

            'shipment.packages.required' => 'Gói hàng không được để trống',
            'shipment.packages.*.package_quantity.required' => 'Số lượng gói hàng không được để trống',
            'shipment.packages.*.package_type.required' => 'Loại gói hàng không được để trống',
            'shipment.packages.*.package_length.required' => 'Chiều dài gói hàng không được để trống',
            'shipment.packages.*.package_width.required' => 'Chiều rộng gói hàng không được để trống',
            'shipment.packages.*.package_height.required' => 'Chiều cao gói hàng không được để trống',
            'shipment.packages.*.package_weight.required' => 'Trọng lượng gói hàng không được để trống',

            'shipment.invoices.*.invoice_goods_details.required' => 'Chi tiết hàng hóa không được để trống',
            'shipment.invoices.*.invoice_goods_details.min' => 'Độ dài hàng hóa không được nhỏ hơn 1',
            'shipment.invoices.*.invoice_goods_details.max' => 'Độ dài hàng hóa không được lớn hơn 225',

            'shipment.invoices.*.invoice_quantity.required' => 'Số lượng hàng hóa không được để trống',
            'shipment.invoices.*.invoice_quantity.numeric' => 'Số lượng  hàng hóa phải là 1 số',
            'shipment.invoices.*.invoice_quantity.min' => 'Số lượng hàng hóa không được nhỏ hơn 1',

            'shipment.invoices.*.invoice_unit.required' => 'Phân loại hàng hóa không được để trống',
            'shipment.invoices.*.invoice_unit.numeric' => 'Phân loại hàng hóa phải là 1 số',
            'shipment.invoices.*.invoice_unit.min' => 'Phân loại hóa không được nhỏ hơn 1',
            'shipment.invoices.*.invoice_unit.in' => 'Phân loại hóa phải thuộc [0, 1, 2, 3, 4]',

            'shipment.invoices.*.invoice_price.required' => 'Giá trị hàng hóa không được để trống',
            'shipment.invoices.*.invoice_price.numeric' => 'Giá trị hàng hóa phải là 1 số',
            'shipment.invoices.*.invoice_price.min' => 'Giá trị hàng hóa không được nhỏ hơn 1',

            'shipment.invoices.*.invoice_total_price.required' => 'Tổng giá trị hàng hóa không được để trống',
            'shipment.invoices.*.invoice_total_price.numeric' => 'Tổng giá trị hàng hóa phải là 1 số',
            'shipment.invoices.*.invoice_total_price.min' => 'Tổng giá trị hàng hóa không được nhỏ hơn 1',
        ]);
    }

    /**
     * Cập nhật hoặc tạo mới thông tin người nhận
     * @param mixed $shipment
     * @param mixed $user_sale
     * @return Receiver|\Illuminate\Database\Eloquent\Model
     */
    private function onUpdateOrCreateReceiver($shipment, $user_sale = [], $save_receiver_id = null, $user = [])
    {
        $receiver = Receiver::updateOrCreate(
            [
                'receiver_id' => $shipment['receiver_id'] ?? null,
            ],
            [
                'user_id' => $save_receiver_id ?? $user['user_id'],
                'receiver_contact_name' => $shipment['receiver_contact_name'],
                'receiver_company_name' => isset($shipment['receiver_company_name']) ? $shipment['receiver_company_name'] : '',
                'receiver_telephone' => $shipment['receiver_telephone'],
                'receiver_country_id' => $shipment['receiver_country_id'],
                'receiver_state_id' => $shipment['receiver_state_id'] ?? null,
                'receiver_city_id' => $shipment['receiver_city_id'],
                'receiver_postal_code' => $shipment['receiver_postal_code'],
                'receiver_address_1' => $shipment['receiver_address_1'],
                'receiver_address_2' => $shipment['receiver_address_2'],
                'receiver_address_3' => $shipment['receiver_address_3'],
            ]
        );

        return $receiver;
    }

    /**
     * Tạo mới shipment
     * @param mixed $shipment
     * @param mixed $user_sale
     * @param mixed $user
     * @param mixed $receiver
     * @param mixed $save_receiver_flg
     * @param mixed $save_receiver_id
     * @return Shipment|\Illuminate\Database\Eloquent\Model
     */
    private function onCreateShipment($shipment, $user_sale = [], $user = [], $receiver, $save_receiver_flg = false, $save_receiver_id = null)
    {
        $shipment_created = Shipment::create([
            'user_id' => $shipment['user_id'],

            'shipment_code' => $shipment['shipment_code'],
            'shipment_service_id' => $shipment['shipment_service_id'],
            'shipment_signature_flg' => json_decode($shipment['shipment_signature_flg']) ? SHIPMENT_SIGNATURE_YES : SHIPMENT_SIGNATURE_NO,
            'shipment_branch_id' => $shipment['shipment_branch_id'],
            'shipment_reference_code' => $shipment['shipment_reference_code'],
            'shipment_status' => SHIPMENT_STATUS_CREATE_BILL,
            'shipment_goods_name' => $shipment['shipment_goods_name'],
            'shipment_value' => $shipment['shipment_value'],
            'shipment_export_as' => $shipment['shipment_export_as'],
            'shipment_iosscode' => $shipment['shipment_iosscode'] ?? null,
            'shipment_payment_status' => 0,
            'shipment_paid_by' => $shipment['shipment_paid_by'] ?? 0,

            'sender_company_name' => $shipment['sender_company_name'],
            'sender_contact_name' => $shipment['sender_contact_name'],
            'sender_telephone' => $shipment['sender_telephone'],
            'sender_city' => $shipment['sender_city'],
            'sender_district' => $shipment['sender_district'],
            'sender_ward' => $shipment['sender_ward'],
            'sender_address' => $shipment['sender_address'],
            'sender_longitude' => isset($shipment['sender_longitude']) ? $shipment['sender_longitude'] : null,
            'serder_latitude' => isset($shipment['serder_latitude']) ? $shipment['serder_latitude'] : null,

            'receiver_id' => $save_receiver_flg ? $receiver['receiver_id'] : null,
            'receiver_company_name' => isset($shipment['receiver_company_name']) ? $shipment['receiver_company_name'] : '',
            'receiver_contact_name' => $shipment['receiver_contact_name'],
            'receiver_telephone' => $shipment['receiver_telephone'],
            'receiver_country_id' => $shipment['receiver_country_id'],
            'receiver_state_id' => $shipment['receiver_state_id'] ?? null,
            'receiver_state_name' => $shipment['receiver_state_name'] ?? null,
            'receiver_city_id' => $shipment['receiver_city_id'],
            'receiver_postal_code' => $shipment['receiver_postal_code'],
            'receiver_address_1' => $shipment['receiver_address_1'],
            'receiver_address_2' => $shipment['receiver_address_2'],
            'receiver_address_3' => $shipment['receiver_address_3'],

            'document_id' => $shipment['document_id'] ?? null,

            "shipment_amount_total_customer" => $shipment['shipment_amount_total_customer'] ?? 0,
            "shipment_domestic_charges" => $shipment['shipment_domestic_charges'] ?? 0,
            "shipment_amount_surcharge" => $shipment['shipment_amount_surcharge'] ?? 0,
            "shipment_collection_fee" => $shipment['shipment_collection_fee'] ?? 0,
            "shipment_amount_insurance" => $shipment['shipment_amount_insurance'] ?? 0,
            "shipment_amount_vat" => $shipment['shipment_amount_vat'] ?? 0,
            "shipment_amount_original" => $shipment['shipment_amount_original'] ?? 0,

            "receiver_sms_phone" => $shipment['sender_telephone'],
            "completed_date" => Carbon::now()->setTimezone('Asia/Ho_Chi_Minh'),

        ]);

        return $shipment_created;
    }

    /**
     * Tạo mới gói hàng
     * @param mixed $packages
     * @param mixed $shipment
     * @return void
     */
    private function onCreatePackages($packages, $shipment)
    {
        $service = $shipment['service'];
        $branch = $shipment['branch'];
        if ($branch['branch_name'] == 'HCM') {
            $branch_name = 'HO CHI MINH, VN';
        } else if ($branch['branch_name'] == 'HN') {
            $branch_name = 'HA NOI, VN';
        } else if ($branch['branch_name'] == 'DN') {
            $branch_name = 'DA NANG, VN';
        } else {
            $branch_name = 'HO CHI MINH, VN';
        }
        $total_weight = getTotalPackage($packages, $service['service_volumetric_mass']);
        foreach ($packages as $package) {
            $package['shipment_id'] = $shipment['shipment_id'];
            // $package['package_weight'] = formatNumberCeil($package['package_weight'], $total_weight, $service['promotion_flg']);
            $package['package_weight'] = $package['package_weight'];
            $package['package_converted_weight'] = $package['package_length'] * $package['package_width'] * $package['package_height'] / $service['service_volumetric_mass'];
            $package['package_charged_weight'] = $package['package_weight'] < $package['package_converted_weight'] ? $package['package_converted_weight'] : $package['package_weight'];
            $package['package_charged_weight'] = formatNumberCeil($package['package_charged_weight'], $total_weight, $service['promotion_flg']);
            $package['package_description'] = '';
            $package_quantity = $package['package_quantity'];

            for ($i = 1; $i <= $package_quantity; $i++) {
                $package_code = getCodePackage();
                $package['package_code'] = $package_code['package_code'];
                $package['package_hawb_code'] = $package_code['package_code'];
                $package['package_quantity'] = 1;

                // Tính giá trị của gói hàng
                $package['package_price'] = computedPricePackage($package['package_charged_weight'], $shipment);

                Package::create($package);

                onCreatePackageTrackingStatus($package['package_code'], $branch_name, 'LABEL CREATED');
            }
        }
    }

    /**
     * Tạo mới hóa đơn
     * @param mixed $invoices
     * @param mixed $shipment
     * @return void
     */
    private function onCreateInvoices($invoices, $shipment)
    {
        foreach ($invoices as $invoice) {
            $invoice['invoice_code'] = initializationInvoiceCode();
            $invoice['shipment_id'] = $shipment['shipment_id'];
            Invoice::create($invoice);
        }
    }

    /**
     * Xử lý tạo order pickup
     * @param mixed $shipment
     * @param mixed $packages
     * @param mixed $user
     * @param mixed $request
     * @return void
     */
    private function onCreateOrderPickup($shipment, $packages, $user, $request)
    {
        $request_pickup = [
            "user_id" => $shipment['user_id'],
            "order_pickup_name" => $shipment['sender_contact_name'],
            "sale_id" => isset($shipment['sale_id']) ? $shipment['sale_id'] : null,
            "shipment_id" => $shipment['shipment_id'],
            "branch_id" => $shipment['shipment_branch_id'],
            "order_pickup_awb" => $shipment['shipment_code'],
            "order_pickup_address" => $shipment['sender_address'],
            'longitude' => isset($shipment['sender_longitude']) ? $shipment['sender_longitude'] : null,
            'latitude' => isset($shipment['serder_latitude']) ? $shipment['serder_latitude'] : null,
            "order_pickup_phone" => $shipment['sender_telephone'],
            "order_pickup_note" => 'Đơn tạo tự động thông qua shipment',
            "order_pickup_type" => 0,
            "order_pickup_status" => ORDER_PICKUP_STATUS_WAITING,
            "order_pickup_date_time" => Carbon::now()->setTimezone('Asia/Ho_Chi_Minh')->format('d-m-Y H:i:s'),
            "order_pickup_number_packages" => count($packages),
            "order_pickup_gross_weight" => collect($packages)->sum('package_weight'),
        ];

        $order_pickup = new OrderPickupController();
        $res_pickup = $order_pickup->onUpdateOrCreateOrderPickup($request_pickup, $user, $request);
        if ($res_pickup['status'] == HTTP_OK) {
            Shipment::find($shipment['shipment_id'])->update([
                'order_pickup_id' => $res_pickup['data']['order_pickup_id'],
            ]);
        } else {
            Session::flash('error', 'Khởi tạo Order Pickup thất bại');
        }
    }

    public function displayShipmentDetail($shipment_code)
    {
        try {
            $user = auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);
            $view_operating_costs_position_ids = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_VIEW_AND_UPDATE_SHIPMENT_OPERATING_COST)['position_ids'];
            $is_view_shipment_operating_costs = !onCheckPositionAccounts(SETTING_FWD_ACCOUNT, $user->position_id);
            $service_kind_label = config('constans.constans.service_kind');
            $shipment = Shipment::with([
                'user',
                'service',
                'branch',
                'country',
                'city',
                'invoices',
                'shipmentOperatingCosts' => function ($query) {
                    return $query->leftJoin('operating_costs', 'shipment_operating_costs.operating_cost_id', '=', 'operating_costs.operating_cost_id')
                        ->select('shipment_operating_costs.*', 'operating_costs.operating_cost_name');
                }
            ])->where('shipment_code', $shipment_code)->isDelete(NO_DELETE);
            // if (!$is_view_all_shipment) {
            //     $shipment = $shipment->where('user_id', $user['user_id']);
            // }
            $shipment = $shipment->first();
            if (!$shipment) {
                return viewError404(true);
            }
            if ($shipment['shipment_status'] != SHIPMENT_STATUS_CREATE_BILL) {
                onUpdateCompletedDate($shipment['shipment_id']);
            }
            updateStatusShipment($shipment['shipment_id']);
            $label = ModelsLabelNz::isShow()->where('shipment_id', $shipment->shipment_id)->first();
            return view('shipment.detail', compact('shipment', 'is_view_shipment_operating_costs', 'service_kind_label', 'label'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function getDetailPackage(Request $request)
    {
        try {
            $request_data = $request->all();
            $package = Package::where('package_hawb_code', $request_data['code'])->isShow()->first();
            if ($package) {
                $packageSg = PackageSg::where('package_id', $package->package_id)->isShow()->get();
                return response()->json([
                    "status" => HTTP_OK,
                    "package" => $package,
                    "packageSg" => $packageSg,
                ]);
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy gói hàng');
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function showListPackages(Request $request)
    {
        $this->validate($request, ['shipment_code' => 'required']);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            $shipment = Shipment::with(['service'])
                ->where('shipment_code', $request_data['shipment_code'])->isDelete(NO_DELETE)->first();

            if (!$shipment) {
                return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy dữ liệu shipment');
            }

            $packages = Package::isShow()->where('shipment_id', $shipment->shipment_id)
                ->select('packages.*', DB::raw(
                    "CASE
                        WHEN package_status = 0 THEN 'Create Bill'
                        WHEN package_status = 1 THEN 'Import Bill'
                        WHEN package_status = 2 THEN 'Export Bill'
                        WHEN package_status = 3 THEN 'Return Bill'
                        ELSE 'Other'
                    END as status_label"
                ))->get();

            $promotion_flg = $shipment['service']['promotion_flg'];
            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'packages' => $packages,
                    'promotion_flg' => $promotion_flg,
                ]);
            }
            $html = view('shipment.ajax.table-packages', compact('packages', 'shipment', 'promotion_flg'))->render();
            return response()->json([
                'status' => HTTP_OK,
                'html' => $html
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function updatePackageSurchageGoods(Request $request)
    {
        $this->validate($request, ['package_hawb_code' => 'required']);
        try {
            $user = $request->user();
            $request_data = $request->all();
            $package = Package::where('package_hawb_code', $request_data['package_hawb_code'])
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->select(
                    'package_hawb_code',
                    'package_status',
                    'shipments.user_id',
                    'shipments.shipment_id',
                    'services.promotion_flg',
                    'package_id'
                )->first();

            if ($package['promotion_flg'] == IS_EPACKET && $package['package_status'] != SHIPMENT_STATUS_CREATE) {
                return returnResponse(HTTP_BAD_REQUEST, 'Dịch vụ epacket không thể cập nhật khi đã thay đổi', null, 'warning');
            } else if ($package['package_status'] > SHIPMENT_STATUS_IMPORTED && !isGetAllData()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn không đủ quyền hạn để cập nhật', null, 'warning');
            }

            $new_packageSg = $request_data['package_sg'];

            DB::beginTransaction();
            // xóa hết các package sg cũ và tạo lại mới
            PackageSg::where('package_id', $package->package_id)->isShow()->delete();
            foreach ($new_packageSg as $key => $value) {
                PackageSg::create([
                    "package_id" => $package->package_id,
                    "surcharge_goods_id" => $value['surcharge_goods_id'],
                    "price" => $value['price'],
                    "count" => $value['count'],
                ]);
            }
            onUpdateFinalAmountShipment($package['shipment_id']);
            onCreateHistoryShipment(
                $package['shipment_id'],
                $user['user_id'],
                'Cập nhật lại phụ thu package' . $package['package_hawb_code']
            );
            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'icon' => 'success',
                    'title' => 'Thông báo',
                    'text' => 'Cập nhật chi phí phụ thu thành công'
                ]
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function getShipmentDetailOperatingCosts(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $shipment_code = $request_data['shipment_code'];
            $user = $is_api ? $request->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);
            $view_operating_costs_position_ids = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_VIEW_AND_UPDATE_SHIPMENT_OPERATING_COST)['position_ids'];
            $is_view_shipment_operating_costs = !onCheckPositionAccounts([SETTING_FWD_ACCOUNT], $user->position_id);

            if (!$is_view_shipment_operating_costs) {
                return showMessageError();
            }

            $shipment = Shipment::isDelete(NO_DELETE)->where('shipments.import_approval', ACTIVE)->with(['packages'])->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();
            $shipment_id = $shipment['shipment_id'];

            $operating_costs = OperatingCost::with([
                'childOperatingCost' => function ($query) use ($shipment_id) {
                    return $query->isActive(ACTIVE)->isDelete(NO_DELETE)
                        ->leftJoin('shipment_operating_costs', function ($join) use ($shipment_id) {
                            return $join->on('shipment_operating_costs.operating_cost_id', '=', 'operating_costs.operating_cost_id')
                                ->where('shipment_operating_costs.shipment_id', $shipment_id);
                        })
                        ->select(
                            'operating_costs.*',
                            'shipment_operating_costs.shipment_operating_cost_id',
                            'shipment_operating_costs.shipment_operating_cost_amount',
                            'shipment_operating_costs.shipment_operating_cost_quantity',
                        );
                }
            ])->isActive(ACTIVE)->isDelete(NO_DELETE)
                ->whereNull('operating_costs.parent_operating_cost_id')->get();

            if (empty($operating_costs)) {
                return showMessageError();
            }

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'operating_costs' => $operating_costs,
                    'shipment' => $shipment,
                ]);
            }

            $html = view('shipment.shipment-operating-cost', compact('operating_costs', 'shipment'))->render();
            return response()->json([
                'status' => HTTP_OK,
                'html' => $html
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleShipmentDetailOperatingCosts(Request $request)
    {
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $shipment_code = $request_data['shipment_code'];
            $operating_costs = collect($request_data['operating_costs'])->except('receiver-sms-name', 'receiver-sms-phone')->toArray();
            $user = $is_api ? $request->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);
            $view_operating_costs_position_ids = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_VIEW_AND_UPDATE_SHIPMENT_OPERATING_COST)['position_ids'];
            $is_view_shipment_operating_costs = !onCheckPositionAccounts(SETTING_FWD_ACCOUNT, $user->position_id);

            if (!$is_view_shipment_operating_costs) {
                return showMessageError();
            }

            $shipment = Shipment::isDelete(NO_DELETE)->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();
            $old_shipment = $shipment;
            $shipment_id = $shipment['shipment_id'];

            $shipment->receiver_sms_name = $request_data['operating_costs']['receiver-sms-name'];
            $shipment->receiver_sms_phone = $request_data['operating_costs']['receiver-sms-phone'];
            $shipment->shipment_amount_operating_costs = 0;
            ShipmentOperatingCost::where('shipment_id', $shipment_id)->delete();
            $operating_costs = collect($operating_costs)->flatten(2)->toArray();

            foreach ($operating_costs as $operating_cost) {
                $shipment_operating_cost_quantity = $operating_cost['shipment-operating-cost-quantity'];
                $shipment_operating_cost_amount = $operating_cost['operating-cost-amount'] * $shipment_operating_cost_quantity;
                $shipment->shipment_amount_operating_costs += $shipment_operating_cost_amount;

                ShipmentOperatingCost::create([
                    'shipment_id' => $shipment_id,
                    'operating_cost_id' => $operating_cost['operating-cost-id'],
                    'shipment_operating_cost_amount' => $operating_cost['operating-cost-amount'],
                    'shipment_operating_cost_total_amount' => $shipment_operating_cost_amount,
                    'shipment_operating_cost_quantity' => $shipment_operating_cost_quantity,
                ]);
            }

            // $result_update_wallet = onUpdateWalletUser($shipment['shipment_id']);
            $shipment->save();

            // tính giá trị cuối cùng của đơn và lợi nhuận 
            onUpdateFinalAmountShipment($shipment_id);

            $shipment_operating_costs = ShipmentOperatingCost::where('shipment_id', $shipment_id)
                ->leftJoin('operating_costs', 'shipment_operating_costs.operating_cost_id', '=', 'operating_costs.operating_cost_id')
                ->select('shipment_operating_costs.*', 'operating_costs.operating_cost_name')->get();

            // Nếu dịch vụ là ePacket thì tính tiền ở ví
            if ($shipment['service']['promotion_flg'] == IS_EPACKET) {
                $check_packages = collect($shipment['packages'])->where('package_approve', '!=', PACKAGE_APPROVE_WAITING)->count();
                if ($check_packages > 0) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Gói hàng đã được duyệt không thể cập nhật chi phí vận hành'
                        ]
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Phân bổ tiền khi cập nhật shipment
                $result = allocateFundsWhenUpdatingShipment($shipment['shipment_id'], $old_shipment['shipment_final_amount'], true);
                if ($result['status'] != HTTP_OK) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Bạn không đủ tiền trong ví để thanh toán, vui lòng nạp thêm tiền vào ví'
                        ]
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Cập nhật trạng thái thanh toán của shipment
                $shipment->shipment_payment_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment->accountant_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;

                onCreateHistoryShipment(
                    $shipment['shipment_id'],
                    $user['user_id'],
                    'Cập nhật lại chi phí vận hành'
                );

                $shipment->save();
            }

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Cập nhật chi phí vận hành thành công.',
                    'icon' => 'success',
                ],
                'shipment' => Shipment::find($shipment_id),
                'shipment_operating_costs' => $shipment_operating_costs,
                'shipment_amount_profit' => $shipment->shipment_amount_profit,
                'amount_wallet' => Wallet::where('user_id', $shipment['user_id'])->first()['amount']
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function displayShipmentsList()
    {
        $branchs = Branch::isShow()->get();
        $shipment_status = config('constans.constans.shipment_status');
        $shipment_filter_by = config('constans.constans.shipment_filter_by');
        $services = Service::isShow()->select('service_id', 'service_name')->get();
        return view('shipment.list', compact('branchs', 'services', 'shipment_status', 'shipment_filter_by'));
    }

    public function getShipments(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? $request->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);
            $filters = isset($request_data['filters']) ? (is_array($request_data['filters']) ? $request_data['filters'] : json_decode($request_data['filters'], true)) : [];

            $shipments = Shipment::isDelete(NO_DELETE)->where('shipments.import_approval', ACTIVE)
                ->orderBy('created_at', 'desc')->isFilters($filters)->with(['service']);
            if (!$is_view_all_shipment) {
                if (onCheckPositionAccount(SETTING_SALE_ACCOUNT, $user->position_id)) {
                    $fwd_ids = SaleLinkFwd::isShow()->where('sale_id', $user->user_id)
                        ->select('fwd_id')->pluck('fwd_id');
                    $data_query = [
                        'fwd_ids' => [],
                        // 'fwd_ids' => $fwd_ids ?? [],
                        'user_id' => $user->user_id
                    ];
                    $shipments = $shipments->where(function ($query) use ($data_query) {
                        return $query->where('user_id', $data_query['user_id'])
                            ->orWhereIn('user_id', $data_query['fwd_ids']);
                    });
                } else {
                    $shipments = $shipments->where('shipments.user_id', $user['user_id']);
                }
                $shipments = $shipments->where('shipments.active_flg', ACTIVE);
            } else {
                $active_flg = $filters['active_flg'] ?? null;
                if (isset($active_flg) && in_array($active_flg, [0, 1])) {
                    $shipments = $shipments->where('shipments.active_flg', $active_flg);
                }
            }
            if (isset($request_data['debit_id'])) {
                $shipments = $shipments->where('shipments.shipment_debit_id', $request_data['debit_id']);
            }
            // Xuất file excel
            if (isset($request_data['export_excel']) && json_decode($request_data['export_excel'])) {
                $data = $shipments->get();
                $name_excel = 'export-excel-shipment.xlsx';
                $others = null;
                $viewName = 'shipment.excels.shipments2';
                $column_widths = null;
                return Excel::download(new ExportFile($data, $viewName, $others, $column_widths), $name_excel);
            }
            $shipments = $shipments->groupBy('shipments.shipment_id');

            if (Route::currentRouteName() == 'accountant.shipment-paginate') {
                $shipment = $shipments->leftJoin('services', 'shipment.shipment_service_id', 'services.service_id')
                    ->where('services.promotion_flg', IS_EPACKET)->select('shipments.*');
            }

            $shipments = $shipments->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            $have_show_label_LF2 = onShowLabel2ByAccount(auth()->user());

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'shipments' => $shipments,
                ]);
            } else {
                if (isset($request_data['html']) && $request_data['html'] == "accountant") {
                    $html = view('accounting-management.shipment.ajax.table-shipment', compact('shipments', 'have_show_label_LF2'))->render();
                } else {
                    $html = view(
                        'shipment.shipment-pagination',
                        compact('shipments', 'have_show_label_LF2')
                    )->render();
                }
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function onUploadLabel(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;

            if (empty($request_data['numbers'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn cần nhập mã shipment!', null, 'warning');
            }

            if (!is_array($request_data['numbers'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Lỗi dữ liệu truyền lên');
            }

            DB::beginTransaction();
            Shipment::isShow()->whereIn('shipment_code', $request_data['numbers'])
                ->update(['created_lable_at' => Carbon::now()->format('Y-m-d H:i:s')]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function removeCheckedLabel(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;

            if (empty($request_data['numbers'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn cần nhập mã shipment!', null, 'warning');
            }

            if (!is_array($request_data['numbers'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Lỗi dữ liệu truyền lên');
            }

            DB::beginTransaction();
            Shipment::isShow()->whereIn('shipment_code', $request_data['numbers'])
                ->update(['created_lable_at' => Carbon::now()->format('Y-m-d H:i:s')]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function displayShipmentsPrintBill($shipment_code)
    {
        try {
            $method = request()->method();
            $user = $method == 'POST' ? request()->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);
            $system_settings_kinds = [SETTING_COMPANY_NAME, SETTING_HEADQUARTERS_ADDRESS, SETTING_EMAIL_ADDRESS, SETTING_WEBSITE_ADDRESS, SETTING_HOTLINE];
            $system_settings = json_decode(Storage::get($path), true);
            $system_settings = collect($system_settings)->whereIn('kind', $system_settings_kinds)->keyBy('kind')->toArray();

            $shipment = Shipment::with(['branch', 'service.logoService', 'user', 'packages', 'invoices'])->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();
            if (!isset($shipment)) {
                return abort(HTTP_NOT_FOUND);
            }

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            $barcode = $generator->getBarcode($shipment_code, $generator::TYPE_CODE_128);

            $to_receiver_packages = collect($shipment['packages'])->groupBy(function ($item) {
                return $item['package_type'] . '-' . $item['package_length'] . '-' . $item['package_width'] . '-' . $item['package_height'] . '-' . $item['package_weight'];
            })->toArray();

            $scaner = HistoryScan::isShow()
                ->leftJoin('packages', 'history_scan.package_id', 'packages.package_id')
                ->leftJoin('users', 'history_scan.scan_by', 'users.user_id')
                ->where('packages.shipment_id', $shipment['shipment_id'])
                ->where('history_scan.status', SHIPMENT_STATUS_IMPORTED)
                ->select(
                    'user_id',
                    'user_contact_name',
                    'user_signature',
                    'history_scan.created_at',
                )->first();

            return view('shipment.prints.bill', compact('shipment', 'system_settings', 'barcode', 'to_receiver_packages', 'scaner'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function displayShipmentsPrintLabel($shipment_code)
    {
        try {
            $method = request()->method();
            $user = $method == 'POST' ? request()->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);


            $shipment = Shipment::with(['service', 'packages', 'user'])->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();

            if (!isset($shipment)) {
                return abort(HTTP_NOT_FOUND);
            }

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            $barcode = $generator->getBarcode($shipment_code, $generator::TYPE_CODE_128, 2, 70);

            $content_qrcode = env('TRACKING_POST_URL') . "?shipment_code=$shipment_code";
            $qrcode = generatorQRCode($content_qrcode, 'simple');

            return view('shipment.prints.label', compact('shipment', 'barcode', 'qrcode'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function displayShipmentsPrintLabelV2($shipment_code)
    {
        try {
            $method = request()->method();
            $user = $method == 'POST' ? request()->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);

            $shipment = Shipment::with(['service', 'packages'])->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();

            if (!isset($shipment)) {
                return abort(HTTP_NOT_FOUND);
            }

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            $barcode = $generator->getBarcode($shipment_code, $generator::TYPE_CODE_128, 3, 60);

            return view('shipment.prints.label-v2', compact('shipment', 'barcode'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function displayShipmentsPrintLabelV2F($shipment_code)
    {
        try {
            // if (!isGetAllData() || !isUserIdsShipment() || !onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || !onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT)) {
            //     return abort(HTTP_NOT_FOUND);
            // }
            $method = request()->method();
            $user = $method == 'POST' ? request()->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT) || onShowLabel2ByAccount($user);

            $shipment = Shipment::with(['service', 'packages'])->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();

            if (!isset($shipment)) {
                return abort(HTTP_NOT_FOUND);
            }
            $countries = Country::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->select('country_id', 'country_name')->get()->toArray();

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            $barcode = $generator->getBarcode($shipment_code, $generator::TYPE_CODE_128, 2, 70);

            return view('shipment.prints.label-v2-f', compact('shipment', 'countries', 'barcode'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function displayShipmentsPrintLabelV2Changed(Request $request)
    {
        try {
            // if (!isGetAllData() || !isUserIdsShipment() || !onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || !onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT)) {
            //     return showMessageError();
            // }
            $request_data = $request->all();
            $shipment_code = $request_data['shipment_code'];
            $method = request()->method();
            $user = $method == 'POST' ? request()->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);

            $shipment = Shipment::with(['service', 'packages'])->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();

            if (!isset($shipment)) {
                return abort(HTTP_NOT_FOUND);
            }

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            $barcode = $generator->getBarcode($shipment_code, $generator::TYPE_CODE_128, 3, 60);

            return response()->json([
                'status' => HTTP_OK,
                'html' => view('shipment.prints.site.label-v2-changed', compact('shipment', 'barcode', 'request_data'))->render()
            ], HTTP_OK);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function displayShipmentsPrintInvoice($shipment_code)
    {
        try {
            $method = request()->method();
            $user = $method == 'POST' ? request()->user() : auth()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);

            $shipment = Shipment::with(['user', 'service', 'packages', 'invoices', 'state', 'country'])
                ->where('shipments.import_approval', ACTIVE)
                ->where('shipment_code', $shipment_code);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();

            if (!isset($shipment)) {
                return abort(HTTP_NOT_FOUND);
            }

            $to_receiver_packages = collect($shipment['packages'])->groupBy(function ($item) {
                return $item['package_type'] . '-' . $item['package_length'] . '-' . $item['package_width'] . '-' . $item['package_height'] . '-' . $item['package_weight'];
            })->toArray();

            return view('shipment.prints.invoice', compact('shipment', 'to_receiver_packages'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function handleShipmentDelete(Request $request)
    {
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? $request->user() : auth()->user();
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT);

            $shipment = Shipment::where('shipment_code', $request_data['shipment_code']);


            if (!$is_view_all_shipment) {
                return returnResponse(HTTP_BAD_REQUEST, 'Không thể xóa đơn hàng, bạn không đủ quyền hạn!', null, 'warning');
                $shipment = $shipment->where('user_id', $user['user_id']);
            }

            $shipment = $shipment->where('shipments.import_approval', ACTIVE)->first();

            if (!isset($shipment)) {
                return showReturnNotFound();
            }

            // Hoàn tiền đơn hàng ePacket
            if ($shipment['service']['promotion_flg'] == IS_EPACKET && $shipment['shipment_payment_status'] == SUCCESS) {
                $check_packages = collect($shipment['packages'])->where('package_approve', '!=', PACKAGE_APPROVE_WAITING)->count();
                if ($check_packages > 0) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Không thể xóa khi đơn hàng ePacket đã duyệt gói hàng'
                        ]
                    ]);
                }
                $wallet = Wallet::where('user_id', $shipment['user_id'])->first();
                $wallet->amount += $shipment['shipment_final_amount'];
                $wallet->save();

                WalletFluctuation::create([
                    'wallet_id' => $wallet['wallet_id'],
                    'secure_hash' => 'WF' . Carbon::now()->format('YmdHisv'),
                    'user_id' => $shipment['user_id'],
                    'amount' => $shipment['shipment_final_amount'],
                    'wallet_amount' => $wallet['amount'],
                    'content' => 'Hoàn tiền đơn hàng ePacket khi xóa mang mã: ' . $shipment['shipment_code'],
                    'kind' => PLUS,
                ]);
            }

            // if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, auth()->user()['position_id'])) {
            //     $shipment->delete_flg = DELETED;
            // }else{
            //     $shipment->active_flg = DELETED;
            // }
            if (onCheckAdminAccount()) {
                $shipment->delete_flg = DELETED;
            } else {
                $shipment->active_flg = INACTIVE;
            }
            $shipment->save();

            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $user['user_id'],
                'Xóa shipment'
            );

            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa shipment thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onChangePaymentMethod(Request $request)
    {
        $this->validate($request, [
            'shipment_code' => 'required',
        ]);

        try {
            $request_data = $request->all();

            if (!isGetAllData()) {
                return returnResponse(HTTP_FORBIDDEN, 'Không có quyền thay đổi');
            }

            $shipment = Shipment::where('shipment_code', $request_data['shipment_code'])->first();
            if ($shipment['shipment_payment_status'] == SUCCESS) {
                return returnResponse(HTTP_BAD_REQUEST, 'Không thể thay đổi khi đơn hàng đã thanh toán');
            }
            $shipment_id = $shipment['shipment_id'];

            DB::beginTransaction();
            $shipment->shipment_paid_by = $shipment->shipment_paid_by == SHIPMENT_PAID_BY_REMAINING ? SHIPMENT_PAID_BY_WALLET : SHIPMENT_PAID_BY_REMAINING;
            $shipment->shipment_payment_status = 0;
            $shipment->shipment_payment_step = 0;
            foreach (onConvertStrArrImage($shipment['shipment_file_proof_of_payment'], false) as $item) {
                if (!empty($item) && File::exists($item)) {
                    File::delete($item);
                }
            };
            $shipment->shipment_file_proof_of_payment = null;

            $shipment->save();
            onCreateHistoryShipment(
                $shipment_id,
                $request->user()['user_id'],
                'Cập nhật lại phương thức thanh toán'
            );
            DB::commit();
            Session::flash('success', 'Thay đổi phương thức thanh toán thành công');
            return returnResponse(HTTP_OK, 'Thay đổi phương thức thanh toán thành công', null, 'success');
        } catch (\Throwable $th) {
            return returnResponse();
        }
    }

    public function displayShipmentEdit($shipment_code)
    {
        try {
            $user = auth()->user();
            $receivers = Receiver::isUser(auth()->id())->isActive(ACTIVE)->isDelete(NO_DELETE)->select('receiver_id', 'receiver_contact_name')->get()->toArray();
            $countries = Country::isActive(ACTIVE)->isDelete(NO_DELETE)->select('country_id', 'country_name')->get()->toArray();
            $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->select('branch_id', 'branch_name')->get()->toArray();
            $package_types = config('constans.constans.package_types');
            $invoice_exports_as = config('constans.constans.invoice_exports_as');
            $invoice_units = config('constans.constans.invoice_units');


            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);

            $shipment = Shipment::with([
                'packages' => function ($query) {
                    return $query->select('packages.package_id', 'packages.shipment_id', 'packages.package_quantity', 'packages.package_type', 'packages.package_length', 'packages.package_width', 'packages.package_height', 'packages.package_weight');
                },
                'invoices' => function ($query) {
                    return $query->select('invoices.invoice_id', 'invoices.shipment_id', 'invoices.invoice_goods_details', 'invoices.invoice_quantity', 'invoices.invoice_unit', 'invoices.invoice_price', 'invoices.invoice_total_price');
                }
            ])->where('shipment_code', $shipment_code)->isDelete(NO_DELETE);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->select(
                'shipments.shipment_id',
                'shipments.shipment_code',
                'shipments.shipment_service_id',
                'shipments.shipment_signature_flg',
                'shipments.shipment_branch_id',
                'shipments.shipment_reference_code',
                'shipments.shipment_goods_name',
                'shipments.shipment_value',
                'shipments.shipment_export_as',
                'shipments.user_id',
                'shipments.receiver_id',
                'shipments.sender_company_name',
                'shipments.sender_contact_name',
                'shipments.sender_telephone',
                'shipments.sender_city',
                'shipments.sender_district',
                'shipments.sender_ward',
                'shipments.sender_address',
                'shipments.sender_longitude',
                'shipments.serder_latitude',
                'shipments.receiver_company_name',
                'shipments.receiver_contact_name',
                'shipments.receiver_telephone',
                'shipments.receiver_country_id',
                'shipments.receiver_state_id',
                'shipments.receiver_state_name',
                'shipments.receiver_city_id',
                'shipments.receiver_postal_code',
                'shipments.receiver_address_1',
                'shipments.receiver_address_2',
                'shipments.receiver_address_3',
                'shipments.order_pickup_id',
                'shipments.document_id',
            )->first();
            if (!$shipment) {
                return abort(HTTP_NOT_FOUND);
            }

            // Cập nhật ngày chốt tiền đơn hàng theo scan import
            if ($shipment['shipment_status'] != SHIPMENT_STATUS_CREATE_BILL) {
                onUpdateCompletedDate($shipment['shipment_id']);
            }

            $service = Service::isShow()->where('service_id', $shipment->shipment_service_id)->first();

            $address = address($shipment['sender_city'], $shipment['sender_district']);

            $sales = null;
            if (onCheckPositionAccount(SETTING_DOCUMENT_ACCOUNT, $user['position_id'])) {
                $sales = User::isShow()->where('position_id', getPositionAccountSetting(SETTING_SALE_ACCOUNT));
                $sales = $sales->select('users.user_id', 'users.user_code', 'users.user_contact_name')->get();
            };

            $city_receiver = City::find($shipment['receiver_city_id']);
            return view(
                'shipment.edit',
                compact(
                    'shipment',
                    'sales',
                    'service',
                    'address',
                    'receivers',
                    'countries',
                    'city_receiver',
                    'branchs',
                    'package_types',
                    'invoice_exports_as',
                    'invoice_units'
                )
            );
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    /*************  ✨ Codeium Command ⭐  *************/
    /**
     * Xử lý cập nhật lại shipment
     * Gọi hàm validate dữ liệu khi tạo shipment
     * Lấy dữ liệu từ request
     * Lấy dữ liệu user
     * Lấy dữ liệu shipment
     * Lấy dữ liệu service
     * Lấy dữ liệu shipment cũ
     * Lấy dữ liệu user cũ
     * Lấy dữ liệu package
     * Kiểm tra xem đơn hàng đã quét chưa nếu đã quét thì không thể chỉnh sửa của đơn hàng ePacket
     * Kiểm tra xem dịch vụ ePacket chỉ cho phép tạo 1 gói hàng
     * Kiểm tra xem Reference Code đã tồn tại hay chưa
     * Kiểm tra xem tài khoản sale có tồn tại hay không
     * Tạo receiver mới
     * Kiểm tra xem tài khoản sale có tồn tại hay không
     * Tạo receiver mới
     * Cập nhật lại dữ liệu shipment
     * Xóa các gói hàng cũ
     * Tạo các gói hàng mới
     * Tạo các hóa đơn mới
     * Xóa các hóa đơn cũ
     * Cập nhật lại dữ liệu hóa đơn
     * Tính lại giá trị cuối cùng của đơn và lợi nhuận
     * Tính lại giá trị cuối cùng của đơn và lợi nhuận
     * Cập nhật lại giá trị cuối cùng của đơn và lợi nhuận
     * Khởi tạo order_pickup
     * Tính tiền ở ví
     * Cập nhật trạng thái thanh toán của shipment
     * Cập nhật lại trạng thái thanh toán của shipment
     * Cập nhật lại trạng thái thanh toán của shipment

     * Tạo lịch sử của shipment
     * Trả về kết quả
     */
    public function handleShipmentEdit(Request $request)
    {
        // Gọi hàm validate dữ liệu khi tạo shipment
        $this->onValidateUpdateShipment($request);
        DB::beginTransaction();
        try {
            // Lấy dữ liệu từ request
            $request_data = $request->all();
            // Lấy dữ liệu user
            $user = $request->user();

            // Lấy dữ liệu shipment
            $shipment = $request_data['shipment'];
            // Lấy dữ liệu service
            $service = Service::isActive(ACTIVE)->isDelete(NO_DELETE)->findOrFail($shipment['shipment_service_id'])->toArray();
            // Lấy dữ liệu shipment cũ
            $old_sm = Shipment::findOrFail($shipment['shipment_id']);
            // Lấy dữ liệu user cũ
            $user = User::findOrFail($old_sm['user_id']);
            // Lấy dữ liệu package
            $packages = $request_data['shipment']['packages'];

            $distance_time = Carbon::parse($old_sm['created_at'])->diffInMinutes(Carbon::now());
            if ((onCheckSaleAccount() || onCheckSaleAccount(SETTING_FWD_ACCOUNT)) && $distance_time >= 5) { {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'title' => 'Thất bại!',
                            'text' => 'Đơn này đã quá hạn chỉnh sửa. Chỉ được chỉnh sửa trong vòng 5 phút kể từ lúc tạo đơn.',
                            'icon' => 'error',
                        ]
                    ]);
                }
            }
            if ($old_sm['shipment_status'] == SHIPMENT_STATUS_EXPORED && !isGetAllData()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Kiện hàng đã xuất bạn không đủ quyền hạn để cập nhật!', null, 'warning');
            }

            // Kiểm tra xem đơn hàng đã quét chưa nếu đã quét thì không thể chỉnh sửa của đơn hàng ePacket
            if ($service['promotion_flg'] == IS_EPACKET) {
                $count_package_scan = Package::where('shipment_id', $shipment['shipment_id'])->where('package_approve', '!=', PACKAGE_APPROVE_WAITING)->count();
                if ($count_package_scan > 0) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'title' => 'Thất bại!',
                            'text' => 'Không thể cập nhật khi đơn hàng đã quét',
                            'icon' => 'error',
                        ]
                    ]);
                }
            }

            $count_check_e_package = collect($packages)->where('package_quantity', '>', 1)->count();
            // Nếu dịch vụ là ePacket và tổng số gói hàng lớn hơn 1 thì hiện thông báo lỗi
            if ($service['promotion_flg'] == IS_EPACKET && (count($packages) > 1 || $count_check_e_package > 0)) {
                return response()->json([
                    'status' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => [
                        'icon' => 'error',
                        'title' => 'Thông báo',
                        'text' => 'Dịch vụ ePacket chỉ cho phép tạo 1 gói hàng'
                    ]
                ], HTTP_INTERNAL_SERVER_ERROR);
            }

            // if (isset($shipment['shipment_reference_code']) && $shipment['shipment_reference_code'] != "") {
            //     if (Shipment::isShow()->where('shipment_reference_code', $shipment['shipment_reference_code'])->where('shipment_id', '!=', $shipment['shipment_id'])->exists()) {
            //         return response()->json([
            //             'status' => HTTP_INTERNAL_SERVER_ERROR,
            //             'message' => [
            //                 'title' => 'Thất bại!',
            //                 'text' => 'Reference Code đã tồn tại.',
            //                 'icon' => 'error',
            //             ]METHOD-2024012030-0-1735547789.jpg
            //         ]);
            //     }
            // }

            if (!isset($shipment['agree_terms_use_service']) || !json_decode($shipment['agree_terms_use_service'])) {
                return response()->json([
                    'status' => HTTP_UNPROCESSABLE_ENTITY,
                    'message' => 'Bạn cần đồng ý với điều khoản của chúng tôi',
                ]);
            }

            if (isset($shipment['sale_id'])) {
                $user = User::find($shipment['sale_id']);
                if (!$user) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy tài khoản sale');
                }
            }
            $save_receiver_flg = isset($shipment['save_receiver_flg']) ? json_decode($shipment['save_receiver_flg']) : false;
            if ($save_receiver_flg) {
                $save_receiver_id = null;
                if (onCheckPositionAccounts(SETTING_DOCUMENT_ACCOUNT, auth()->user()['position_id'])) {
                    $save_receiver_id = auth()->id();
                }
                $receiver = Receiver::updateOrCreate(
                    [
                        'receiver_id' => isset($shipment['receiver_id']) ? $shipment['receiver_id'] : null,
                    ],
                    [
                        'user_id' => $save_receiver_id ?? $user['user_id'],
                        'receiver_company_name' => isset($shipment['receiver_company_name']) ? $shipment['receiver_company_name'] : '',
                        'receiver_contact_name' => $shipment['receiver_contact_name'],
                        'receiver_telephone' => $shipment['receiver_telephone'],
                        'receiver_country_id' => $shipment['receiver_country_id'],
                        'receiver_state_id' => $shipment['receiver_state_id'] ?? null,
                        'receiver_city_id' => $shipment['receiver_city_id'],
                        'receiver_postal_code' => $shipment['receiver_postal_code'],
                        'receiver_address_1' => $shipment['receiver_address_1'],
                        'receiver_address_2' => $shipment['receiver_address_2'],
                        'receiver_address_3' => $shipment['receiver_address_3'],
                    ]
                );
            }

            // $check_accountant_status = onCheckAccountantStatus($old_sm->accountant_status);
            // if (!$check_accountant_status['status']) {
            //     return $check_accountant_status['res'];
            // }
            $old_shipment_reference_code = $old_sm->shipment_reference_code;

            if (!isset($shipment['receiver_state_id'])) {
                $state = State::isActive(ACTIVE)->isDelete(NO_DELETE)
                    ->where('country_id', $shipment['receiver_country_id'])
                    ->where(function ($query) use ($shipment) {
                        $query->where('state_name', 'like', $shipment['receiver_state_name'])
                            ->orWhere('state_code', 'like', $shipment['receiver_state_name']);
                    })->first();

                if (isset($state)) {
                    $shipment['receiver_state_id'] = $state['state_id'];
                }
            }

            Shipment::findOrFail($shipment['shipment_id'])->update([
                'user_id' => $user['user_id'],
                'shipment_service_id' => $shipment['shipment_service_id'],
                'shipment_signature_flg' => json_decode($shipment['shipment_signature_flg']) ? SHIPMENT_SIGNATURE_YES : SHIPMENT_SIGNATURE_NO,
                'shipment_branch_id' => $shipment['shipment_branch_id'],
                'shipment_reference_code' => $shipment['shipment_reference_code'],
                'shipment_status' => SHIPMENT_STATUS_CREATE_BILL,
                'shipment_goods_name' => $shipment['shipment_goods_name'],
                'shipment_value' => $shipment['shipment_value'],
                'shipment_export_as' => $shipment['shipment_export_as'],

                'sender_company_name' => $shipment['sender_company_name'],
                'sender_contact_name' => $shipment['sender_contact_name'],
                'sender_telephone' => $shipment['sender_telephone'],
                'sender_city' => $shipment['sender_city'],
                'sender_district' => $shipment['sender_district'],
                'sender_ward' => $shipment['sender_ward'],
                'sender_address' => $shipment['sender_address'],
                'sender_longitude' => isset($shipment['sender_longitude']) ? $shipment['sender_longitude'] : null,
                'serder_latitude' => isset($shipment['serder_latitude']) ? $shipment['serder_latitude'] : null,

                'receiver_id' => $save_receiver_flg ? $receiver['receiver_id'] : null,
                'receiver_company_name' => isset($shipment['receiver_company_name']) ? $shipment['receiver_company_name'] : '',
                'receiver_contact_name' => $shipment['receiver_contact_name'],
                'receiver_telephone' => $shipment['receiver_telephone'],
                'receiver_country_id' => $shipment['receiver_country_id'],
                'receiver_state_id' => $shipment['receiver_state_id'] ?? null,
                'receiver_state_name' => $shipment['receiver_state_name'] ?? null,
                'receiver_city_id' => $shipment['receiver_city_id'],
                'receiver_postal_code' => $shipment['receiver_postal_code'],
                'receiver_address_1' => $shipment['receiver_address_1'],
                'receiver_address_2' => $shipment['receiver_address_2'],
                'receiver_address_3' => $shipment['receiver_address_3'],

                'document_id' => $old_sm->document_id,
            ]);

            $_packages = [];
            $package_ids = collect($packages)->whereNotNull('package_id')->pluck('package_id')->toArray();
            Package::where('shipment_id', $shipment['shipment_id'])->whereNotIn('package_id', $package_ids)->delete();
            if (collect($packages)->whereNotNull('package_id')->where('package_quantity', '>', 1)->toArray()) {
                return response()->json([
                    'status' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => [
                        'title' => 'Thất bại!',
                        'text' => 'Có đơn hàng đã khởi tạo số lượng lớn hơn 1.',
                        'icon' => 'error',
                    ]
                ]);
            }
            $total_weight = getTotalPackage($packages, $service['service_volumetric_mass']);
            foreach ($packages as $package) {
                $package['shipment_id'] = $shipment['shipment_id'];
                // $package['package_weight'] = formatNumberCeil($package['package_weight'], $total_weight, $service['promotion_flg']);
                $package['package_weight'] = $package['package_weight'];
                $package['package_converted_weight'] = $package['package_length'] * $package['package_width'] * $package['package_height'] / $service['service_volumetric_mass'];
                $package['package_charged_weight'] = $package['package_weight'] < $package['package_converted_weight'] ? $package['package_converted_weight'] : $package['package_weight'];
                $package['package_charged_weight'] = formatNumberCeil($package['package_charged_weight'], $total_weight, $service['promotion_flg']);

                $package['package_description'] = '';
                $package_quantity = $package['package_quantity'];

                for ($i = 1; $i <= $package_quantity; $i++) {
                    if (!isset($package['package_id'])) {
                        $package_code = getCodePackage();
                        $package['package_code'] = $package['package_code'] ?? $package_code['package_code'];
                        $package['package_hawb_code'] = $package['package_code'] ?? $package_code['package_code'];
                        // $package['package_code'] = $package_code['package_code'];
                        // $package['package_tracking_code'] = $package_code['package_tracking_code'];
                        $package['package_quantity'] = 1;
                    }
                    Package::updateOrCreate(
                        [
                            'package_id' => isset($package['package_id']) ? $package['package_id'] : null,
                        ],
                        $package
                    );
                    if (!isset($package['package_id'])) {
                        onCreatePackageTrackingStatus($package['package_code'], 'HO CHI MINH,VN ', 'LABEL CREATED');
                    }

                    array_push($_packages, [
                        'length' => (float) $package['package_length'],
                        'width' => (float) $package['package_width'],
                        'height' => (float) $package['package_height'],
                        'weight' => (float) $package['package_charged_weight'],
                    ]);
                }
            }

            $invoices = !empty($request_data['shipment']['invoices']) ? $request_data['shipment']['invoices'] : [];
            $invoice_ids = collect($invoices)->whereNotNull('invoice_id')->pluck('invoice_id')->toArray();
            Invoice::where('shipment_id', $shipment['shipment_id'])->whereNotIn('invoice_id', $invoice_ids)->delete();
            $_invoices = [];
            foreach ($invoices as $invoice) {
                if (!isset($invoice['invoice_id'])) {
                    $invoice['invoice_code'] = initializationInvoiceCode();
                }
                $invoice['shipment_id'] = $shipment['shipment_id'];
                $invoice_created = Invoice::updateOrCreate(
                    [
                        'invoice_id' => isset($invoice['invoice_id']) ? $invoice['invoice_id'] : null,
                    ],
                    $invoice
                );
                array_push($_invoices, $invoice_created->toArray());
            }

            // refreshPrice($shipment_created['shipment_id']);


            if (isset($service['promotion_flg']) && $service['promotion_flg'] != IS_EPACKET) {
                ModelsLabelNz::isShow()->where("shipment_id", $shipment['shipment_id'])->update(['delete_flg' => DELETED]);
            }
            $result_save_amount = onSaveAmountShipment($shipment['shipment_id']);

            if ($result_save_amount['status'] === HTTP_INTERNAL_SERVER_ERROR) {
                return response()->json($result_save_amount);
            }

            // onUpdateOperatingCostsAmount($shipment['shipment_id']);

            // tính giá trị cuối cùng của đơn và lợi nhuận 
            onUpdateFinalAmountShipment($shipment['shipment_id']);

            // $result_update_wallet = onUpdateWalletUser($shipment['shipment_id']);
            // if ($result_update_wallet['status'] === HTTP_INTERNAL_SERVER_ERROR) {
            //     Session::flash('error', $result_update_wallet['message']);
            // }

            $shipment_created = Shipment::findOrFail($shipment['shipment_id']);

            // Khởi tạo order_pickup
            if (isset($request_data['order_pickup']) && json_decode($request_data['order_pickup'])) {
                if (empty($shipment['serder_latitude']) || empty($shipment['sender_longitude'])) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Để tạo Order Pickup bạn cần xác định vị trí trên bản đồ', null, 'warning');
                }
                // Xử lý tạo order pickup
                $this->onCreateOrderPickup($shipment, $packages, $user, $request);
            }

            // Nếu dịch vụ là ePacket thì tính tiền ở ví
            if ($shipment_created['service']['promotion_flg'] == IS_EPACKET) {
                // Phân bổ tiền khi cập nhật shipment
                $result = allocateFundsWhenUpdatingShipment($shipment_created['shipment_id'], $old_sm['shipment_final_amount'], true);
                if ($result['status'] != HTTP_OK) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Bạn không đủ tiền trong ví để thanh toán, vui lòng nạp thêm tiền vào ví'
                        ]
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Cập nhật trạng thái thanh toán của shipment
                $shipment_created->shipment_payment_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment_created->accountant_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;

                $shipment_created->save();
            }

            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $request->user()['user_id'],
                'Cập nhật lại shipment'
            );

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => 'Cập nhật shipment thành công',
                'redirect_url' => route('shipments.detail.show', ['shipment_code' => $shipment['shipment_code']]),
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    //api
    public function getUnitCreatePackageAndInvoice(Request $request)
    {
        $package_types = config('constans.constans.package_types');
        $invoice_exports_as = config('constans.constans.invoice_exports_as');
        $invoice_units = config('constans.constans.invoice_units');

        return response()->json([
            'status' => HTTP_OK,
            'data' => [
                "package_types" => $package_types,
                "invoice_exports_as" => $invoice_exports_as,
                "invoice_units" => $invoice_units,
            ],
        ]);
    }

    public function getReceivers(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $user = $is_api ? auth()->user() : $request->user();

            $receivers = Receiver::isUser($user['user_id'])->isShow()->select('receiver_id', 'receiver_contact_name')->get()->toArray();

            return response()->json([
                'status' => HTTP_OK,
                'receivers' => $receivers,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getDetailReceiver(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $user =  $request->user();

            $receiver = Receiver::isUser($user['user_id'])->isShow()
                ->leftJoin('cities', 'receivers.receiver_city_id', 'cities.city_id')
                ->where('receiver_id', $request_data['receiver_id'])
                ->select('receivers.*', 'cities.city_name')
                ->first();
            if (!$receiver) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy người dùng cũ của bạn');
            }

            return response()->json([
                'status' => HTTP_OK,
                'receivers' => $receiver,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getShipmentDetail(Request $request)
    {
        try {
            $request_data = $request->all();
            $shipment_code = $request_data['shipment_code'];
            $user = request()->user();
            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);
            $view_operating_costs_position_ids = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_VIEW_AND_UPDATE_SHIPMENT_OPERATING_COST)['position_ids'];
            $is_view_shipment_operating_costs = !onCheckPositionAccounts(SETTING_FWD_ACCOUNT, $user->position_id);

            $shipment = Shipment::with([
                'user',
                'service',
                'branch',
                'country',
                'city',
                'packages',
                'invoices',
                'shipmentOperatingCosts' => function ($query) {
                    return $query->leftJoin('operating_costs', 'shipment_operating_costs.operating_cost_id', '=', 'operating_costs.operating_cost_id')
                        ->select('shipment_operating_costs.*', 'operating_costs.operating_cost_name');
                }
            ])->where('shipments.import_approval', ACTIVE)
                ->where('shipment_code', $shipment_code)->isDelete(NO_DELETE);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->first();
            if (!$shipment) {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => 'Không tìm thấy shipment',
                ]);
            }
            return response()->json([
                'status' => HTTP_OK,
                'shipment' => $shipment,
                'is_view_shipment_operating_costs' => $is_view_shipment_operating_costs
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getShipmentEdit(Request $request)
    {
        try {
            $request_data = $request->all();
            $shipment_code = $request_data['shipment_code'];
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? $request->user() : auth()->user();
            $receivers = Receiver::isUser($user['user_id'])->isActive(ACTIVE)->isDelete(NO_DELETE)->select('receiver_id', 'receiver_contact_name')->get()->toArray();
            $countries = Country::isActive(ACTIVE)->isDelete(NO_DELETE)->select('country_id', 'country_name')->get()->toArray();
            $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->select('branch_id', 'branch_name')->get()->toArray();
            $package_types = config('constans.constans.package_types');
            $invoice_exports_as = config('constans.constans.invoice_exports_as');
            $invoice_units = config('constans.constans.invoice_units');

            $path = 'public/config/system-settings.json';
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);

            $shipment = Shipment::with([
                'packages' => function ($query) {
                    return $query->select('packages.package_id', 'packages.shipment_id', 'packages.package_quantity', 'packages.package_type', 'packages.package_length', 'packages.package_width', 'packages.package_height', 'packages.package_weight');
                },
                'invoices' => function ($query) {
                    return $query->select('invoices.invoice_id', 'invoices.shipment_id', 'invoices.invoice_goods_details', 'invoices.invoice_quantity', 'invoices.invoice_unit', 'invoices.invoice_price', 'invoices.invoice_total_price');
                }
            ])->where('shipments.import_approval', ACTIVE)->where('shipment_code', $shipment_code)->isDelete(NO_DELETE);
            if (!$is_view_all_shipment) {
                $shipment = $shipment->where('user_id', $user['user_id']);
            }
            $shipment = $shipment->select(
                'shipments.shipment_id',
                'shipments.shipment_code',
                'shipments.shipment_service_id',
                'shipments.shipment_signature_flg',
                'shipments.shipment_branch_id',
                'shipments.shipment_reference_code',
                'shipments.shipment_goods_name',
                'shipments.shipment_value',
                'shipments.shipment_export_as',
                'shipments.user_id',
                'shipments.receiver_id',
                'shipments.sender_company_name',
                'shipments.sender_contact_name',
                'shipments.sender_telephone',
                'shipments.sender_city',
                'shipments.sender_district',
                'shipments.sender_ward',
                'shipments.sender_address',
                'shipments.receiver_company_name',
                'shipments.receiver_contact_name',
                'shipments.receiver_telephone',
                'shipments.receiver_country_id',
                'shipments.receiver_state_id',
                'shipments.receiver_state_name',
                'shipments.receiver_city_id',
                'shipments.receiver_postal_code',
                'shipments.receiver_address_1',
                'shipments.receiver_address_2',
                'shipments.receiver_address_3',
            )->first();

            if (!$shipment) {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => 'Không tìm thấy shipment',
                ]);
            }

            $address = address($shipment['sender_city'], $shipment['sender_district']);
            return response()->json([
                'status' => HTTP_OK,
                'shipment' => $shipment,
                'address' => $address,
                'receivers' => $receivers,
                'countries' => $countries,
                'branchs' => $branchs,
                'package_types' => $package_types,
                'invoice_exports_as' => $invoice_exports_as,
                'invoice_units' => $invoice_units,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function onCreateLabelReload(Request $request)
    {
        $this->validate($request, [
            "shipment_id" => 'required|numeric',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $shipment = Shipment::isShow()->find($request_data['shipment_id']);
            if (!$shipment) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đơn vận chuyển', $request_data['shipment_id']);
            }
            if ($shipment->shipment_check_create_label == 1) {
                return returnResponse(HTTP_NOT_FOUND, 'Đơn này đã tạo label thành công rồi');
            }
            $service = Service::isShow()->findOrFail($shipment->shipment_service_id);
            if (!$service) {
                return returnResponse(HTTP_NOT_FOUND, 'Dịch vụ không còn tồn tại để cập nhật');
            }

            if ($service->promotion_flg != IS_EPACKET) {
                return returnResponse(HTTP_BAD_REQUEST, 'Dịch vụ không hợp lệ');
            }

            $packages = Package::isShow()->where('shipment_id', $shipment->shipment_id)
                ->select(
                    'packages.*',
                    'package_length as length',
                    'package_width as width',
                    'package_height as height',
                    'package_charged_weight as weight'
                )->get();
            $invoices = Invoice::isShow()->where('shipment_id', $shipment->shipment_id)->get();

            $epacket = onCreateLabel($shipment, $packages, $invoices, $service);
            if ($epacket['status'] == HTTP_OK) {
                $res_label = $epacket['res_label'];
                $OrderId = $epacket['OrderId'];
                if ($res_label['status'] != HTTP_OK) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Khởi tạo label thất bại');
                } else {
                    Shipment::find($shipment->shipment_id)->update([
                        'shipment_reference_code' => $OrderId,
                        'shipment_check_create_label' => 1,
                    ]);
                    Session::flash('success', 'Cập nhật label cho shipment thành công');
                    return returnResponse(HTTP_OK, 'Khởi tạo label thành công', null, 'success');
                }
            } else {
                return returnResponse(HTTP_BAD_REQUEST, 'Khởi tạo label thất bại');
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }


    public function handleLabelEdit(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $user = $is_api ? $request->user() : auth()->user();

            $shipment = Shipment::where('shipment_code', $request_data['shipment_code'])->where('shipments.import_approval', ACTIVE)->first();


            if ($shipment) {

                if ($shipment->user_id != $user['user_id'] && (!isGetAllData() || !onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || !onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT))) {
                    return response()->json([
                        'status' => HTTP_FORBIDDEN,
                        'message' => [
                            "icon" => "warning",
                            "title" => "Cảnh báo",
                            "text" => "Bạn không đủ quyền hạn để cập nhật"
                        ],
                    ]);
                }

                $shipment_file_label = '';
                $old_file = '';
                if ($is_api) {
                    $path = PUBLIC_PATH_SHIPMENT_LABEL;
                    $file_name = "LABEL-" . $shipment->shipment_code . '-' . time();

                    $res = createFileUpload($request_data['file_label'], $path, $file_name, 'pdf');
                    if ($res['status']) {
                        $shipment_file_label = $res['path'];
                        $old_file = $shipment->shipment_file_label;
                    } else {
                        return $res['res'];
                    }
                } else {
                    if ($request->hasFile('file_label')) {
                        $file = $request->file('file_label');
                        $fileName = 'Label-' . time() . '.' . pathinfo($file->getClientOriginalName(), PATHINFO_EXTENSION);; // Lấy tên gốc của tệp
                        $file->move(public_path(PUBLIC_PATH_SHIPMENT_LABEL), $fileName);

                        $shipment_file_label = PUBLIC_PATH_SHIPMENT_LABEL . $fileName;
                        $old_file = $shipment->shipment_file_label;
                    }
                }

                DB::beginTransaction();
                $shipment->update([
                    "shipment_file_label" => $shipment_file_label,
                ]);

                onCreateHistoryShipment(
                    $shipment['shipment_id'],
                    $request->user()['user_id'],
                    'Cập nhật label'
                );

                DB::commit();

                if (!empty($shipment_file_label) && !empty($old_file) && File::exists($old_file)) {
                    File::delete($old_file);
                }

                return response()->json([
                    'status' => HTTP_OK,
                    'url' => $shipment_file_label,
                    'message' => [
                        "icon" => "success",
                        "title" => "Thông báo",
                        "text" => "Cập nhật label thành công"
                    ]
                ]);
            } else {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => [
                        "icon" => "error",
                        "title" => "Cảnh báo",
                        "text" => "Không tìm thấy shipment"
                    ]
                ]);
            }
        } catch (\Throwable $th) {
            if (isset($shipment_file_label) && File::exists($shipment_file_label)) {
                File::delete($shipment_file_label);
            }
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleMethodEdit(Request $request)
    {
        $this->validate($request, [
            'file_method' => 'required'
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $user = $is_api ? $request->user() : auth()->user();
            $shipment = Shipment::where('shipment_code', $request_data['shipment_code'])->where('shipments.import_approval', ACTIVE)->first();
            // if (!onCheckAccountantStatus($shipment['accountant_status'])['status']) {
            //     return returnResponse(HTTP_BAD_REQUEST), null, 'warning');
            // }

            if ($shipment) {
                if ($shipment->user_id != $user['user_id'] && !isGetAllData()) {
                    return response()->json([
                        'status' => HTTP_FORBIDDEN,
                        'message' => [
                            "icon" => "warning",
                            "title" => "Cảnh báo",
                            "text" => "Bạn không đủ quyền hạn để cập nhật"
                        ],
                    ]);
                }


                $new_file = [];
                $old_file = $shipment['shipment_file_proof_of_payment'];

                $path = PUBLIC_PATH_SHIPMENT_METHOD;
                $file_name = "METHOD-" . $shipment->shipment_code;
                $new_file = convertImagesBase64ToDirectory($request_data['file_method'], $path, $file_name, 'jpg');

                if (empty($new_file)) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Xảy ra lỗi khi cập nhật hình ảnh');
                }
                if (!in_array($shipment['shipment_payment_step'], [0, 2])) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Vui lòng chờ kế toán duyệt đơn', null, 'warning');
                }

                $old_step = $shipment['shipment_payment_step'];

                $shipment['shipment_payment_step'] = 3;
                if ($shipment['shipment_paid_by'] == SHIPMENT_PAID_BY_REMAINING && $old_step == 0) {
                    $shipment['shipment_payment_step'] = 1;
                }
                DB::beginTransaction();
                $shipment->update([
                    "shipment_file_proof_of_payment" => json_encode($new_file),
                    "shipment_payment_method" => $request_data['shipment_payment_method'],
                    "shipment_payment_date" => Carbon::now()->format('Y-m-d H:i:s'),
                ]);
                DB::commit();
                if (!empty($old_file)) {
                    foreach (onConvertStrArrImage($old_file) as $key => $img) {
                        if (File::exists($img)) {
                            File::delete($img);
                        }
                    }
                }
                if (!$is_api) {
                    Session::flash('Cập nhật thành công', 'success');
                }
                return response()->json([
                    'status' => HTTP_OK,
                    'message' => [
                        "icon" => "success",
                        "title" => "Thông báo",
                        "text" => "Cập nhật phương thức thanh toán thành công"
                    ]
                ]);
            } else {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => [
                        "icon" => "error",
                        "title" => "Cảnh báo",
                        "text" => "Không tìm thấy shipment"
                    ]
                ]);
            }
        } catch (\Throwable $th) {
            if (!empty($new_file)) {
                foreach ($new_file as $key => $img) {
                    if (File::exists($new_file)) {
                        File::delete($new_file);
                    }
                }
            }
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleAmountFreight(Request $request)
    {
        $request->validate([
            "shipment_amount_total_customer" => 'required|numeric|min:0',
            "shipment_domestic_charges" => 'required|numeric|min:0',
            "shipment_amount_surcharge" => 'required|numeric|min:0',
            "shipment_collection_fee" => 'required|numeric|min:0',
            "shipment_amount_insurance" => 'required|numeric|min:0',
            "shipment_amount_vat" => 'required|numeric|min:0',
            "shipment_amount_original" => 'required|numeric|min:0',
            // "shipment_amount_residential" => 'required|numeric|min:0',
            // "shipment_amount_peak" => 'required|numeric|min:0',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $user = $is_api ? $request->user() : auth()->user();
            $shipment = Shipment::where('shipment_code', $request_data['shipment_code'])->where('shipments.import_approval', ACTIVE)->first();
            $old_shipment = $shipment;

            $check_accountant_status = onCheckAccountantStatus($shipment->accountant_status);
            if (!$check_accountant_status['status'] && $shipment['service']['promotion_flg'] != IS_EPACKET) {
                return response()->json([
                    'status' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => $check_accountant_status['message'],
                ], HTTP_INTERNAL_SERVER_ERROR);
            }

            // refreshPrice($shipment['shipment_id']);
            // if (!isGetAllData() || !onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT)) {
            //     return response()->json([
            //         'status' => HTTP_FORBIDDEN,
            //         'message' => [
            //             "icon" => "warning",
            //             "title" => "Cảnh báo",
            //             "text" => "Bạn không đủ quyền hạn để cập nhật"
            //         ],
            //     ]);
            // }

            $shipment_amount_operating_costs = ShipmentOperatingCost::where('shipment_id', $shipment->shipment_id)->sum('shipment_operating_cost_amount');
            $shipment_total_amount = $request_data['shipment_domestic_charges'] + $request_data['shipment_amount_surcharge']
                + $request_data['shipment_collection_fee'] + $request_data['shipment_amount_insurance']
                + $request_data['shipment_amount_vat'] + $request_data['shipment_amount_original']
                + $shipment_amount_operating_costs;
            $shipment_amount_profit = $request_data['shipment_amount_total_customer'] - $shipment_total_amount;

            $shipment->shipment_amount_total_customer = $request_data['shipment_amount_total_customer'];
            $shipment->shipment_domestic_charges = $request_data['shipment_domestic_charges'];
            $shipment->shipment_amount_surcharge = $request_data['shipment_amount_surcharge'];
            $shipment->shipment_collection_fee = $request_data['shipment_collection_fee'];
            $shipment->shipment_amount_insurance = $request_data['shipment_amount_insurance'];
            $shipment->shipment_amount_vat = $request_data['shipment_amount_vat'];
            $shipment->shipment_amount_original = $request_data['shipment_amount_original'];
            $shipment->shipment_amount_residential = $request_data['shipment_amount_residential'] ?? $shipment['shipment_amount_residential'];
            $shipment->shipment_amount_peak = $request_data['shipment_amount_peak'] ?? $shipment['shipment_amount_peak'];
            $shipment->shipment_amount_profit = $shipment_amount_profit;
            $shipment->save();
            onUpdateFinalAmountShipment($shipment['shipment_id']);

            HistoryEditShipment::create([
                'shipment_id' => $shipment['shipment_id'],
                'edit_by' => $user['user_id'],
                'edit_at' => Carbon::now()
            ]);

            // Nếu dịch vụ là ePacket thì tính tiền ở ví
            if ($shipment['service']['promotion_flg'] == IS_EPACKET) {
                $check_packages = collect($shipment['packages'])->where('package_approve', '!=', PACKAGE_APPROVE_WAITING)->count();
                if ($check_packages > 0) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => 'Không thể cập nhật giá cước do có kiện hàng đã được duyệt'
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Phân bổ tiền khi cập nhật shipment
                $result = allocateFundsWhenUpdatingShipment($shipment['shipment_id'], $old_shipment['shipment_final_amount'], true);
                if ($result['status'] != HTTP_OK) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Bạn không đủ tiền trong ví để thanh toán, vui lòng nạp thêm tiền vào ví'
                        ]
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Cập nhật trạng thái thanh toán của shipment
                $shipment->shipment_payment_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment->accountant_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;

                $shipment->save();
            }

            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $request->user()['user_id'],
                'Cập nhật chi phí phụ thu khách'
            );

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'shipment' => Shipment::find($shipment['shipment_id']),
                'message' => [
                    "icon" => "success",
                    "title" => "Thông báo",
                    "text" => "Cập nhật giá cước thành công"
                ]
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }


    public function displayLocation(Request $request, $code)
    {

        try {
            $request_data = $request->all();

            $is_api = isset($shipment_code) ? false : true;

            return view('shipment.location');
        } catch (\Throwable $th) {
            //throw $th;
        }
    }

    public function convertDataShipmentToLabel($shippingMethodCode, $data)
    {
        try {
            $data['weightUnit'] = "kg";
            $data['total_weight'] = collect($data['packages'])->sum('weight');
            $data['total_package_weight'] = collect($data['packages'])->sum('package_weight');

            $weight_convert = WeightConversion::where('service_id', $data['shipment_service_id'])
                ->where('initial_weight', '>=', $data['total_package_weight'])
                ->first();

            if (isset($weight_convert)) {
                $data['total_weight'] = $weight_convert->weight_conversion;
                $data['weightUnit'] = Str::lower($weight_convert->weight_unit);
            }

            $data_ex = [
                'referenceNo' => isset($data['shipment_reference_code']) ? $data['shipment_reference_code'] : null,
                'shippingMethodCode' => $shippingMethodCode,
                'shippingFirstName' => $data['receiver_contact_name'],
                'shippingPhone' => $data['receiver_telephone'],
                'email' => isset($data['email']) ? $data['email'] : auth()->user()['user_name'],
                'shippingAddress' => $data['receiver_address_1'],
                'shippingAddress1' => $data['receiver_address_2'],
                'shippingCity' => $data['city_code'],
                'shippingState' => $data['state_code'],
                'shippingZip' => $data['receiver_postal_code'],
                'iosscode' => isset($data['shipment_iosscode']) ? $data['shipment_iosscode'] : "",
                'countryCode' => $data['country_code'],
                'instruction' => isset($data['instruction']) ? $data['instruction'] : '',
                'itemCount' => count($data['packages']),
                'applicationName' => $data['shipment_goods_name'],
                'unitPrice' => $data['shipment_value'],
                'weight' => $data['total_weight'],
                'height' => collect($data['packages'])->sum('height'),
                'length' => collect($data['packages'])->sum('length'),
                'width' => collect($data['packages'])->sum('width'),
                'dimensionUnit' => 'cm',
                'weightUnit' => $data['weightUnit'],
                'shipperName' => $data['sender_contact_name'],
                'childOrders' => [],
            ];
            $childOrders = [];
            foreach ($data['packages'] as $key => $value) {
                $item = [
                    'length' => (int) $value['length'],
                    'width' => (int) $value['width'],
                    'height' => (int) $value['height'],
                    'weight' => (float) $value['weight'],
                ];
                array_push($childOrders, $item);
            }

            $data_ex['childOrders'] = $childOrders;

            $nz_controller = new NzPostController();
            $form_post = $nz_controller->createLabel($data_ex);

            if ($form_post['status']) {
                return [
                    'status' => HTTP_OK,
                    'data' => $form_post['data']
                ];
            }

            return [
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'data' => [
                    'code' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => $form_post['message']
                ],
            ];
        } catch (\Throwable $th) {
            return [
                'status' => false,
                'message' => "lỗi khi cố gắp khởi tạo label"
            ];
        }
    }

    public function handleCheckHoldShipment(Request $request)
    {
        $this->validate($request, [
            "shipment_code" => 'required',
        ]);

        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $shipment = Shipment::with(['country'])->isShow()->where('shipment_code', $request_data['shipment_code'])->where('shipments.import_approval', ACTIVE)->first()->toArray();

            $shipment_status = $shipment['shipment_status'] == SHIPMENT_STATUS_HOLD ? SHIPMENT_STATUS_EXPORED : SHIPMENT_STATUS_HOLD;
            $shipment_total_amount = $shipment['shipment_domestic_charges'] + $shipment['shipment_amount_surcharge']
                + $shipment['shipment_collection_fee'] + $shipment['shipment_amount_insurance']
                + $shipment['shipment_amount_vat'] + $shipment['shipment_amount_original']
                + $shipment['shipment_amount_operating_costs'];

            $shipment_amount_profit = $shipment['shipment_amount_total_customer'] - $shipment_total_amount;

            if ($shipment_status == SHIPMENT_STATUS_HOLD && $shipment['country']['country_code'] == 'US') {
                $shipment_amount_profit = 0;
            }

            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $request->user()['user_id'],
                'Check hold shipment'
            );

            $shipment = Shipment::findOrFail($shipment['shipment_id'])->update([
                'shipment_status' => $shipment_status,
                'shipment_amount_profit' => $shipment_amount_profit,
            ]);


            DB::commit();
            Session::flash('success', 'Cập nhật trạng thái thành công');
            return response()->json([
                'status' => HTTP_OK
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handlePaymentShipment(Request $request)
    {
        $this->validate($request, [
            "shipment_code" => 'required|exists:shipments,shipment_code',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();

            $shipment = Shipment::leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->where('shipments.shipment_code', $request_data['shipment_code'])
                ->where('shipments.shipment_payment_status', PENDING)
                ->where('services.promotion_flg', IS_EPACKET)
                ->where('shipments.import_approval', ACTIVE)
                ->select('shipments.*')
                ->first();

            if (!isset($shipment)) {
                return response()->json([
                    'status' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => [
                        'title' => 'Thất bại!',
                        'text' => 'Đơn hàng không tồn tại hoặc đã được thanh toán',
                        'icon' => 'error',
                    ]
                ], HTTP_INTERNAL_SERVER_ERROR);
            }

            $wallet = Wallet::where('user_id', $shipment['user_id'])->first();
            if ($wallet['amount'] < $shipment['shipment_final_amount']) {
                return response()->json([
                    'status' => HTTP_INTERNAL_SERVER_ERROR,
                    'message' => [
                        'title' => 'Thất bại!',
                        'text' => 'Số dư ví không đủ để thanh toán',
                        'icon' => 'error',
                    ]
                ], HTTP_INTERNAL_SERVER_ERROR);
            }

            $shipment->shipment_payment_status = SUCCESS;
            $shipment->accountant_status = SUCCESS;

            $wallet->amount -= $shipment['shipment_final_amount'];

            WalletFluctuation::create([
                'wallet_id' => $wallet['wallet_id'],
                'secure_hash' => 'WF' . Carbon::now()->format('YmdHisv'),
                'user_id' => $shipment['user_id'],
                'amount' => $shipment['shipment_final_amount'],
                'wallet_amount' => $wallet['amount'],
                'content' => 'Thanh toán đơn hàng ' . $shipment['shipment_code'],
                'kind' => MINUS,
            ]);

            $shipment->save();
            $wallet->save();

            DB::commit();
            Session::flash('success', 'Cập nhật trạng thái thanh toán thành công');
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Cập nhật trạng thái thanh toán thành công',
                    'icon' => 'success',
                ]
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function displayApproveShipment()
    {
        $branchs = Branch::isShow()->get();
        $shipment_filter_by = config('constans.constans.shipment_filter_by');
        $services = Service::isShow()->select('service_id', 'service_name')->get();
        return view('shipment-approve.list', compact('branchs', 'services', 'shipment_filter_by'));
    }

    public function getApproveShipments(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $filters = isset($request_data['filters']) ? $request_data['filters'] : [];
            $filters = is_array($filters) ? $filters : json_decode($filters, true);
            $is_view_all_shipment = isGetAllData() || onCheckDocumentAccount(SETTING_OPS_LEADER_ACCOUNT) || onCheckDocumentAccount(SETTING_SHIPPER_ACCOUNT);

            $shipments = Shipment::with(['user', 'service'])
                ->isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('shipments.import_approval', INACTIVE);

            if (!$is_view_all_shipment) {
                $shipments = $shipments->where('shipments.user_id', request()->user()['user_id']);
            }

            $shipments = $shipments->isFilters($filters)
                ->orderBy('shipment_id', 'DESC')
                ->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            $html = view('shipment-approve.pagination', compact('shipments'))->render();
            return response()->json([
                'status' => HTTP_OK,
                'html' => $html
            ], HTTP_OK);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleConfirmApproveShipment(Request $request)
    {
        $this->validate($request, [
            "shipment_code" => 'required|exists:shipments,shipment_code',
        ], [
            'shipment_code.required' => 'Mã lô hàng không được để trống',
            'shipment_code.exists' => 'Mã lô hàng không tồn tại',
        ]);
        try {
            DB::beginTransaction();
            $request_data = $request->all();

            $shipment = Shipment::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('shipment_code', $request_data['shipment_code'])->first();

            $service = $shipment['service'];

            // Nếu dịch vụ là ePacket thì tính tiền ở ví
            if ($service['promotion_flg'] == IS_EPACKET) {
                // Phân bổ tiền khi cập nhật shipment
                $result = allocateFundsWhenUpdatingShipment($shipment['shipment_id']);
                if ($result['status'] != HTTP_OK) {
                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => [
                            'icon' => 'error',
                            'title' => 'Thông báo',
                            'text' => 'Bạn không đủ tiền trong ví để thanh toán, vui lòng nạp thêm tiền vào ví'
                        ]
                    ], HTTP_INTERNAL_SERVER_ERROR);
                }

                // Cập nhật trạng thái thanh toán của shipment
                $shipment->shipment_payment_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment->accountant_status = $result['status'] != HTTP_OK ? PENDING : SUCCESS;
                $shipment->save();
            }

            $shipment->import_approval = ACTIVE;
            $shipment->save();

            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $request->user()['user_id'],
                'Duyệt lô hàng khi bulk upload'
            );

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Duyệt lô hàng thành công',
                    'icon' => 'success',
                ]
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleDeleteApproveShipment(Request $request)
    {
        $this->validate($request, [
            "shipment_code" => 'required|exists:shipments,shipment_code',
        ], [
            'shipment_code.required' => 'Mã lô hàng không được để trống',
            'shipment_code.exists' => 'Mã lô hàng không tồn tại',
        ]);
        try {
            DB::beginTransaction();
            $request_data = $request->all();

            $shipment = Shipment::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('import_approval', INACTIVE)
                ->where('shipment_code', $request_data['shipment_code'])->first();

            $shipment->delete_flg = DELETED;
            $shipment->save();

            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $request->user()['user_id'],
                'Xóa lô hàng khi bulk upload'
            );

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Xóa lô hàng thành công',
                    'icon' => 'success',
                ]
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
