<?php

namespace App\Http\Controllers;

use App\Utils\Seapay;
use App\Models\Recharge;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use SePay\SePay\Datas\SePayWebhookData;
use SePay\SePay\Models\SePayTransaction;
use Illuminate\Validation\ValidationException;

class SepayController extends Controller
{
    /**
     * Xử lý xác thực token thanh toán qua Seapay
     * 
     * @return string
     */
    private function bearerToken(Request $request)
    {
        $header = $request->header('Authorization', '');

        $position = strrpos($header, 'Apikey ');

        if ($position !== false) {
            $header = substr($header, $position + 7);

            return str_contains($header, ',') ? (strstr($header, ',', true) ?: null) : $header;
        }

        return null;
    }

    /**
     * Xử lý webhook từ Seapay
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function handleSepayWebhook(Request $request)
    {
        DB::beginTransaction();
        try {
            // Khởi tạo seapay
            $sepay = new Seapay();

            // Lấy token từ header
            $token_header = $this->bearerToken($request);
            // Lấy token xác thực chính
            $token = $sepay->getWebhookToken();
            // Lấy tiền tố mã đơn hàng
            $match_pattern = $sepay->getMatchPattern();

            // Kiểm tra token
            throw_if(
                $token && $token_header !== $token,
                ValidationException::withMessages([
                    'message' => ['Unauthorized']
                ])
            );

            // Khởi tạo SePayWebhookData
            $sePayWebhookData = new SePayWebhookData(
                $request->integer('id'),
                $request->string('gateway')->value(),
                $request->string('transactionDate')->value(),
                $request->string('accountNumber')->value(),
                $request->string('subAccount')->value(),
                $request->string('code')->value(),
                $request->string('content')->value(),
                $request->string('transferType')->value(),
                $request->string('description')->value(),
                $request->integer('transferAmount'),
                $request->string('referenceCode')->value(),
                $request->integer('accumulated')
            );

            // Kiểm tra xem giao dịch đã tồn tại trong database chưa
            throw_if(
                SePayTransaction::query()->whereId($sePayWebhookData->id)->exists(),
                ValidationException::withMessages([
                    'message' => ['Giao dịch này đã thực hiện']
                ])
            );

            // Lưu giao dịch vào database
            $sepay_transaction = new SePayTransaction();
            $sepay_transaction->id = $sePayWebhookData->id;
            $sepay_transaction->gateway = $sePayWebhookData->gateway;
            $sepay_transaction->transactionDate = $sePayWebhookData->transactionDate;
            $sepay_transaction->accountNumber = $sePayWebhookData->accountNumber;
            $sepay_transaction->subAccount = $sePayWebhookData->subAccount;
            $sepay_transaction->code = $sePayWebhookData->code;
            $sepay_transaction->content = $sePayWebhookData->content;
            $sepay_transaction->transferType = $sePayWebhookData->transferType;
            $sepay_transaction->description = $sePayWebhookData->description;
            $sepay_transaction->transferAmount = $sePayWebhookData->transferAmount;
            $sepay_transaction->referenceCode = $sePayWebhookData->referenceCode;
            $sepay_transaction->save();

            // Biểu thức regex để khớp với mã đơn hàng
            $pattern = '/\b' . $match_pattern . '([a-zA-Z0-9-_])+/';
            preg_match($pattern, $sePayWebhookData->content, $matches);

            if (isset($matches[0])) {
                // Lấy bỏ phần pattern chỉ còn lại mã nạp tiền
                $recharge_id = Str::of($matches[0])->replaceFirst($match_pattern, '')->value();
                
                // Lấy ra lịch sử nạp tiền
                $recharge = Recharge::find($recharge_id);
                // Nếu recharge tồn tại và transferType là in
                if (isset($recharge) && $sePayWebhookData->transferType === 'in') {
                    $time_expired = Carbon::parse($recharge->created_at)->setTimezone('Asia/Ho_Chi_Minh')->addMinutes(TIME_EXPIRED_PAYMENT);
                    // Nếu thời gian hiện tại nhỏ hơn thời gian hết hạn thì cập nhật trạng thái nạp tiền
                    if (Carbon::now()->setTimezone('Asia/Ho_Chi_Minh')->lessThan($time_expired)) {
                        $recharge->status = 1;
                        $recharge->active_flg = ACTIVE;
                        $recharge->sepay_transaction_id = $sePayWebhookData->id;
                        $recharge->amount = $sePayWebhookData->transferAmount;
                        $recharge->save();

                        $data = [
                            "user_id" => $recharge->user_id,
                            "amount" => $sePayWebhookData->transferAmount,
                            "kind" => PLUS,
                            "content" => "Công tiền giao dịch nạp, mã: " . Str::of($matches[0])
                        ];
                        $res = actionWallet($data);
                    }
                }
            }

            DB::commit();
            return response()->noContent();
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    /**
     * Thêm giao dịch nạp tiền vào tài khoản
     * 
     * Thông tin gửi lên:
     * - amount: số tiền nạp
     * 
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function handleSepayRecharge(Request $request)
    {
        $this->validate($request, [
            'amount' => 'required|numeric|min:10000',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $user = $request->user();

            $amount = $request_data['amount'];
            $note = isset($request_data['note']) ? $request_data['note'] : '';
            $code_token = "KG" . time();

            $recharge = Recharge::create([
                'user_id' => isset($request_data['user_id']) ? $request_data['user_id'] : $user['user_id'],
                'amount' => $amount,
                'note' => $note,
                'type' => PAY_VNPAY,
                'admin_id' => auth()->id(),
                'active_flg' => INACTIVE,
                'code' => $code_token
            ]);

            $vnp_Url = route('sepay.recharge.show', ['recharge_id' => $recharge->recharge_id]);

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => "thành công",
                'url' => $vnp_Url,
                'recharge' => $recharge
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    /**
     * Hiển thị giao diện kiểm tra giao dịch nạp tiền
     * 
     * @param \Illuminate\Http\Request $request
     * @param int $recharge_id
     *
     * @return \Illuminate\Http\JsonResponse
     * @return \Illuminate\Contracts\View\View
     */
    public function displaySepayRecharge(Request $request, $recharge_id)
    {
        $is_api = request()->is('api/*');
        try {
            // Kiểm tra xem request có phải là api không
            $user = $request->user();

            // Khởi tạo Seapay
            $sepay = new Seapay();

            // Lấy danh sách tài khoản ngân hàng
            $banks = $sepay->getBankAccountList();
            // Nếu lấy danh sách tài khoản ngân hàng không thành công thì trả về thông báo lỗi
            if ($banks['status'] !== HTTP_OK) {
                return showMessageError();
            }

            // Lấy tài khoản ngân hàng đầu tiên
            $bank_account_detail = collect($banks['bankaccounts'])->first();
            // Nếu lấy thông tin tài khoản ngân hàng không thành công thì trả về thông báo lỗi
            if (empty($bank_account_detail)) {
                return showMessageError();
            }

            // Lấy thông tin giao dịch nạp tiền
            $recharge = Recharge::findOrFail($recharge_id)->toArray();
            $code = $sepay->getMatchPattern() . $recharge['recharge_id'];

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'recharge' => $recharge,
                    'bank_account_detail' => $bank_account_detail,
                    'code'=> $code,
                    'time_expired_payment' => TIME_EXPIRED_PAYMENT
                ], HTTP_OK);
            }

            return view('payments.sepay.recharge', compact('recharge', 'code', 'bank_account_detail'));
        } catch (\Throwable $th) {
            if ($is_api) {
                return showMessageError();
            }
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function checkSepayPayment(Request $request)
    {
        $this->validate($request, [
            'recharge_id' => 'required|integer',
        ]);
        try {
            $recharge_id = $request->input('recharge_id');
            $recharge = Recharge::findOrFail($recharge_id);

            return response()->json([
                'status' => HTTP_OK,
                'is_paid' => $recharge->status == 1 ? true : false, 
            ], HTTP_OK);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }
}
