<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Branch;
use App\Models\Wallet;
use App\Models\BagCode;
use App\Models\Package;
use App\Models\Service;
use App\Models\Shipment;
use App\Models\PackageSg;
use App\Models\BagCodeItem;
use App\Models\HistoryScan;
use Illuminate\Http\Request;
use App\Models\SurchargeGoods;
use App\Models\ShipmentTracktry;
use App\Models\WalletFluctuation;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Session;

class ScanController extends Controller
{
    public function displayImport()
    {
        return view('scan-system.scan.import');
    }

    public function displayImportHawbCode($hawb_code)
    {
        if (onCheckPositionAccounts([SETTING_FWD_ACCOUNT, SETTING_SALE_ACCOUNT], auth()->user()['position_id'])) {
            // Session::flash('error', 'Bạn không có quyền truy cập vào trang này');
            return viewWarning('Cảnh báo', 'Bạn không có quyền truy cập vào trang này');
        }

        $package = Package::where('package_code', $hawb_code)->isShow()->first();
        if (empty($package)) {
            return viewError404();
        }
        $shipment = Shipment::find($package['shipment_id']);

        if ($package['package_status'] != SHIPMENT_STATUS_CREATE) {
            // Session::flash('warning', 'Chỉ dành cho kiện hàng trong trạng thái tạo đơn (Create)');
            return viewWarning('Cảnh báo', 'Chỉ dành cho kiện hàng trong trạng thái tạo đơn (Create)');
        }

        $count_package = Package::isShow()->where('shipment_id', $shipment['shipment_id'])->count();
        $count_export_package = Package::isShow()
            ->where('shipment_id', $shipment['shipment_id'])
            ->where('package_status', SHIPMENT_STATUS_EXPORED)->count();
        $count_scan = ($count_export_package + 1) . '/' .  $count_package;

        return view('scan-system.scan.import-handle', compact('package', 'shipment', 'count_scan'));
    }

    public function displayTransitHN()
    {
        return view('scan-system.scan.transit-hn');
    }
    public function displayScanBagCode(Request $request)
    {
        $awb_code = $request->query('awb_code');
        $name_mawb = '';
        $sm_tracktry_id = '';
        if (isset($awb_code)) {
            $mawb = ShipmentTracktry::find($awb_code);
            $name_mawb = $mawb['awb_code'] . ' [' . $mawb['service'] . ']';
            $sm_tracktry_id = $awb_code;
        }
        return view('scan-system.scan.bag-code', compact('name_mawb', 'sm_tracktry_id'));
    }

    public function checkBarCode(Request $request)
    {
        $this->validate($request, [
            'sm_tracktry_id' => 'required',
            'bag_code' => 'required',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            if (empty($request_data['bag_code'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Vui lòng chọn bar code');
            }

            $code = BagCode::isShow()
                ->leftJoin('bag_code_items', 'bag_codes.bag_code_id', 'bag_code_items.bag_code_id')
                ->where('item_code', $request_data['bag_code'])->first();

            if ($code) {
                $sm = ShipmentTracktry::isShow()->where('sm_tracktry_id', $request_data['sm_tracktry_id'])->first();
                if (!empty($code->dest_code) && $sm->dest != $code->dest_code) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Mã DEST không trùng khớp');
                }
                if ($is_api) {
                    return response()->json([
                        'status' => HTTP_OK,
                        'data' => [
                            'mawb' => $request_data['sm_tracktry_id'],
                            'name' => $sm->service . " [$sm->awb_code]",
                            'bag_code' => $request_data['bag_code']
                        ],
                    ]);
                } else {
                    return response()->json([
                        'status' => HTTP_OK,
                        'route' => route('scan.export', ["bag_code" => $request_data['bag_code'], 'mawb' => $request_data['sm_tracktry_id']])
                    ]);
                }
            } else {
                return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy mã bar code');
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }


    public function displayExport(Request $request)
    {
        $bag_code = $request->query('bag_code');
        $mawb = $request->query('mawb');
        if (empty($bag_code) && empty($mawb)) {
            abort("404");
        }
        $sm = ShipmentTracktry::isShow()->where('sm_tracktry_id', $mawb)->first();
        return view('scan-system.scan.export', compact('bag_code', 'mawb', 'sm'));
    }



    public function displayReturn()
    {
        return view('scan-system.scan.return');
    }

    public function displayConfirmBill()
    {
        return view('scan-system.scan.confirm');
    }

    public function displayDocument()
    {
        return view('scan-system.scan.document');
    }


    public function showDetailPackage(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        $this->validate($request, [
            // 'code' => 'required',
            // 'sm_tracktry_id' => 'required',
            // 'bag_code' => 'required',
        ]);
        try {
            $package = Package::where('package_hawb_code', $request_data['code'])->isShow()->first();
            if (!$package) {
                return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy mã package');
            }

            if (isset($request_data['status'])) {
                $result = onCheckValidStatusPackage($package['package_status'], $request_data['status']);
                if ($result['status'] != HTTP_OK) {
                    return returnResponse($result['status'], $result['message'], null, 'warning');
                }
            }
            $shipment = Shipment::find($package['shipment_id']);
            if (!$shipment) {
                return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy shipment của kiện hàng', null, 'warning');
            }

            $user_scan = request()->user();
            if ($user_scan['branch_id'] != $shipment['shipment_branch_id']) {
                return returnResponse(HTTP_BAD_REQUEST, 'Tài khoản này thuộc chi nhánh ' . $user_scan['branch']['branch_name'] . ' không thể scan lô hàng thuộc chi nhánh '. $shipment['branch']['branch_name'], null, 'warning');
            }
            $count_package = Package::isShow()->where('shipment_id', $shipment['shipment_id'])->count();
            
            $count_import_package = Package::isShow()
                ->where('shipment_id', $shipment['shipment_id'])
                ->where('package_status', SHIPMENT_STATUS_IMPORTED)->count();

            if ($package['package_status'] == SHIPMENT_STATUS_CREATE) {
                return response()->json([
                    'status' => HTTP_OK,
                    'count_scan' => ($count_import_package + 1) . '/' .  $count_package,
                    'package' => $package,
                    'shipment_code' => $shipment['shipment_code'],
                    'route' => route('scan.import.hawb', ['hawb_code' => $package['package_code']])
                ]);
            }

            $user = User::find($shipment['user_id']);
            if ($user['is_export'] == INACTIVE) {
                return returnResponse(HTTP_BAD_REQUEST, "Tài khoản này đã bị chặn xuất hàng, vui lòng liên hệ lại với kế toán để mở");
            }

            $sm = ShipmentTracktry::find($request_data['sm_tracktry_id']);
            $bag_code = BagCodeItem::isShow()->where('item_code', $request_data['bag_code'])->first();

            if (!$sm) {
                return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy mã mawb');
            }

            $service_ids = json_decode($sm['service_ids'], true) ?? [];
            if(!in_array($shipment['shipment_service_id'], $service_ids)){
                return returnResponse(HTTP_BAD_REQUEST, 'Dịch vụ không hợp lệ');
            }
            if (!$bag_code) {
                return returnResponse(HTTP_BAD_REQUEST, 'Mã bag không tồn tại');
            }
            if ($shipment['accountant_status'] == 0) {
                return returnResponse(HTTP_BAD_REQUEST, 'Đơn hàng chưa được kế toán duyệt', null, 'warning');
            }

            $conver_package = getPakageServiceEpacket($shipment['service']['promotion_flg'], $package);

            $count_export_package = Package::isShow()
                ->where('shipment_id', $shipment['shipment_id'])
                ->where('package_status', SHIPMENT_STATUS_EXPORED)->count();
            $modal = [
                'package_code' => $package['package_hawb_code'],
                'count_scan' => ($count_export_package + 1) . '/' .  $count_package,
                'shipment_code' => $shipment['shipment_code'],
                'bag_code' => $request_data['bag_code'],
                'awb_code' => $sm['awb_code'],
                'weight' => $conver_package['package_weight'],
                'size' => $conver_package['package_length'] . ' x ' . $conver_package['package_width'] . ' x ' . $conver_package['package_height'],
            ];
            if ($is_api) {

                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $modal,
                ]);
            } else {
                if ($package['package_status'] == SHIPMENT_STATUS_CREATE) {
                    return response()->json([
                        'status' => HTTP_OK,
                        'shipment_code' => $shipment['shipment_code'],
                        'package' => $package,
                    ]);
                }
                $html = view('scan-system.ajax.modal-export', compact('modal'))->render();

                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return returnResponse(HTTP_BAD_REQUEST, $th->getMessage());
        }
    }

    //  tạo lịch sử scan và update status của package
    public function handleCreateHistory(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        $this->validate($request, [
            'status' => 'required',
            'code' => 'required',
            'sm_tracktry_id' => $request_data['status'] == SHIPMENT_STATUS_EXPORED ? 'required' : '',
        ]);
        try {
            $awb_code = '';
            if (!in_array($request_data['status'], [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_IMPORTED, SHIPMENT_STATUS_EXPORED, SHIPMENT_STATUS_RETURNED, SHIPMENT_STATUS_TRANSIT_HN])) {
                return [
                    'status' => HTTP_BAD_REQUEST,
                    'message' => [
                        "icon" => 'warning',
                        'title' => 'Thông báo',
                        'text' => 'Lỗi phương thức trạng thái cập nhật'
                    ]
                ];
            }

            DB::beginTransaction();

            $package = Package::where('package_hawb_code', $request_data['code'])->isShow()->first();
            $text = null;
            if ($package) {
                $shipment = Shipment::find($package['shipment_id']);
                $user_scan = request()->user();
                $branchs = [
                    'DN' => 'DA NANG, VN',
                    'HN' => 'HA NOI, VN',
                    'HCM' => 'HO CHI MINH, VN',
                ];
                $local_by_branch = $branchs[$user_scan['branch']['branch_name']] ?? 'HO CHI MINH, VN';

                $user = User::find($shipment['user_id']);

                $check_warining = false;
                $package_update = [];
                switch ($request_data['status']) {
                    case SHIPMENT_STATUS_IMPORTED:
                        if ($package->package_status == SHIPMENT_STATUS_IMPORTED) {
                            $check_warining = true;
                            $text = 'Kiện hàng này đã được scan imported rồi';
                            break;
                        }

                        if ($package->package_status == SHIPMENT_STATUS_RETURNED) {
                            $check_warining = true;
                            $text = 'Kiện hàng này đã bị trả về';
                            break;
                        }

                        if ($package->package_status == SHIPMENT_STATUS_EXPORED) {
                            $check_warining = true;
                            $text = 'Kiện hàng này đã được scan exported';
                            break;
                        }
                        if (isset($request_data['surchage_goods'])) {
                            $surchage_goods = $request_data['surchage_goods'];
                            $result_sg = onCreateSurchargeGoods($surchage_goods, $package['package_id']);
                            $total_price = $result_sg['total_amount'];
                        }
                        onCreatePackageTrackingStatus($package['package_code'], $local_by_branch, 'We Have Your Parcel');

                        if ($shipment['service']['promotion_flg'] == IS_EPACKET) {
                            if ($shipment['shipment_payment_status'] != SUCCESS) {
                                $check_warining = true;
                                $text = 'Kiện hàng này chưa được thanh toán';
                                break;
                            }
                            $this->updateEPacketWhenScanImport($request_data, $shipment);
                        } else {
                            // Cập nhật lại package và giá tiền
                            $package_length = $request_data['package_length'] ?? $package['package_length'];
                            $package_width = $request_data['package_width'] ?? $package['package_width'];
                            $package_height = $request_data['package_height'] ?? $package['package_height'];
                            $package_weight = $request_data['package_weight'] ?? $package['package_weight'];
                            if (
                                $package['package_length'] != (float) $package_length ||
                                $package['package_width'] != (float) $package_width ||
                                $package['package_height'] != (float) $package_height ||
                                $package['package_weight'] != (float) $package_weight
                            ) {

                                $package['package_length'] = $package_length;
                                $package['package_width'] = $package_width;
                                $package['package_height'] = $package_height;
                                $package['package_weight'] = $package_weight;


                                $new_package = onUpdatePackage($shipment, $package);
                                if (!$new_package) {
                                    DB::rollBack();
                                    return returnResponse(HTTP_BAD_REQUEST, 'Xảy ra lỗi khi cập nhật lại package');
                                }
                                unset($new_package['created_at']);

                                $new_package = $new_package->toArray();
                                Package::find($package['package_id'])->update($new_package);

                                $result_save_amount = onSaveAmountShipment($shipment['shipment_id']);
                                if ($result_save_amount['status'] === HTTP_INTERNAL_SERVER_ERROR) {
                                    return response()->json($result_save_amount);
                                }
                            }
                            onUpdateFinalAmountShipment($shipment['shipment_id']);
                        }

                        $sms_content = 'Đã nhận được đơn hàng mã HAWB ' . $package->package_hawb_code . ' của bạn';
                        sendSms($user->user_phone, $sms_content, $shipment->user_id, $package->package_hawb_code);

                        if(onCheckPositionAccount(SETTING_FWD_ACCOUNT, $shipment['user']['position'])){
                            Shipment::find($shipment['shipment_id'])->update([
                                'completed_date' => Carbon::now()->setTimezone('Asia/Ho_Chi_Minh'),
                            ]);
                        }

                        break;

                        // Export 
                    case SHIPMENT_STATUS_EXPORED:

                        if ($user['is_export'] == INACTIVE) {
                            return returnResponse(HTTP_BAD_REQUEST, "Tài khoản này đã bị chặn xuất hàng, vui lòng liên hệ lại với kế toán để mở");
                        }

                        if (empty($request_data['bag_code'])) {
                            return returnResponse(HTTP_BAD_REQUEST, "Không được để trống bag code");
                        }
                        if (!$request['package_image']) {
                            return returnResponse(HTTP_BAD_REQUEST, "Không được để trống hình ảnh");
                        }

                        $package_update['sm_tracktry_id'] = $request_data['sm_tracktry_id'];
                        $awb_code = $package_update['sm_tracktry_id'];
                        $package_update['bag_code'] = $request_data['bag_code'];
                        $package_update['package_image'] = '';
                        if ($package['package_status'] == SHIPMENT_STATUS_RETURNED) {
                            $check_warining = true;
                            $text = 'Kiện hàng đã được đã bị trả về';
                            break;
                        }

                        if ($package['package_status'] == SHIPMENT_STATUS_CREATE) {
                            $check_warining = true;
                            $text = 'Kiện hàng chưa được scan imported';
                            break;
                        }

                        if ($package['package_status'] == SHIPMENT_STATUS_EXPORED) {
                            $check_warining = true;
                            $text = 'Kiện hàng đã được xuất rồi';
                            break;
                        }


                        // Kiểm tra xem lô hàng của ePacket đã thanh toán đủ tiền chưa
                        if ($package['package_approve'] == PACKAGE_APPROVE_PAYMENT_FAIL && $shipment['service']['promotion_flg'] == IS_EPACKET) {
                            $wallet = Wallet::where('user_id', $shipment['user_id'])->first();
                            // Nếu ví của khách hàng không đủ tiền thì thông báo
                            if ($wallet['amount'] < 0) {
                                $check_warining = true;
                                $text = 'Kiện hàng này chưa được thanh toán đủ tiền nên không thể xuất, xin vui lòng báo cáo để khách nạp thêm tiền vào tài khoản';
                                break;
                            }

                            $package->package_approve = PACKAGE_APPROVE_PAID;
                            $package->save();
                        } else {
                            if (!$shipment) {
                                returnResponse(HTTP_BAD_REQUEST, "Không tìm thấy đơn hàng shipment");
                            } else {
                                if ($shipment['accountant_status'] != SUCCESS) {
                                    returnResponse(HTTP_BAD_REQUEST, "Đơn hàng phải được kế toán xác nhận mới có thể tiếp tục", null, 'warning');
                                }
                            }
                        }

                        $base64String = $request['package_image'];
                        $file_name = $package['package_code'];
                        $images = convertImagesBase64ToDirectory($base64String, PUBLIC_PATH_PACKAGE_IMAGE, $file_name);
                        $package_update['package_image'] = json_encode($images);

                        onCreatePackageTrackingStatus($package['package_code'], $local_by_branch, 'Export Scan & Leave our warehouse');

                        $time = Carbon::now()->addMinutes(30)->format('Y-m-d H:i:s');
                        onCreatePackageTrackingStatus($package['package_code'], $local_by_branch, 'Origin customs processing', $time, 0, INACTIVE);
                        break;

                    case SHIPMENT_STATUS_RETURNED:
                        if ($package['package_status'] == SHIPMENT_STATUS_EXPORED) {
                            $check_warining = true;
                            $text = 'Kiện hàng đã được xuất không thể trả hàng';
                            break;
                        } else if ($package['package_status'] == SHIPMENT_STATUS_RETURNED) {
                            $check_warining = true;
                            $text = 'Kiện hàng đã được trả về rồi';
                            break;
                        }

                        onCreatePackageTrackingStatus($package['package_code'], ' ', 'Return To Sender');

                        if ($shipment['service']['promotion_flg'] == IS_EPACKET) {
                            $this->updateEPacketWhenScanReturned($shipment, $package);
                        } else {
                            // code update lại giá tiền shipment
                            onSaveAmountShipment($shipment['shipment_id']);
                            onUpdateFinalAmountShipment($shipment['shipment_id']);
                        }
                        break;

                    case SHIPMENT_STATUS_TRANSIT_HN:
                        if ($package['package_status'] != SHIPMENT_STATUS_CREATE) {
                            onCreatePackageTrackingStatus($package['package_code'], $local_by_branch, 'IN TRANSIT NOI BAI AIRPORT');
                            $request_data['status'] = $package['package_status'];
                            break;
                        } else {
                            $check_warining = true;
                            $text = 'Kiện hàng đang ở trạng thái khởi tạo (Created)';
                            break;
                        }

                    default:
                        $check_warining = true;
                        $text = 'Trạng thái cập nhật không tồn tại';
                        break;
                }

                if ($check_warining) {
                    DB::rollBack();
                    return response()->json([
                        'status' => HTTP_NOT_FOUND,
                        'message' => [
                            "icon" => 'warning',
                            'title' => 'Thông báo',
                            'text' => $text
                        ]
                    ]);
                }

                HistoryScan::create([
                    'package_id' => $package->package_id,
                    'status' => $request_data['status'],
                    'scan_by' => $is_api ? $request->user()['user_id'] : auth()->id(),
                ]);

                $package_update['package_status'] = $request_data['status'];
                Package::find($package['package_id'])->update($package_update);
                updateStatusShipment($package['shipment_id']);
            } else {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => [
                        "icon" => 'error',
                        'title' => 'Thông báo',
                        'text' => 'Không tìm thấy đơn hàng của bạn'
                    ]
                ]);
            }

            DB::commit();
            if ($package['package_status'] == SHIPMENT_STATUS_EXPORED || $package['package_status'] == SHIPMENT_STATUS_IMPORTED) {
                Session::flash('success', 'Scan package thành công');
            }
            return response()->json([
                'status' => HTTP_OK,
                'awb_code' => $awb_code,
                'message' => [
                    "icon" => isset($text) ? 'warning' : 'success',
                    'title' => 'Thông báo',
                    'text' => isset($text) ? $text : 'Cập nhật đơn hàng thành công'
                ],
            ]);
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    /**
     * Cập nhật lại ePacket khi scan imported
     * @param mixed $data
     * @param mixed $shipment
     * @return void
     */
    private function updateEPacketWhenScanImport($data, $shipment)
    {
        $service = $shipment['service'];

        $convert_weight = $data['package_length'] * $data['package_width'] * $data['package_height'] / $service['service_volumetric_mass'];
        $data['package_converted_weight'] = formatWeightPackage($convert_weight, IS_EPACKET);

        $charged_weight = $data['package_weight'] < $data['package_converted_weight'] ? $data['package_converted_weight'] : $data['package_weight'];
        $data['package_charged_weight'] = $charged_weight;

        // Tính giá tiền sau khi scan
        $data['package_price'] = computedPricePackage($data['package_charged_weight'], $shipment);

        Package::where('package_hawb_code', $data['code'])->update([
            'package_length_actual' => $data['package_length'],
            'package_width_actual' => $data['package_width'],
            'package_height_actual' => $data['package_height'],
            'package_weight_actual' => $data['package_weight'],
            'package_converted_weight_actual' => $data['package_converted_weight'],
            'package_charged_weight_actual' => $data['package_charged_weight'],
            'package_price_actual' => $data['package_price'],
            'processing_staff_id' => request()->user()['user_id'],
        ]);

        // Lưu lại tổng tiền thực tế
        onSaveAmountShipmentActual($shipment['shipment_id']);

        // Thanh toán tiền sau khi scan imported
        chargeAndDeductFromWallet($shipment['shipment_id'], $data);

        $packages = $shipment['packages'];
        $invoices = $shipment['invoices'];

        $_packages = [];
        foreach ($packages as $package) {
            array_push($_packages, [
                'package_code' => $package['package_code'],
                'length' => $package['package_length_actual'],
                'width' => $package['package_width_actual'],
                'height' => $package['package_height_actual'],
                'weight' => $package['package_charged_weight_actual'],
                'package_weight' => $package['package_weight_actual'],
            ]);
        }

        $label_epacket = onCreateLabel($shipment, $_packages, $invoices, $service);
        if ($label_epacket['status'] == HTTP_OK) {
            $res_label = $label_epacket['res_label'];
            $OrderId = $label_epacket['OrderId'];
            if ($res_label['status'] == HTTP_OK) {
                Shipment::find($shipment['shipment_id'])->update([
                    'shipment_reference_code' => $OrderId,
                    'shipment_check_create_label' => 1,
                ]);
            }
        }

        $package_scan_imported = Package::where('package_hawb_code', $data['code'])->first();
        $shipment_scan_imported = Shipment::find($package_scan_imported['shipment_id']);

        // Cập nhật lại trạng thái của package
        $package_length_old = $package_scan_imported->package_length;
        $package_width_old = $package_scan_imported->package_width;
        $package_height_old = $package_scan_imported->package_height;
        $package_weight_old = $package_scan_imported->package_weight;
        $package_converted_weight_old = $package_scan_imported->package_converted_weight;
        $package_charged_weight_old = $package_scan_imported->package_charged_weight;
        $package_price_old = $package_scan_imported->package_price;

        $package_scan_imported->package_length = $package_scan_imported->package_length_actual;
        $package_scan_imported->package_length_actual = $package_length_old;
        $package_scan_imported->package_width = $package_scan_imported->package_width_actual;
        $package_scan_imported->package_width_actual = $package_width_old;
        $package_scan_imported->package_height = $package_scan_imported->package_height_actual;
        $package_scan_imported->package_height_actual = $package_height_old;
        $package_scan_imported->package_weight = $package_scan_imported->package_weight_actual;
        $package_scan_imported->package_weight_actual = $package_weight_old;
        $package_scan_imported->package_converted_weight = $package_scan_imported->package_converted_weight_actual;
        $package_scan_imported->package_converted_weight_actual = $package_converted_weight_old;
        $package_scan_imported->package_charged_weight = $package_scan_imported->package_charged_weight_actual;
        $package_scan_imported->package_charged_weight_actual = $package_charged_weight_old;
        $package_scan_imported->package_price = $package_scan_imported->package_price_actual;
        $package_scan_imported->package_price_actual = $package_price_old;

        $package_scan_imported->save();

        // Cập nhật lại tổng tiền thực tế        
        $shipment_final_amount_old = $shipment_scan_imported->shipment_final_amount;

        $shipment_scan_imported->shipment_final_amount = $shipment_scan_imported->shipment_total_amount_actual;
        $shipment_scan_imported->shipment_total_amount_actual = $shipment_final_amount_old;

        $shipment_scan_imported->save();
    }

    /**
     * Cập nhật lại ePacket khi scan returned
     * @param mixed $shipment
     * @param mixed $package
     * @return void
     */
    private function updateEPacketWhenScanReturned($shipment, $package)
    {
        // Trả lại tiền khi chưa scan imported
        $shipment_total_amount_return = $shipment['shipment_final_amount'];

        // Trả lại tiền khi huỷ
        $wallet = Wallet::where('user_id', $shipment['user_id'])->first();
        $wallet['amount'] += $shipment_total_amount_return;
        $wallet->save();

        // Lưu vào lịch sử ví
        WalletFluctuation::create([
            'wallet_id' => $wallet['wallet_id'],
            'secure_hash' => 'WF' . Carbon::now()->format('YmdHisv'),
            'user_id' => $shipment['user_id'],
            'amount' => $shipment_total_amount_return,
            'wallet_amount' => $wallet['amount'],
            'content' => 'Hoàn tiền khi xác nhận trả hàng cho package mã HAWB: ' . $package->package_hawb_code,
            'kind' => PLUS,
        ]);
    }

    public function showListHistoryScan(Request $request)
    {

        try {
            $request_data = $request->all();
            $is_api = $request->is('api/*');
            if (!in_array($request_data['status'], [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_IMPORTED, SHIPMENT_STATUS_EXPORED, SHIPMENT_STATUS_RETURNED])) {
                return [
                    'status' => HTTP_BAD_REQUEST,
                    'message' => [
                        "icon" => 'warning',
                        'title' => 'Thông báo',
                        'text' => 'Trạng thái cập nhật không đúng'
                    ]
                ];
            }

            $histories = HistoryScan::isShow()
                ->leftJoin('packages', 'history_scan.package_id', 'packages.package_id')
                ->leftJoin('users', 'history_scan.scan_by', 'users.user_id')
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('countries', 'shipments.receiver_country_id', 'countries.country_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->where('shipments.delete_flg', NO_DELETE)
                ->where('packages.package_status', $request_data['status']);

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                $histories = $histories->where(function ($query) use ($filters) {
                    $keywords = isset($filters['keywords']) ? $filters['keywords'] : '';
                    return $query->where('users.user_name', 'like', '%' . $keywords . '%')
                        ->orWhere('users.user_contact_name', 'like', '%' . $keywords . '%')
                        ->orWhere('packages.package_hawb_code', 'like', '%' . $keywords . '%')
                        ->orWhere('services.service_name', 'like', '%' . $keywords . '%')
                        ->orWhere('shipments.sender_company_name', 'like', '%' . $keywords . '%');
                });
                if (!empty($filters['service_ids'])) {
                    $filters['service_ids'] = is_array($filters['service_ids']) ? $filters['service_ids'] : [$filters['service_ids']];
                    $histories->whereIn("shipments.shipment_service_id", $filters['service_ids']);
                }
            }
            $check_over_time = isset($request_data['over_time']);
            if ($check_over_time) {
                $time = is_numeric($request_data['over_time']) ? (int) $request_data['over_time'] : 48;
                $timeLimit = Carbon::now()->subHours($time)->format('Y-m-d H:i:s');
                $histories = $histories->where('history_scan.updated_at', '<', $timeLimit);
            }

            $histories = $histories->orderBy('history_scan.created_at', 'desc')
                ->select(
                    'shipments.receiver_contact_name',
                    'shipments.sender_contact_name',
                    'packages.*',
                    'users.user_name',
                    'users.user_contact_name as scan_by_name',
                    'countries.country_name',
                    'services.service_name',
                    'scan_by',
                    'history_scan_id',
                    'history_scan.created_at as scan_at',
                    DB::raw("
                    CASE 
                        WHEN packages.package_status = 0 THEN 'created_bill'
                        WHEN packages.package_status = 1 THEN 'imported'
                        WHEN packages.package_status = 3   THEN 'returned'
                        WHEN packages.package_status = 2 THEN 'expored'
                        ELSE 'Không xác định'
                    END as status_label
                    ")
                );

            $histories = $histories->isPaginate(isset($request_data['paginate']) ? $request_data['paginate'] : null);
            if ($is_api) {
                return response()->json([
                    'status' => 200,
                    'data' => $histories->toArray()['data']
                ]);
            }


            switch ($request_data['status']) {
                case SHIPMENT_STATUS_IMPORTED:
                    if ($check_over_time) {
                        $html = view('scan-system.ajax.tables-over-time48', compact('histories'))->render();
                        # code...
                    } else {
                        $html = view('scan-system.ajax.tables-import', compact('histories'))->render();
                    }
                    break;

                case SHIPMENT_STATUS_EXPORED:
                    # code...
                    break;

                case SHIPMENT_STATUS_RETURNED:
                    # code...
                    break;
                default:
                    return response()->json([
                        'status' => HTTP_OK,
                        'html' => null
                    ]);
                    break;
            }

            return response()->json([
                'status' => HTTP_OK,
                'html' => $html
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function showSurchageGoods(Request $request)
    {

        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            $surchage_goods = SurchargeGoods::isShow()->get();

            if ($is_api) {
            } else {
            }
            return response()->json([
                "status" => HTTP_OK,
                "surchage_goods" => $surchage_goods
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleCancelStatus(Request $request)
    {
        $this->validate($request, [
            'history_id' => 'required',
        ]);
        $request_data = $request->all();

        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        DB::beginTransaction();
        try {
            $history = HistoryScan::where('history_scan_id', $request_data['history_id'])->isShow()->first();
            if (!isset($history)) {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => [
                        "icon" => 'error',
                        'title' => 'Thông báo',
                        'text' => 'Không tìm thấy đơn hàng của bạn'
                    ]
                ], HTTP_NOT_FOUND);
            }

            $package = Package::find($history->package_id);
            if ($package->package_status != SHIPMENT_STATUS_IMPORTED) {
                return returnResponse(HTTP_NOT_FOUND, 'Chỉ có thể hủy khi gói hàng đang ở trạng thái imported');
            }

            $shipment = Shipment::find($package->shipment_id);
            $pg_sgs = PackageSg::where('package_id', $package->package_id)->get()->toArray();
            $service = $shipment['service'];

            // Nếu là ePacket và đã scan imported thì mới huỷ
            if ($service['promotion_flg'] == IS_EPACKET && $package['package_status'] == SHIPMENT_STATUS_IMPORTED) {
                // Tính chênh lệch giá tiền khi huỷ
                $diff_package_price = $package['package_price'] - $package['package_price_actual'];
                $text_content = ', chênh lệch cân nặng khi scan imported trước đó là: ' . $package['package_charged_weight_actual'] - $package['package_charged_weight'] . 'kg';
                // Trả lại tiền khi huỷ
                if ($diff_package_price > 0) {
                    $wallet = Wallet::where('user_id', $shipment['user_id'])->first();
                    $wallet['amount'] += $diff_package_price;
                    $wallet->save();
                    WalletFluctuation::create([
                        'wallet_id' => $wallet['wallet_id'],
                        'secure_hash' => 'WF' . Carbon::now()->format('YmdHisv'),
                        'user_id' => $shipment['user_id'],
                        'amount' => $diff_package_price,
                        'wallet_amount' => $wallet['amount'],
                        'content' => 'Hoàn tiền chênh lệch giá tiền khi hủy scan import gói hàng mang mã HAWB: ' . $package->package_hawb_code . $text_content,
                        'kind' => PLUS,
                    ]);
                } else if ($diff_package_price < 0) {
                    // Trừ tiền khi huỷ
                    $wallet = Wallet::where('user_id', $shipment['user_id'])->first();
                    $wallet['amount'] -= abs($diff_package_price);
                    $wallet->save();

                    WalletFluctuation::create([
                        'wallet_id' => $wallet['wallet_id'],
                        'secure_hash' => 'WF' . Carbon::now()->format('YmdHisv'),
                        'user_id' => $shipment['user_id'],
                        'amount' => abs($diff_package_price),
                        'wallet_amount' => $wallet['amount'],
                        'content' => 'Trừ tiền chênh lệch giá tiền khi hủy scan import gói hàng mang mã HAWB: ' . $package->package_hawb_code . $text_content,
                        'kind' => MINUS,
                    ]);
                }

                // Trả lại trạng thái ban đầu khi huỷ
                $package->package_length = $package->package_length_actual;
                $package->package_length_actual = 0;
                $package->package_width = $package->package_width_actual;
                $package->package_width_actual = 0;
                $package->package_height = $package->package_height_actual;
                $package->package_height_actual = 0;
                $package->package_weight = $package->package_weight_actual;
                $package->package_weight_actual = 0;
                $package->package_converted_weight = $package->package_converted_weight_actual;
                $package->package_converted_weight_actual = 0;
                $package->package_charged_weight = $package->package_charged_weight_actual;
                $package->package_charged_weight_actual = 0;
                $package->package_price = $package->package_price_actual;
                $package->package_price_actual = 0;
                $package->processing_staff_id = null;
                $package->package_approve = PACKAGE_APPROVE_WAITING;
                $package->save();

                // Trả lại tiền phụ thu khi huỷ
                $total_price = 0;
                if (count($pg_sgs) > 0) {
                    foreach ($pg_sgs as $key => $value) {
                        $total_price += $value['price'] * $value['count'];
                    }

                    $data = [
                        'kind' => PLUS,
                        'user_id' => $shipment->user_id,
                        'amount' => $total_price,
                        'content' => 'Hoàn tiền chi phí phụ thu các hàng hóa của package mã HAWB: ' . $package->package_hawb_code,
                    ];

                    if ($total_price > 0) {
                        $result = actionWallet($data);
                        if ($result['status'] != HTTP_OK) {
                            return returnResponse($result['status'], $result['message']);
                        }
                    }
                }
                PackageSg::where('package_id', $package->package_id)->delete();

                $shipment->shipment_amount_service_actual = collect($shipment['packages'])->sum('package_price_actual');
                $shipment->shipment_final_amount = $shipment->shipment_total_amount_actual;
                $shipment->shipment_total_amount_actual = 0;
                $shipment->save();
            }

            $package->update(['package_status' => SHIPMENT_STATUS_CREATE]);
            $history->update(['delete_flg' => DELETED]);

            $result_update_sm = updateStatusShipment($package->shipment_id);
            if ($result_update_sm['status'] !== HTTP_OK) {
                return returnResponse($result_update_sm['status'], $result_update_sm['message']);
            }

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    "icon" => 'success',
                    'title' => 'Thông báo',
                    'text' => 'Hủy bỏ đơn hàng thành công'
                ]
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }


    public function displayListScan()
    {
        $services = Service::isActive(ACTIVE)->isDelete(NO_DELETE)->get();
        return view('scan-system.list.scan-import', compact('services'));
    }

    public function displayExpire24h()
    {
        $services = Service::isActive(ACTIVE)->isDelete(NO_DELETE)->get();
        return view('scan-system.list.expired24h', compact('services'));
    }

    public function displayBagCode()
    {
        $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->select('branch_id', 'branch_name')->get()->toArray();
        return view('scan-system.list.bag-code', compact('branchs'));
    }


    public function displayBagCodeUK()
    {
        $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->select('branch_id', 'branch_name')->get()->toArray();
        return view('scan-system.list.bog-code-uk', compact('branchs'));
    }

    public function showListBags(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $filters = isset($request_data['filters']) ? $request_data['filters'] : null;

            $bag_codes = BagCode::isShow()
                ->where('bag_code_kind', $request_data['bag_code_kind']);

            if (isset($filters)) {
                if (
                    (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                    && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
                ) {
                    $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                    $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                    $bag_codes = $bag_codes->whereBetween("bag_codes.created_at", [$start_date, $end_date]);
                }

                if (isset($filters['keywords']) && $filters['keywords'] != "") {
                    $keywords = convertSearchString($filters['keywords']);
                    $bag_codes = $bag_codes->where(function ($query) use ($keywords) {
                        $query->where('dest_code', 'like', $keywords);
                    });
                }
            }

            $bag_codes = $bag_codes->with(['branch'])
                ->orderBy('bag_codes.created_at', 'desc')
                ->isPaginate($request->paginate);

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'bag_codes' => $bag_codes,
                ]);
            }

            if ($request_data['bag_code_kind'] == 0) {
                $html = view('scan-system.ajax.table-bag-codes', compact('bag_codes'))->render();
            } else {
                $html = view('scan-system.ajax.table-bag-code-uks', compact('bag_codes'))->render();
            }
            return response()->json([
                'status' => HTTP_OK,
                'html' => $html,
            ]);
        } catch (\Throwable $th) {
            return returnResponse(HTTP_BAD_REQUEST, $th->getMessage());
        }
    }

    public function onPrintBagCode($bag_code_id)
    {
        try {
            $bag_code = BagCode::findOrFail($bag_code_id);
            if (!$bag_code) {
                return abort('404');
            }
            $bag_codes = BagCodeItem::isShow()->where('bag_code_id', $bag_code_id)->get();
            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();


            if ($bag_code['bag_code_kind'] == 0) {
                foreach ($bag_codes as $key => $value) {
                    $bag_codes[$key]['barcode'] = $generator->getBarcode($value['item_code'], $generator::TYPE_CODE_128, 2, 70);
                }
                return view('scan-system.ajax.print.bag-code', compact('bag_codes', 'bag_code'));
            } else {
                $bag_code['item_code'] = $bag_codes[0]['item_code'];
                $bag_code['barcode'] = $generator->getBarcode($bag_code['item_code'], $generator::TYPE_CODE_128, 2, 70);
                return view('scan-system.ajax.print.bag-code-uk', compact('bag_code'));
            }
        } catch (\Throwable $th) {
            return abort('404');
        }
    }

    public function handleCreateBagCode(Request $request)
    {
        $data_vali = [
            "bag_code_kind" => "required|numeric|min:0|max:1",
            "count_bag" => "required|numeric|min:1|max:100",
            "branch_id" => "required|numeric",
        ];

        if ($request->bag_code_kind == 1) {
            $data_vali['total_package'] = "required|numeric|min:1|max:100000";
            $data_vali['total_weight'] = "required|numeric|min:1|max:1000000";
        }
        $this->validate($request, $data_vali);
        try {
            DB::beginTransaction();
            $bag_code = BagCode::create([
                'bag_code_kind' => $request->bag_code_kind,
                'dest_code' => $request->dest_code,
                'branch_id' => $request->branch_id,
                'count_bag' => $request->count_bag,
                'total_package' => isset($request->total_package) ? $request->total_package : 0,
                'total_weight' => isset($request->total_weight) ? $request->total_weight : 0,
                'bag_code_type' => $request->bag_code_type,
            ]);

            $branch = Branch::find($request->branch_id);
            if (!$branch) {
                return returnResponse(HTTP_BAD_REQUEST, 'Chi nhánh không tồn tại');
            }

            if ($request->bag_code_kind == 0) { // cho loại bth
                for ($i = 0; $i < $request->count_bag; $i++) {
                    $a = BagCodeItem::orderBy('bag_code_item_id', 'desc')->first();
                    $code = $branch->branch_name . '' . (100000000 + (isset($a->bag_code_item_id) ? $a->bag_code_item_id : 0));
                    BagCodeItem::create([
                        'bag_code_id' => $bag_code->bag_code_id,
                        "item_code" => $code
                    ]);
                }
            } else if ($request->bag_code_kind == 1) {
                $a = BagCodeItem::orderBy('bag_code_item_id', 'desc')->first();
                $code = $branch->branch_name . '' . (100000000 + (isset($a->bag_code_item_id) ? $a->bag_code_item_id : 0));
                BagCodeItem::create([
                    'bag_code_id' => $bag_code->bag_code_id,
                    "item_code" => $code
                ]);
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Khởi tạo bag code thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
