<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\SaleKpi;
use App\Models\Service;
use App\Models\Position;
use App\Models\RangeKpi;
use App\Models\SaleTeam;
use App\Models\Shipment;
use App\Models\SaleMember;
use App\Exports\ExportFile;
use App\Exports\SaleExport;
use App\Models\SaleLinkFwd;
use Illuminate\Http\Request;
use App\Exports\SaleTeamExport;
use App\Models\FwdCostByService;
use Illuminate\Support\Facades\DB;
use App\Models\HistoryEditShipment;
use function Laravel\Prompts\select;
use Maatwebsite\Excel\Facades\Excel;

use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;

class SaleManagerController extends Controller
{
    private function getPositionSale($kind = SETTING_SALE_ACCOUNT)
    {
        $path = 'public/config/system-settings.json';
        $system_settings = json_decode(Storage::get($path), true);
        $sale = collect($system_settings)->firstWhere('kind', $kind);

        return $sale;
    }

    private function onCheckSaleManage($kind = SETTING_SALE_ACCOUNT)
    {
        $path = 'public/config/system-settings.json';
        $system_settings = json_decode(Storage::get($path), true);
        $sale = collect($system_settings)->firstWhere('kind', $kind);

        return auth()->user()['position_id'] == $sale['position_id'];
    }


    public function displayIndex(Request $request)
    {
        try {
            $is_api = $request->is('api/*');
            $is_view_all_shipment = isGetAllData();
            $data_table = Shipment::isShow();
            if (!$is_view_all_shipment) {
                $data_table = $data_table->where('shipments.user_id', request()->user()->user_id);
            } else {
                // $data_table = $data_table->leftJoin('users', 'shipments.user_id', 'users.user_id')
                // ->where('users.position_id', $this->getPositionSale()['position_id']);
                $user_ids = User::where('position_id', $this->getPositionSale()['position_id'])->pluck('user_id')->toArray();
                $data_table = $data_table->whereIn('shipments.user_id', $user_ids);
            }
            $data_table = $data_table->select(
                DB::raw('DATE_FORMAT(shipments.created_at, "%m-%Y") as month'),
                DB::raw('SUM(shipments.shipment_amount_profit) as total_profit'),
                DB::raw('SUM(shipments.shipment_amount_total_customer) as total_price'),
            )
                ->whereBetween('shipments.created_at', [Carbon::now()->subMonths(6)->startOfMonth(), Carbon::now()->endOfMonth()])
                ->groupBy('month')
                ->orderByRaw('STR_TO_DATE(CONCAT("01-", month), "%d-%m-%Y") ASC')
                ->get();

            $month = [];
            $data1 = [];
            $data2 = [];
            foreach ($data_table as $key => $value) {
                array_push($month, $value['month']);
                array_push($data1, $value['total_price']);
                array_push($data2, $value['total_profit']);
            }
            $charts = [
                'month' => $month,
                'price' => $data1,
                'profit' => $data2,
            ];

            $sales = SaleMember::leftJoin('users', 'sales_members.user_id', 'users.user_id')
                ->where('users.delete_flg', NO_DELETE)->where('users.active_flg', ACTIVE)
                ->select(
                    DB::raw('COUNT(CASE WHEN sales_members.delete_flg = 1 AND sales_members.active_flg = 1 THEN 1 END) as total_sale_in_team'),
                    DB::raw('COUNT(CASE WHEN member_kind = 1 AND sales_members.delete_flg = 1 AND sales_members.active_flg = 1 THEN 1 END) as total_leader'),
                    DB::raw('COUNT(CASE WHEN sales_members.delete_flg = 1 AND sales_members.active_flg = 0 THEN 1 END) as total_sale_pending'),
                )
                ->first();
            $sale = $this->getPositionSale();
            $users = User::isShow()->where('position_id', $sale['position_id'])->selectRaw('COUNT(*) as total_sale')->first();
            $fwd_costs = FwdCostByService::isShow()
                ->select(
                    DB::raw('COUNT(*) as total_fwd_cost'),
                    DB::raw('COUNT(*) - SUM(IF(leader_id IS NULL OR leader_id = "", 1, 0)) as total_links')
                )->first();

            $sale_link_fwd = SaleLinkFwd::select(
                DB::raw('COUNT(CASE WHEN delete_flg = 1 AND active_flg = 1 THEN 1 END) as total_sale_link_fwd'),
                DB::raw('COUNT(CASE WHEN delete_flg = 1 AND active_flg = 0 THEN 1 END) as total_pending_sale_link_fwd')
            )->first();
            $dashboard = [
                "manager_leader" => $sales->total_leader . ' sale leader',
                "manager_sales" => $users->total_sale . ' nhân viên sale',
                "confirm_sale" => $sales->total_sale_pending . ' Member cần duyệt',
                "cost_fwd" => $fwd_costs->total_fwd_cost . ' bảng giá',
                "manager_leader_link_fwd" => $fwd_costs->total_links . ' Liên kết',
                "manager_sale_link_fwd" => $sale_link_fwd->total_sale_link_fwd . ' FWD hỗ trợ',
                "confirm_fwd" => $sale_link_fwd->total_pending_sale_link_fwd . ' Liên kết chờ duyệt',
            ];
            onCheckPositionAccount(SETTING_ADMIN_ACCOUNT, auth()->user()['position_id']);
            $path = 'public/config/system-menu-sale.json';
            $menus = json_decode(Storage::get($path), true) ?? [];

            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "charts" => $charts,
                    "dashboard" => $dashboard,
                    "sale" => $sale,
                    "menus" => $menus,
                ]);
            }

            return view('manage.sale.index', compact('charts', 'dashboard', 'sale', 'menus'));
        } catch (\Throwable $th) {
            return viewError404();
        }
    }

    public function displayLeader(Request $request)
    {
        $user = request()->user();
        $is_api = $request->is('api/*');
        $is_leader = $user['leader'];
        $leader_id = $is_leader ? $user['user_id'] : null;
        $is_sale = onCheckPositionAccount(SETTING_SALE_ACCOUNT, $user['position_id']);

        if ($is_api) {
            return response()->json([
                "status" => HTTP_OK,
                "is_sale" => $is_sale,
                "is_leader" => $is_leader,
                "leader_id" => $leader_id,
            ]);
        }

        return view('manage.sale.leader.index', compact('leader_id', 'is_leader', 'is_sale'));
    }
    // lấy danh sách salers
    public function showListSaleLeader(Request $request)
    {
        try {
            $request_data = [
                'filters' => $request->query('filters'),
                'paginate' => $request->query('paginate'),
                'team_id' => $request->query('team_id'),
                // 'leader_id' => $request->query('leader_id'),
            ];
            // $request_data = $request->all();
            $is_api = $request->is('api/*');
            $user = request()->user();
            $is_sale = onCheckPositionAccount(SETTING_SALE_ACCOUNT, $user['position_id']);
            $check_edit = $is_sale;

            $teams = SaleTeam::isShow()
                ->leftJoin('sales_members', 'sales_teams.sale_team_id', 'sales_members.sale_team_id')
                ->leftJoin('users', 'sales_members.user_id', 'users.user_id')
                ->where('sales_members.member_kind', SALE_LEADER);

            if (isset($request_data['only_id']) && isset($request_data['team_id'])) {
                $teams = $teams->where('sales_teams.sale_team_id', '!=', $request_data['team_id'])
                    ->select('sales_teams.sale_team_id', 'users.user_contact_name', 'users.user_code')
                    ->groupBy('sales_teams.sale_team_id')->get()->toArray();

                $convert_data = [];
                foreach ($teams as $key => $value) {
                    $convert_data[$value['sale_team_id']] = $value['user_contact_name'] . '-' . $value['user_code'];
                }
                return response()->json([
                    "status" => HTTP_OK,
                    "teams" => $convert_data,
                ]);
            }

            if ($is_sale) {
                $teams = $teams->where('sales_teams.leader_id', $user['user_id']);
            }

            $filters = $request_data['filters'] ?? null;
            if (isset($filters)) {
                $teams = $teams->where('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%');
                if (
                    (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                    && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
                ) {
                    $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                    $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                    $teams->where("sales_teams.created_at", '<=', $start_date);
                }
            }
            // $teams = $teams->with(['members']);
            $teams = $teams->select(
                'users.user_contact_name',
                'users.user_code',
                'sales_teams.sale_team_id',
                'sales_teams.created_at',
            );
            $teams = $teams->isPaginate(isset($request_data['paginate']) ? $request_data['paginate'] : null);

            if ($is_api) {
                $data = $teams->toArray()['data'];
                foreach ($data as $key => $value) {
                    $data[$key]['members'] = getTeamMember($value['sale_team_id'], $filters);
                }
                return response()->json([
                    "status" => HTTP_OK,
                    "teams" => $data,
                    "check_edit" => $check_edit,
                    "filters" => $filters,
                ]);
            } else {
                $html = view('manage.sale.leader.ajax.table-teams', compact('teams', 'check_edit', 'filters'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    // Xuất báo cáo team
    public function onExportTeam(Request $request)
    {
        try {
            $request_data = $request->all();
            $sale_team_id = $request_data['team_id'];
            $monthDate = $request_data['monthDate'];

            $filter_date = $this->getFilterDate($request->query('monthDate'));
            $filters = isset($request_data['filters']) ? (is_array($request_data['filters']) ? $request_data['filters'] : json_decode($request_data['filters'], true)) : [];
            $filters['date_range'] = $filter_date;



            $data = [];
            $total_team = [
                'team_profit' => 0,
                'team_price' => 0,
            ];

            $leader = null;
            $commission_leader_team = 0;
            $sale_team = SaleMember::where('sale_team_id', $sale_team_id)->isShow()
                ->leftJoin('users', 'sales_members.user_id', 'users.user_id')
                ->where('users.delete_flg', NO_DELETE)
                ->select('member_kind', 'user_code', 'user_contact_name', 'users.user_id', 'user_kpi_id', 'sales_members.created_at')
                ->get()->toArray();
            $sale_team = $sale_team ?? [];
            foreach ($sale_team as $key => $value) {
                if ($value['member_kind'] == SALE_LEADER) {
                    $leader = $value;
                    unset($sale_team[$key]);
                    continue;
                }
                $createdAt = Carbon::parse($value['created_at']);
                $isInRange = Carbon::parse($createdAt)->lessThanOrEqualTo($filter_date['end_date']);
                // Check xem thời gian member được add vào team
                if (!$isInRange) {
                    $sale_team[$key]['check_null'] = 'Sale được thêm vào ngày ' . Carbon::parse($value['created_at'])->format('Y-m-d') . ', nên không có thống kê';
                    $sale_team[$key]['salary'] = 0;
                    $sale_team[$key]['total_price'] = 0;
                    $sale_team[$key]['total_profit'] = 0;
                    $sale_team[$key]['ratio_commission'] = 0;
                    $sale_team[$key]['price_commission'] = 0;
                    continue;
                }
                $convert = getDataSmPackageByDate($value['user_id'], $filters);

                // Lấy KPI của member để tính hoa hồng leader dc hưởng từ team
                $member_kpi = onGetKpiSale($value['user_kpi_id'], $monthDate);
                $member_range_kpi = getRangeSaleKpi($member_kpi['range_kpis'] ?? [], $convert['total_profit']);
                if ($member_range_kpi) {
                    // Nếu KPI của member đạt chỉ tiêu thì leader sẽ được hưởng FWD Weight cost
                    $member_commission = $convert['total_profit'] - $member_range_kpi['salary'];
                    $member_commission = $member_commission > 0 ? $member_commission : 0;
                    if ($member_range_kpi['leader_have_commission'] == 1) {
                        $commission_leader_team += ($member_commission * 0.07);
                    }
                    $sale_team[$key]['salary'] = $member_range_kpi['salary'];
                    $sale_team[$key]['ratio_commission'] = $member_range_kpi['ratio_commission'];
                    $sale_team[$key]['range_kpi_name'] = $member_range_kpi['range_kpi_name'];
                    $sale_team[$key]['price_commission'] = $member_commission * $member_range_kpi['ratio_commission'] / 100;
                } else {
                    $sale_team[$key]['salary'] = 0;
                    $sale_team[$key]['ratio_commission'] = 0;
                    $sale_team[$key]['range_kpi_name'] = isset($value['user_kpi_id']) ? 'Sale chưa đạt đủ KPI' : 'Sale chưa gắn kpi';
                    $sale_team[$key]['price_commission'] = 0;
                }

                $sale_team[$key]['total_price'] = $convert['total_price']; // Lợi nhuận member
                $sale_team[$key]['total_profit'] = $convert['total_profit']; // Lợi nhuận member


                $total_team['team_profit'] +=  $convert['total_profit'];
            }
            $total_team['commission_leader_team'] = $commission_leader_team;
            $data = [
                'date' => $monthDate,
                'leader' => $leader,
                'teams' => $sale_team,
                'total_team' => $total_team
            ];
            $export_name = 'Báo cáo team ' . $request->query('monthDate') . '-' . $leader['user_contact_name'] . '.xlsx';

            return Excel::download(new SaleTeamExport($data), $export_name);
        } catch (\Throwable $th) {
            return viewError404();
        }
    }

    // lấy danh sách tài khoản user
    public function showListUsers(Request $request)
    {

        try {
            $request_data = $request->all();
            $is_api = $request->is('api/*');

            $user_not_in = SaleMember::isShow()->get()->toArray();
            $user_not_in = collect($user_not_in)->pluck('user_id')->toArray();
            $sale = $this->getPositionSale();
            $users = User::isShow()
                ->whereNotIn('users.user_id', array_merge($user_not_in, isset($request_data['user_not_in']) ? $request_data['user_not_in'] : []))
                // ->leftJoin('positions', 'users.position_id', 'positions.position_id')
                // ->where('positions.delete_flg', NO_DELETE)
                // ->where('position_name', 'like', '%' . $request_data['position_name'] . '%')
                ->where('position_id', $sale['position_id'])
                ->isFilters(isset($request_data['filters']) ? $request_data['filters'] : null)
                ->select('user_contact_name', 'user_code', 'user_name', 'user_id')
                ->isPaginate(isset($request_data['paginate']) ? $request_data['paginate'] : null);

            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $users->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.leader.ajax.table-users', compact('users'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function addMiltipleLeader(Request $request)
    {
        $request_data = $request->all();
        $is_api = $request->is('api/*');
        try {
            if (!isGetAllData()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn không đủ quyền hạn để thực hiện chức năng này', null, 'warning');
            }
            if (isset($request_data['arr_user_id']) && count($request_data['arr_user_id']) > 0) {
                DB::beginTransaction();
                foreach ($request_data['arr_user_id'] as $key => $value) {
                    $team = SaleTeam::create([
                        'sale_team_code' => initializationCode('sales_teams', 'sale_team_code', ''),
                        'leader_id' => $value,
                    ]);

                    SaleMember::create([
                        'sale_team_id' => $team->sale_team_id,
                        'user_id' => $value,
                        'member_kind' => SALE_LEADER
                    ]);
                }
                DB::commit();
                return returnResponse(HTTP_OK, 'Tạo nhóm sale thành công', null, 'success');
            } else {
                return returnResponse(HTTP_NO_CONTENT, 'Hãy chọn sale làm trưởng nhóm', null, 'warning');
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function addMiltipleMember(Request $request)
    {
        $this->validate($request, [
            'arr_user_id' => 'required',
            // 'leader_id' => 'required',
            // 'sale_team_id' => 'required',
        ]);
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            if (isset($request_data['arr_user_id']) && count($request_data['arr_user_id']) > 0) {
                DB::beginTransaction();
                $team = SaleTeam::isShow();
                if (isset($request_data['leader_id'])) {
                    $team = $team->where('leader_id', $request_data['leader_id']);
                } else if (isset($request_data['sale_team_id'])) {
                    $team = $team->where('sale_team_id', $request_data['sale_team_id']);
                } else {
                    return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy team leader cần cập nhật');
                }
                $team =  $team->first();
                if (!$team) {
                    return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy team leader cần cập nhật');
                }
                if (SaleMember::isShow()->whereIn('user_id', $request_data['arr_user_id'])->where('delete_flg', NO_DELETE)->exists()) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Trong danh sách member tải lên có tài khoản đã trong nhóm khác');
                }
                foreach ($request_data['arr_user_id'] as $key => $value) {
                    SaleMember::create([
                        'sale_team_id' => $team->sale_team_id,
                        'user_id' => $value,
                        'member_kind' => SALE_MEMBER,
                        'active_flg' => isGetAllData() ? ACTIVE : INACTIVE
                    ]);
                }
                DB::commit();
                return returnResponse(HTTP_OK, isGetAllData() ? 'Thêm tài khoản vào nhóm thành công' : 'Đã gửi yêu cầu thành công', null, 'success');
            } else {
                return returnResponse(HTTP_NO_CONTENT, 'Hãy chọn sale mà bạn muốn thêm vào nhóm', null, 'warning');
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function updateMemberOnTeam(Request $request)
    {
        $this->validate($request, [
            'team_id' => 'required',
            'member_id' => 'required',
            'kind' => 'required|min:0|max:1',
        ]);
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            if (!isGetAllData()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn không đủ quyền hạn để thực hiện chức năng này', null, 'warning');
            }

            DB::beginTransaction();
            $sale_member = SaleMember::where('sale_team_id', $request_data['team_id'])->where('sale_member_id', $request_data['member_id'])->first();
            if ($sale_member) {
                if ($request_data['kind'] == 0) {
                    $sale_member->delete();
                } else if ($request_data['kind'] == 1) {
                    if ($sale_member->member_kind == SALE_LEADER) {
                        return returnResponse(HTTP_BAD_REQUEST, 'Đây đã là leader rồi', null, 'warning');
                    }
                    SaleMember::where('sale_team_id', $request_data['team_id'])
                        ->where('sale_member_id', '!=', $request_data['member_id'])
                        ->where('member_kind',  SALE_LEADER)
                        ->update(['member_kind' => SALE_MEMBER]);
                    $sale_member->update(['member_kind' => SALE_LEADER]);
                    SaleTeam::find($sale_member->sale_team_id)->update(['leader_id' => $sale_member->user_id]);
                }
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy thành viên cần xóa');
            }

            DB::commit();
            return returnResponse(HTTP_OK, 'Thao tác thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onDeleteTeam(Request $request)
    {
        $this->validate($request, [
            'team_id' => 'required',
        ]);
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            DB::beginTransaction();
            $team = SaleTeam::findOrFail($request_data['team_id']);
            if (!$team) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy team leader cần xóa');
            }
            $team->delete();
            DB::commit();
            return returnResponse(HTTP_OK, 'Đã xóa team thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function transferToTeam(Request $request)
    {
        $this->validate($request, [
            'member_id' => 'required',
            'team_id' => 'required',
        ]);
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            DB::beginTransaction();
            $sale = SaleMember::findOrFail($request_data['member_id']);
            if (!$sale) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy member cần chuyển');
            }
            if ($sale->member_kind == SALE_LEADER) {
                return returnResponse(HTTP_NOT_FOUND, 'Không thể chuyển leader đi');
            }
            $sale->update([
                'sale_team_id' => $request_data['team_id'],
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Đã di chuyển thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    // SALE DETAIL

    private function getFilterDate($monthDate)
    {
        try {
            $date = Carbon::parse($monthDate); // Lấy đối tượng Carbon từ chuỗi tháng truyền vào
            return [
                'start_date' => $date->startOfMonth()->format('Y-m-d'), // Ngày bắt đầu tháng
                'end_date' => $date->endOfMonth()->format('Y-m-d'), // Ngày kết thúc tháng
            ];
        } catch (\Throwable $th) {
            // Nếu có lỗi xảy ra, lấy tháng hiện tại làm mặc định
            return [
                'start_date' => Carbon::now()->startOfMonth()->toDateString(),
                'end_date' => Carbon::now()->endOfMonth()->toDateString(),
            ];
        }
    }

    private function getInfoSale($user)
    {
        return [
            'user_id' => $user['user_id'],
            'user_contact_name' => $user['user_contact_name'],
            'user_code' => $user['user_code'],
            'user_name' => $user['user_name'],
            'user_phone' => $user['user_phone'],
            'is_leader' => $user['is_leader'] ?? false,
            'created_at' => Carbon::parse($user['created_at'])->format('Y-m-d'),
        ];
    }

    private function getStatisticSale($user, $filters, $kpi)
    {
        try {
            $sale_statistic = [
                'total_shipment' => 0,
                'total_charged_weight' => 0,
                'total_amount_customer' => 0,
                'total_amount_profit' => 0,
                'range_kpi_id' => null,
                'range_kpi_name' => 'Xảy ra lỗi khi tính thống kê',
                'salary' => 0,
                'ratio_commission' => 0,
                'commission_amount' => 0,
                'cost_fwd_weight' => 0,
                'cost_fwd_amount' => 0,
            ];
            $shipments_of_sale = getDataSmPackageByDate($user['user_id'], $filters);
            // Số đơn shipment 
            $sale_statistic['total_shipment'] = $shipments_of_sale['total_shipment'];
            // Tổng số charged weight của các shipment 
            $sale_statistic['total_charged_weight'] = $shipments_of_sale['total_charged_weight'];
            // Tổng doanh thu từ các shipment 
            $sale_statistic['total_amount_customer'] = $shipments_of_sale['total_amount_customer'];
            // Tổng lợi nhuận từ các shipment 
            $sale_statistic['total_amount_profit'] = $shipments_of_sale['total_profit'];

            if (!$kpi) {
                $sale_kpi = null;
                $sale_statistic['range_kpi_id'] = null;
                $sale_statistic['range_kpi_name'] = "Chưa gắn KPI";
            } else {
                $sale_kpi['kpi'] = $kpi;
                // Tiền thu khách của sale
                $revenue = $sale_statistic['total_amount_profit'];
                $my_range_kpi = getRangeSaleKpi($kpi['range_kpis'] ?? null, $revenue);
                if (!$my_range_kpi) {
                    $sale_statistic['range_kpi_id'] = null;
                    $sale_statistic['range_kpi_name'] = "Sale chưa đủ điều kiện đạt KPI";
                } else {
                    // Mã mức KPI
                    $sale_statistic['range_kpi_id'] = $my_range_kpi['range_kpi_id'];
                    // Tên mức KPI
                    $sale_statistic['range_kpi_name'] = $my_range_kpi['range_kpi_name'];
                    // Lương cơ bản
                    $sale_statistic['salary'] = $my_range_kpi['salary'];
                    // Tỉ lệ hoa hồng
                    $sale_statistic['ratio_commission'] = $my_range_kpi['ratio_commission'];
                }
            }
            // Hoa hồng sale
            if (isset($sale_statistic['range_kpi_id'])) {
                $sale_statistic['commission_amount'] = getCommistionSale($sale_statistic['total_amount_profit'], $my_range_kpi);
            } else {
                $sale_statistic['commission_amount'] = 0;
            }
            // Lợi nhuận từ số kí fwd quản lý
            $weight_fwds = getTotalWeightCompaniesForSale2($user['user_id'], $filters);
            $sale_statistic['weight_fwd'] = $weight_fwds['total_weight']; // Số kí fwd đã tạo
            $sale_statistic['amount_fwd'] = $user['is_leader'] ? $weight_fwds['total_leader_cost'] : $weight_fwds['total_price']; // Số tiền hưởng fwd weight đã tạo

            $sale_statistic['final_salary_sale'] = $sale_statistic['salary'] + $sale_statistic['commission_amount'] + $sale_statistic['amount_fwd'];

            return $sale_statistic;
        } catch (\Throwable $th) {
            return [
                'total_shipment' => 0,
                'total_charged_weight' => 0,
                'total_amount_customer' => 0,
                'total_amount_profit' => 0,
                'range_kpi_id' => null,
                'range_kpi_name' => 'Xảy ra lỗi khi tính thống kê',
                'salary' => 0,
                'ratio_commission' => 0,
                'commission_amount' => 0,
                'cost_fwd_weight' => 0,
                'cost_fwd_amount' => 0,
                'final_salary_sale' => 0,
            ];
        }
    }



    public function displayDetail(Request $request)
    {
        try {
            $is_api = $request->is('api/*');
            $user_request = request()->user();
            $is_get_all = isGetAllData();

            if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user_request['position_id'])) {
                return $is_api ? returnResponse(HTTP_NOT_FOUND, 'Tài khoản FWD không thể truy cập') : viewError404();
            }
            if (empty($request->query('sale_code')) || !isGetAllData()) {
                $sale_code = $user_request['user_code'];
            } else {
                $sale_code = $request->query('sale_code');
            }

            $user = User::where('user_code', $sale_code)->first();
            if (!$user) {
                return $is_api ? returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy sale') : viewError404();
            }
            if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user['position_id'])) {
                if ($is_api) {
                    return returnResponse(HTTP_NOT_FOUND, 'Chức năng này không dành cho tài khoản FWD');
                }
                Session::flash('warning', 'Chức năng này không dành cho tài khoản FWD');
                return viewError404();
            }

            $sale = SaleMember::where('user_id', $user['user_id'])->isShow()->first();

            $is_leader = false;
            if (isset($sale) && $sale['member_kind'] == SALE_LEADER) {
                $is_leader = true;
            }
            $user['is_leader'] = $is_leader;

            // Thông tin tài khoản sale
            $info_sale = $this->getInfoSale($user);

            $filter_date = $this->getFilterDate($request->query('monthDate'));
            $filters['date_range'] = $filter_date;
            $filters['shipment_payment_status'] = 1;
            $monthDate = $request->query('monthDate') ?? Carbon::now()->format('Y-m');

            $kpi = onGetKpiSale($user['user_kpi_id'], $monthDate);

            // Thống kê tài khoản sale
            $sale_statistic = $this->getStatisticSale($user, $filters, $kpi);
            // Lấy thông tin range kpi của sale
            $info_sale['range_kpi'] = getRangeSaleKpi($kpi['range_kpis'] ?? null, $sale_statistic['total_amount_profit']);


            $sale_team = [];
            $total_team = [
                'team_profit' => 0,
                'team_fwd_weight' => 0,
                'team_fwd_cost' => 0,
            ];

            $hoa_hong_leader_team = 0;
            $commission_leader_team = 0;
            $fw_weight_cost_leader_team = 0;
            if ($is_leader) {
                $sale_team = SaleMember::where('sale_team_id', $sale->sale_team_id)->isShow()
                    ->where('sales_members.user_id', '!=', $user->user_id)
                    ->leftJoin('users', 'sales_members.user_id', 'users.user_id')
                    ->where('users.delete_flg', NO_DELETE)
                    ->select('member_kind', 'user_code', 'user_contact_name', 'users.user_id', 'user_kpi_id', 'sales_members.created_at')
                    ->get()->toArray();
                $sale_team = $sale_team ?? [];
                foreach ($sale_team as $key => $value) {
                    $createdAt = Carbon::parse($value['created_at']);
                    $isInRange = Carbon::parse($createdAt)->lessThanOrEqualTo($filter_date['end_date']);
                    // Check xem thời gian member được add vào team
                    if (!$isInRange) {
                        $sale_team[$key]['note'] = 'Sale được thêm vào ngày ' . Carbon::parse($value['created_at'])->format('Y-m-d') . ', nên không có thống kê';
                        continue;
                    }

                    $convert = getDataSmPackageByDate($value['user_id'], $filters);
                    $fwd_weight_user = getTotalWeightCompaniesForSale2($value['user_id'], $filters);

                    // Lấy KPI của member để tính hoa hồng leader dc hưởng từ team
                    $member_kpi = onGetKpiSale($value['user_kpi_id'], $monthDate);
                    $member_range_kpi = getRangeSaleKpi($member_kpi['range_kpis'] ?? [], $convert['total_profit']);
                    if ($member_range_kpi) {
                        // Nếu KPI của member đạt chỉ tiêu thì leader sẽ được hưởng FWD Weight cost
                        if ($member_range_kpi['leader_have_commission'] == 1) {
                            $member_commission = $convert['total_profit'] - $member_range_kpi['salary'];
                            $member_commission = $member_commission > 0 ? $member_commission : 0;
                            $commission_leader_team += ($member_commission * 0.07);
                        }
                        $sale_team[$key]['salary'] = $member_range_kpi['salary'];
                        $sale_team[$key]['ratio_commission'] = $member_range_kpi['ratio_commission'];
                        $sale_team[$key]['range_kpi_name'] = $member_range_kpi['range_kpi_name'];
                    } else {
                        $sale_team[$key]['salary'] = 0;
                        $sale_team[$key]['ratio_commission'] = 0;
                        $sale_team[$key]['range_kpi_name'] = isset($value['user_kpi_id']) ? 'Sale chưa đạt đủ KPI' : 'Sale chưa gắn kpi';
                    }

                    $sale_team[$key]['total_profit'] = $convert['total_profit']; // Lợi nhuận member
                    $sale_team[$key]['total_weight'] = $fwd_weight_user['total_weight']; // Tổng số fwd weight khách đã tạo của member
                    $sale_team[$key]['fwd_cost'] = $fwd_weight_user['total_price']; // Tổng số tiền hưởng từ fwd weight khách đã tạo của member

                    // Leader được hưởng FWD Weight cost của user
                    $fw_weight_cost_leader_team += $fwd_weight_user['total_leader_member_cost'];

                    $total_team['team_profit'] +=  $convert['total_profit'];
                    $total_team['team_fwd_weight'] +=  $fwd_weight_user['total_weight'];
                    $total_team['team_fwd_cost'] +=  $fwd_weight_user['total_price'];
                }
            }
            $sale_statistic['commission_leader_team'] = $commission_leader_team;
            $sale_statistic['fw_weight_cost_leader_team'] = $fw_weight_cost_leader_team;
            $sale_statistic['final_salary_sale'] += $commission_leader_team + $fw_weight_cost_leader_team;

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'info_sale' => $info_sale,
                    'sale_statistic' => $sale_statistic,
                    'kpi' => $kpi,
                    'is_leader' => $is_leader,
                    'sale_team' => $sale_team,
                    'total_team' => $total_team,
                    'monthDate' => $monthDate,
                    'is_get_all' => $is_get_all,
                ]);
            }

            return view(
                'manage.sale.leader.detail',
                compact('info_sale', 'sale_statistic', 'kpi', 'is_leader', 'sale_team', 'total_team', 'monthDate')
            );
        } catch (\Throwable $th) {
            abort('404');
        }
    }

    public function showShipmentsBySale(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = $request->is('api/*');
            $user = User::find($request_data['user_id']);

            $path = 'public/config/system-settings.json';
            $view_operating_costs_position_ids = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_VIEW_AND_UPDATE_SHIPMENT_OPERATING_COST)['position_ids'];
            $is_view_shipment_operating_costs = in_array(auth()->user()['position_id'], $view_operating_costs_position_ids);

            $filters = isset($request_data['filters']) ? (is_array($request_data['filters']) ? $request_data['filters'] : json_decode($request_data['filters'], true)) : [];
            $shipments = Shipment::with(['service', 'branch', 'country', 'packages', 'shipmentOperatingCosts'])
                ->where('shipments.user_id', $user->user_id)->isShow()
                ->orderBy('shipments.created_at', 'desc')->isFilters($filters);

            if (isset($request_data['debit_id'])) {
                $shipments = $shipments->where('shipments.shipment_debit_id', $request_data['debit_id']);
            }

            if (isset($request_data['is_excel']) && json_decode($request_data['is_excel'])) {
                $excels = $shipments->get()->toArray();

                return response()->json([
                    'status' => HTTP_OK,
                    'excels' => $excels,
                ]);
            }
            $shipments = $shipments->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            $is_fwd = onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user->position_id);

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'shipments' => $shipments,
                ]);
            } else {
                $choosed_id = $request_data['choosed_id'] ?? [];
                $is_company = isset($request_data['is_company']) ? json_decode($request_data['is_company']) : false;
                $html = view('manage.sale.leader.ajax.table-shipments', compact('shipments', 'is_view_shipment_operating_costs', 'is_company', 'is_fwd', 'choosed_id'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function showShipmentsByFwd(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = $request->is('api/*');
            $user = User::find($request_data['user_id']);
            $sale_link = SaleLinkFwd::isShow()
                ->leftJoin('users', 'sale_link_fwds.sale_id', '=', 'users.user_id')
                ->where('fwd_id', $user->user_id)
                ->select('sale_link_fwds.sale_id', 'users.user_code', 'users.user_contact_name', 'users.user_company_name')
                ->first();
            $sale_team = SaleMember::isShow()->where('sales_members.user_id', $sale_link['sale_id'])
                ->leftJoin('sales_teams', 'sales_members.sale_team_id', '=', 'sales_teams.sale_team_id')
                ->select(
                    'sales_members.user_id',
                    'sales_teams.leader_id',
                    'sales_members.member_kind'
                )
                ->groupBy('sales_members.user_id')->first();
            $filters = isset($request_data['filters']) ? (is_array($request_data['filters']) ? $request_data['filters'] : json_decode($request_data['filters'], true)) : [];
            $shipments = Shipment::isShow()
                ->where('shipments.user_id', $user->user_id)
                ->orderBy('shipments.created_at', 'asc')->isFilters($filters);

            if (isset($request_data['debit_id'])) {
                $shipments = $shipments->where('shipments.shipment_debit_id', $request_data['debit_id']);
            }

            if (isset($request_data['is_excel']) && json_decode($request_data['is_excel'])) {
                $shipments = $shipments->with(['service', 'branch', 'country', 'packages', 'debit']);
                $shipments = $shipments
                    ->whereNotIn('shipment_status', [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_RETURNED])
                    ->where('shipment_payment_status', SUCCESS);
                $data = $shipments->get();
                $viewName = 'manage.sale.ajax.fwd.table-shipment-excel';
                $name_excel = $user['user_company_name'] . ' export.xlsx';
                $other_data = [
                    'company' => $user,
                    'sale_link' => $sale_link,
                    'sale_team' => $sale_team
                ];
                return Excel::download(new ExportFile($data, $viewName, $other_data), $name_excel);

                // return response()->json([
                //     'status' => HTTP_OK,
                //     'excels' => $excels,
                // ]);
            }
            $shipments = $shipments->with(['service', 'branch', 'country', 'packages'])
                ->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            if ($is_api) {
                $data_sm = $shipments->toArray()['data'];
                foreach ($data_sm as $key => $shipment) {
                    $packages = $shipment['packages'] ?? [];
                    $weight = collect($packages)->sum('package_charged_weight') ?? 0;
                    $data_costs = getDataFwdCost(
                        $shipment['user_id']  ?? null,
                        $shipment['service']['service_id'],
                        $sale_team['member_kind'] ?? null,
                        $weight,
                    );
                    $data_sm[$key]['data_costs'] = $data_costs;
                    $data_sm[$key]['surcharge_goods_price'] = getSurchargeGoodPackages($shipment['shipment_id'])['price'];
                }

                return response()->json([
                    'status' => HTTP_OK,
                    'shipments' => $data_sm,
                    'sale_team' => $sale_team
                ]);
            } else {
                $html = view('manage.sale.ajax.fwd.table-shipment', compact('shipments', 'sale_team'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    // detail
    public function getDataShipmentForExcel(Request $request)
    {

        // user_code
        // monthDate
        // 
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $filters = isset($request_data['filters']) ? (is_array($request_data['filters']) ? $request_data['filters'] : json_decode($request_data['filters'], true)) : [];

            $filter_date = $this->getFilterDate($request->query('monthDate'));
            $filters['date_range'] = $filter_date;
            $user = User::where('user_code', $request->query('user_code'))->first();
            $export_name = 'Báo cáo sale ' . $request->query('monthDate') . '-' . $user['user_contact_name'] . '.xlsx';

            $shipments = Shipment::leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->leftJoin('branchs', 'shipments.shipment_branch_id', 'branchs.branch_id')
                ->leftJoin('countries', 'shipments.receiver_country_id', 'countries.country_id')
                ->where('shipments.user_id', $user['user_id'])
                ->isShow()
                ->isFilters($filters)
                ->select(
                    'shipments.*',
                    DB::raw("CASE 
                    WHEN shipments.shipment_payment_status = 0 THEN 0 
                    WHEN shipments.shipment_status = 0 THEN 0 
                    WHEN shipments.shipment_status = 3 THEN 0 
                    WHEN shipments.shipment_status = 4 THEN 0 
                    ELSE shipments.shipment_amount_profit 
                    END as shipment_amount_profit"),
                    'branch_name',
                    'service_name',
                    'country_name',
                    DB::raw('COUNT(packages.package_id) as count_package'),
                    DB::raw('SUM(packages.package_weight) as receive_weight'),
                    DB::raw('SUM(packages.package_charged_weight) as charged_weight')
                )
                ->groupBy('shipments.shipment_id')->orderBy('shipments.created_at', 'asc'); // Đảm bảo groupBy chỉ theo cột duy nhất.
            $data = [];
            $data['shipments'] = $shipments->get()->toArray();
            $sale = SaleMember::where('user_id', $user['user_id'])->isShow()->first();
            $data['leader_name'] = '';
            $is_leader = false;


            if ($sale) {
                $is_leader =  $sale['member_kind'] == SALE_LEADER;
                $team = SaleTeam::isShow()->where('sale_team_id', $sale['sale_team_id'])
                    ->leftJoin('users', 'sales_teams.leader_id', 'users.user_id')
                    ->select('users.user_contact_name')->first();

                $data['leader_name'] = $team['user_contact_name'] ?? '';
            }

            $data['is_leader'] = $is_leader;

            $sale_statistic = $request->query('sale_statistic');
            $sale_statistic = isset($sale_statistic) ? json_decode($sale_statistic, true) : null;

            $data['sale_statistic'] = $sale_statistic;
            $data['user'] = $user;

            return Excel::download(new SaleExport($data), $export_name);
        } catch (\Throwable $th) {
            return viewError404();
        }
    }


    public function displayHistoryEdit(Request $request)
    {

        try {
            $shipment_code = $request->query('shipment_code');
            $shipment = Shipment::where('shipment_code', $shipment_code)->isShow()
                ->leftJoin('users', 'shipments.user_id', 'users.user_id')
                ->select('shipments.*', 'users.user_contact_name')->first();
            if (!$shipment) {
                abort('404');
            }
            return view('manage.sale.leader.history-edit', compact('shipment'));
        } catch (\Throwable $th) {
            abort('404');
        }
    }
    public function showListHistoryEdit(Request $request)
    {
        try {
            $request_data = $request->all();

            $histories = HistoryEditShipment::where('shipment_id', $request_data['shipment_id'])
                ->leftJoin('users', 'history_edit_shipments.edit_by', 'users.user_id')
                ->select('history_edit_shipments.*', 'users.user_contact_name', 'user_code');

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                $histories = $histories->where(function ($query) use ($filters) {
                    return $query->where('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%')
                        ->orWhere('history_edit_shipments.note', 'like', '%' . $filters['keywords'] . '%');
                });
                if (isset($filters['edit_at'])) {
                    $histories = $histories->whereDate('edit_at', Carbon::parse($filters['edit_at'])->toDateTimeLocalString());
                }
            }

            $paginate = isset($request_data['paginate']) ? $request_data['paginate'] : null;
            $histories = $histories->orderBy('created_at', 'desc')->isPaginate($paginate);

            $html = view('manage.sale.leader.ajax.table-histories', compact('histories'))->render();

            return [
                'status' => HTTP_OK,
                'html' => $html
            ];
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function onUpdateCreatedAtShipment(Request $request)
    {
        $this->validate($request, [
            'shipment_code' => 'required',
            'date' => 'required'
        ]);
        try {
            $request_data = $request->all();
            $user = $request->user();

            $shipment = Shipment::isShow()->where('shipment_code', $request_data['shipment_code'])->first();
            if (!$shipment) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
            }
            DB::beginTransaction();
            $new_date = Carbon::parse($request_data['date']);
            onCreateHistoryShipment(
                $shipment['shipment_id'],
                $user['user_id'],
                'Cập nhật ngày tạo đơn từ ' . $shipment['created_at'] . ' sang ngày ' . $new_date
            );
            Shipment::find($shipment['shipment_id'])
                ->update([
                    'created_at' => $new_date,
                    // 'completed_date' => $new_date,
                ]);
            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'text' => 'Cập nhật ngày tạo thành công',
                    'title' => 'Thông báo',
                    'icon' => 'success'
                ],
                'created_at' => $new_date
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function displayConfirmAccount(Request $request)
    {
        $action = $request->query('action_kind');
        if (!$action) {
            abort('404');
        }

        return view('manage.sale.confirm-account', compact('action'));
    }

    public function showListAction(Request $request)
    {
        $this->validate($request, [
            'action_kind' => 'required',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            switch ($request_data['action_kind']) {
                case 'fwd':
                    $data = SaleLinkFwd::where('sale_link_fwds.active_flg', INACTIVE)
                        ->where('sale_link_fwds.delete_flg', NO_DELETE)
                        ->leftJoin('users as sales', 'sale_link_fwds.sale_id', 'sales.user_id')
                        ->leftJoin('users as fwds', 'sale_link_fwds.fwd_id', 'fwds.user_id')
                        ->select(
                            'sales.user_code as sale_code',
                            'sales.user_contact_name as sale_name',
                            'sale_link_fwds.sale_id',
                            'fwds.user_code as fwd_code',
                            'fwds.user_company_name as fwd_name',
                            'sale_link_fwds.fwd_id',
                            'sale_link_fwds.created_at',
                            'sale_link_fwds.sale_link_fwd_id',
                        )
                        ->groupBy('sale_link_fwds.sale_link_fwd_id');
                    if (isset($request_data['filters'])) {
                        $filters = $request_data['filters'];
                        if (
                            (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                            && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
                        ) {
                            $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                            $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                            $data = $data->whereBetween("sale_link_fwds.created_at", [$start_date, $end_date]);
                        }

                        if ($filters['keywords']) {
                            $data = $data->where(function ($query) use ($filters) {
                                $query->where("sales.user_contact_name", 'like', '%' . $filters['keywords'] . '%')
                                    ->orWhere("fwds.user_company_name", 'like', '%' . $filters['keywords'] . '%');
                            });
                        }
                    }
                    break;
                case 'member':
                    $data = SaleMember::where('sales_members.active_flg', INACTIVE)
                        ->where('sales_members.delete_flg', NO_DELETE)
                        ->leftJoin('users as members', 'sales_members.user_id', 'members.user_id')
                        ->leftJoin('sales_teams', 'sales_members.sale_team_id', 'sales_teams.sale_team_id')
                        ->leftJoin('users as leaders', 'sales_teams.leader_id', 'leaders.user_id')
                        ->select(
                            'members.user_code as member_code',
                            'members.user_contact_name as member_name',
                            'members.user_id as member_id',
                            'leaders.user_code as leader_code',
                            'leaders.user_contact_name as leader_name',
                            'leaders.user_id as leader_id',
                            'sales_members.created_at',
                            'sales_members.sale_member_id',
                            'sales_members.sale_team_id'
                        )
                        ->groupBy('sales_members.sale_member_id');

                    if (isset($request_data['filters'])) {
                        $filters = $request_data['filters'];
                        if (
                            (isset($filters['date_range']['start_date']) && $filters['date_range']['start_date'] != "")
                            && isset($filters['date_range']['end_date']) && $filters['date_range']['end_date'] != ""
                        ) {
                            $start_date = Carbon::parse($filters['date_range']['start_date'])->startOfDay()->toDateTimeLocalString();
                            $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->toDateTimeLocalString();
                            $data = $data->whereBetween("sales_members.created_at", [$start_date, $end_date]);
                        }

                        if ($filters['keywords']) {
                            $data = $data->where(function ($query) use ($filters) {
                                $query->where("members.user_contact_name", 'like', '%' . $filters['keywords'] . '%')
                                    ->orWhere("leaders.user_contact_name", 'like', '%' . $filters['keywords'] . '%');
                            });
                        }
                    }
                    break;
                default:
                    return returnResponse(HTTP_NOT_FOUND, 'Kiểu hành động không tồn tại');
            }

            $data = $data->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "data" => $data->toArray()['data'],
                ]);
            } else {
                switch ($request_data['action_kind']) {
                    case 'fwd':
                        $html = view('manage.sale.ajax.confirm-fwd', compact('data'))->render();
                        break;
                    case 'member':
                        $html = view('manage.sale.ajax.confirm-member', compact('data'))->render();
                        break;
                    default:
                        return returnResponse(HTTP_NOT_FOUND, 'Kiểu hành động không tồn tại');
                }

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleConfirmAction(Request $request)
    {
        $this->validate($request, [
            'action_kind' => 'required',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            switch ($request_data['action_kind']) {
                case 'fwd':
                    $check = SaleLinkFwd::isShow()
                        ->where('fwd_id', $request_data['first_id'])
                        ->first();

                    if ($check) {
                        return returnResponse(HTTP_BAD_REQUEST, 'fwd company này đã được liên kết với sale khác');
                    }
                    SaleLinkFwd::where('fwd_id', $request_data['first_id'])
                        ->where('sale_id', $request_data['second_id'])
                        ->first()->update(['active_flg' => ACTIVE]);
                    break;
                case 'member':
                    $check = SaleMember::isShow()
                        ->where('user_id', $request_data['first_id'])
                        ->first();
                    if ($check) {
                        return returnResponse(HTTP_BAD_REQUEST, 'Sale này đã thuộc team của leader khác');
                    }
                    SaleMember::where('user_id', $request_data['first_id'])
                        ->where('sale_team_id', $request_data['second_id'])
                        ->first()->update(['active_flg' => ACTIVE]);
                    break;
                default:
                    return returnResponse(HTTP_NOT_FOUND, 'Kiểu hành động không tồn tại');
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Thao tác thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleCancelAction(Request $request)
    {
        $this->validate($request, [
            'action_kind' => 'required',
            'key' => 'required|numeric',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            switch ($request_data['action_kind']) {
                case 'fwd':
                    SaleLinkFwd::find($request_data['key'])->update(['delete_flg' => DELETED]);
                    break;
                case 'member':
                    SaleMember::find($request_data['key'])->update(['delete_flg' => DELETED]);
                    break;
                default:
                    return returnResponse(HTTP_NOT_FOUND, 'Kiểu hành động không tồn tại');
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Thao tác thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    // Màn hình quản lý fwd được liên kết với sale
    public function displayFwd()
    {
        $check_sale = auth()->user()['position_id'] == $this->getPositionSale()['position_id'] ? true : false;

        $sale_id  = $check_sale ? auth()->user()['user_id'] : null;
        $sales = User::isShow()->where('position_id', $this->getPositionSale()['position_id'])
            ->select('user_code', 'user_contact_name', 'user_id')
            ->get();

        return view('manage.sale.fwd.index', compact('sale_id', 'sales'));
    }


    // show danh sách fwd link sale theo html leader
    public function showListSaleSuportFwd(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            $sales = SaleLinkFwd::isShow()->with(['leader', 'companies'])
                ->leftJoin('users', 'sale_link_fwds.sale_id', 'users.user_id')
                ->select(
                    'users.user_code',
                    'users.user_id',
                    'users.user_contact_name',
                    'sale_link_fwds.sale_id',
                    'sale_link_fwds.fwd_id'
                )
                ->groupBy('sale_link_fwds.sale_id');

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['keywords'])) {
                    $sales = $sales->where('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%');
                }
            }

            $sales = $sales->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $sales->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.ajax.fwd.sale-link-fwd', compact('sales'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }
    // show danh sách fwd link sale theo html table
    public function showListSaleSuportFwd2(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $date_range = isset($request_data['filters']['date_range']) ? $request_data['filters']['date_range'] : null;
            $companies = SaleLinkFwd::isShow()
                ->leftJoin('users as sales', 'sale_link_fwds.sale_id', '=', 'sales.user_id')
                ->leftJoin('users as companies', 'sale_link_fwds.fwd_id', '=', 'companies.user_id')
                ->leftJoin('shipments', function ($join) use ($date_range) {
                    $join->on('sale_link_fwds.fwd_id', '=', 'shipments.user_id')
                        ->where('shipments.delete_flg', NO_DELETE)
                        ->where('shipments.active_flg', ACTIVE);
                    if (isset($date_range['start_date']) && isset($date_range['end_date'])) {
                        $start_date = Carbon::parse($date_range['start_date'])->startOfDay()->toDateTimeLocalString();
                        $end_date = Carbon::parse($date_range['end_date'])->endOfDay()->toDateTimeLocalString();
                        $join->whereBetween('shipments.created_at', [$start_date, $end_date]);
                    }
                })
                ->leftJoin('fwd_costs', function ($join) {
                    $join->on('shipments.user_id', '=', 'fwd_costs.fwd_id')
                        ->on('shipments.shipment_service_id', '=', 'fwd_costs.service_id')
                        ->where('fwd_costs.delete_flg', NO_DELETE)
                        ->where('fwd_costs.active_flg', ACTIVE);
                })
                ->leftJoin('packages', function ($join) {
                    $join->on('shipments.shipment_id', '=', 'packages.shipment_id')
                        ->where('packages.delete_flg', NO_DELETE)
                        ->where('packages.active_flg', ACTIVE);
                });

            if (isset($request_data['sale_id'])) {
                $companies = $companies->where('sale_link_fwds.sale_id', $request_data['sale_id']);
            }

            $companies = $companies->leftJoin('sales_members', function ($join) {
                $join->on('sale_link_fwds.sale_id', '=', 'sales_members.user_id')
                    ->where('sales_members.delete_flg', NO_DELETE)
                    ->where('sales_members.active_flg', ACTIVE);
            })->leftJoin('sales_teams', function ($join) {
                $join->on('sales_members.sale_team_id', '=', 'sales_teams.sale_team_id')
                    ->where('sales_teams.delete_flg', NO_DELETE)
                    ->where('sales_teams.active_flg', ACTIVE);
            })->leftJoin('users as leader', function ($join) {
                $join->on('sales_teams.leader_id', '=', 'leader.user_id')
                    ->where('leader.delete_flg', NO_DELETE)
                    ->where('leader.active_flg', ACTIVE);
            });

            if (isset($request_data['filters']['keywords'])) {
                $keywords = $request_data['filters']['keywords'];
                $companies = $companies->where(function ($query) use ($keywords) {
                    return $query->where('companies.user_company_name', 'like', '%' . $keywords . '%')
                        ->orWhere('sales.user_contact_name', 'like', '%' . $keywords . '%');
                });
            }

            $companies = $companies->select(
                'sales.user_contact_name as sale_name',
                'sales.user_code as sale_code',
                'companies.user_company_name as company_name',
                'companies.user_code as company_code',
                'leader.user_contact_name as leader_name',
                'leader.user_code as leader_code',
                'leader.user_id as leader_id',
                'sale_link_fwds.sale_id',
                'sale_link_fwds.fwd_id',
                'sale_link_fwds.sale_link_fwd_id',
                DB::raw('SUM(packages.package_charged_weight) as total_charged_weight'),
                DB::raw('SUM(packages.package_charged_weight * IFNULL(fwd_costs.leader_cost, 5000)) as total_leader_cost'),
                DB::raw('SUM(packages.package_charged_weight * IFNULL(fwd_costs.member_cost, 3000)) as total_member_cost'),
                DB::raw('SUM(packages.package_charged_weight * IFNULL(fwd_costs.leader_member_cost, 2000)) as total_leader_member_cost'),
                DB::raw('SUM(packages.package_charged_weight) as fw_weight')
            )
                ->orderBy('sale_name', 'asc')
                ->groupBy('sale_link_fwds.sale_link_fwd_id');

            if (isset($request_data['is_excel']) && json_decode($request_data['is_excel'])) {
                $start_date = Carbon::parse($date_range['start_date'])->toDateString();
                $end_date = Carbon::parse($date_range['end_date'])->endOfDay()->toDateString();
                return response()->json([
                    "status" => HTTP_OK,
                    "data" => $companies->get(),
                    "start_date" => $start_date,
                    "end_date" => $end_date,
                ]);
            }
            $companies = $companies->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "user" => $companies->toArray()['data'],
                ]);
            } else {
                $sale_id = isset($request_data['sale_id']) ? $request_data['sale_id'] : null;
                $html = view('manage.sale.ajax.fwd.sale-link-fwd2', compact('companies', 'sale_id'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    private function convertDataListMangaLinkFwdForApi($datas, $filters)
    {
        try {
            foreach ($datas as $key => $data) {
                $datas[$key]['companies'] = getCompanyLinkSaleWithWeight($data['user_id'], $filters);
                if (isset($data['members']) && is_array($data['members'])) {
                    foreach ($data['members'] as $key2 => $member) {
                        $datas[$key]['members'][$key2]['companies'] = getCompanyLinkSaleWithWeight($member['user_id'], $filters);
                    }
                }
            }
            return $datas;
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }
    // show danh sách fwd link sale theo web cũ
    public function showListSaleSuportFwd3(Request $request)
    {
        try {
            // Đứa nào bị yêu cầu làm lại cái này nhớ chửi con KT bên đó
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $filters = $request_data['filters'] ?? [];

            if (!isGetAllData()) {
                $user = request()->user();
                $team = SaleTeam::isShow()->where('leader_id', $user['user_id'])->first();
                $datas[0] = [
                    'user_id' => $user['user_id'],
                    'user_code' => $user['user_code'],
                    'user_contact_name' => $user['user_contact_name'],
                    'sale_team_id' => $team['sale_team_id'] ?? null,
                ];
                if (isset($team)) {
                    $members = SaleMember::isShow()
                        ->where('sale_team_id', $team['sale_team_id'])
                        ->where('sales_members.user_id', '!=', $user['user_id'])
                        ->leftJoin('users', 'sales_members.user_id', 'users.user_id')
                        ->select('users.user_id', 'users.user_code', 'users.user_contact_name')
                        ->groupBy('sales_members.user_id')->get()->toArray();
                    $datas[0]['members'] = $members;
                }
                if ($is_api) {
                    $datas = $this->convertDataListMangaLinkFwdForApi($datas, $filters) ?? $datas;
                    return response()->json([
                        "status" => HTTP_OK,
                        "users" => $datas,
                    ]);
                } else {
                    $html = view('manage.sale.ajax.fwd.sale-link-fwd3', compact('datas', 'filters'))->render();

                    return response()->json([
                        "status" => HTTP_OK,
                        "html" => $html,
                    ]);
                }
            }

            $arr_have_team = [];
            $leaders = SaleTeam::isShow()
                ->leftJoin('users', 'sales_teams.leader_id', 'users.user_id')
                ->select(
                    'users.user_id',
                    'users.user_code',
                    'users.user_contact_name',
                    'sales_teams.sale_team_id'
                )
                ->get()->toArray();
            foreach ($leaders as $key => $value) {
                array_push($arr_have_team, $value['user_id']);
                $members = SaleMember::isShow()
                    ->where('sale_team_id', $value['sale_team_id'])
                    ->where('sales_members.user_id', '!=', $value['user_id'])
                    ->leftJoin('users', 'sales_members.user_id', 'users.user_id')
                    ->select('users.user_id', 'users.user_code', 'users.user_contact_name')
                    ->groupBy('sales_members.user_id')->get()->toArray();
                $leaders[$key]['members'] = $members;
                foreach ($members as $key => $member) {
                    array_push($arr_have_team, $value['user_id']);
                }
            }

            $no_team = SaleLinkFwd::isShow()
                ->whereNotIn('sale_id', $arr_have_team)
                ->leftJoin('users', 'sale_link_fwds.sale_id', 'users.user_id')
                ->select(
                    'users.user_code',
                    'users.user_id',
                    'users.user_contact_name',
                )
                ->groupBy('sale_link_fwds.sale_id')
                ->get()->toArray();
            $datas = $leaders;
            foreach ($no_team as $key => $sale) {
                array_push($datas, $sale);
            }
            if ($is_api) {
                $datas = $this->convertDataListMangaLinkFwdForApi($datas, $filters) ?? $datas;
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $datas,
                ]);
            } else {
                $html = view('manage.sale.ajax.fwd.sale-link-fwd3', compact('datas', 'filters'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
        }
    }

    // export excel fwd tổng weight và lợi nhuận của sale
    public function exportExcelFwdWeightCost($companies)
    {
        try {
            foreach ($companies as $key => $value) {
                # code...
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }


    // lấy data fwd chưa liên kết với sale
    public function showListFwdNoLink(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $fwds = User::isShow()->where('position_id', $this->getPositionSale(SETTING_FWD_ACCOUNT)['position_id']);

            $fwd_linked_id = SaleLinkFwd::isShow()->select('fwd_id')->pluck('fwd_id');
            if ($fwd_linked_id) {
                $fwds = $fwds->whereNotIn('user_id', $fwd_linked_id);
            }
            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['keywords']) && $filters['keywords'] !== '') {
                    $fwds = $fwds->where(function ($query) use ($filters) {
                        return $query->where('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%');
                    });
                }
            }
            return response()->json([
                'status' => HTTP_OK,
                'fwds' => $fwds->get()
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleCreateSaleLinkFwd(Request $request)
    {
        try {
            $this->validate($request, [
                'sale_id' => 'required|numeric',
                'fwd_ids.*' => 'required|numeric',
            ]);


            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();

            foreach ($request_data['fwd_ids'] as $key => $fwd_id) {
                if (SaleLinkFwd::isShow()->where('sale_id', $request_data['sale_id'])->where('fwd_id', $fwd_id)->exists()) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Có FWD được chọn có ở trong team khác', null, 'warning');
                }
                if (
                    $this->onCheckSaleManage() &&
                    SaleLinkFwd::where('active_flg', INACTIVE)
                    ->where('delete_flg', NO_DELETE)
                    ->where('sale_id', $request_data['sale_id'])
                    ->where('fwd_id', $fwd_id)->exists()
                ) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Có FWD được chọn đang được chờ xét duyệt', null, 'warning');
                }
                SaleLinkFwd::create([
                    'sale_id' => $request_data['sale_id'],
                    'fwd_id' => $fwd_id,
                    'active_flg' => $this->onCheckSaleManage() ? INACTIVE : ACTIVE
                ]);
            }
            DB::commit();
            if ($this->onCheckSaleManage()) {
                return returnResponse(HTTP_OK, 'Vui lòng chờ xét duyệt yêu cầu', 'Thông tin', 'success');
            }
            return returnResponse(HTTP_OK, 'Tạo liên kết thành công', 'Thông tin', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleDeleteSaleLinkFwd(Request $request)
    {
        $this->validate($request, [
            'key_id' => 'required|numeric',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $slf = SaleLinkFwd::isShow()->where('sale_link_fwd_id', $request_data['key_id'])->first();
            if (!$slf) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy liên kết cần xóa');
            }
            DB::beginTransaction();
            $slf->update(['delete_flg' => DELETED]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa liên kết thành công', 'Thông tin', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }


    // Màn hình thêm bảng giá FWD liên kết với dịch vụ
    public function displayFwdAddCost()
    {
        if (auth()->user()['position_id'] == $this->getPositionSale()['position_id']) {
            abort('404');
        }
        $services = Service::isShow()->select('service_id', 'service_name')->get();
        return view('manage.sale.fwd.add-cost', compact('services'));
    }

    public function showListFwdCost(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sale = $this->getPositionSale(SETTING_FWD_ACCOUNT);

            $fwd_costs = FwdCostByService::isShow()
                ->leftJoin('users', 'fwd_costs.fwd_id', 'users.user_id')
                ->leftJoin('services', 'fwd_costs.service_id', 'services.service_id')
                ->select('fwd_costs.*', 'users.user_company_name', 'users.user_code', 'services.service_name');

            $is_company = isset($request_data['company_id']);
            if ($is_company) {
                $fwd_costs =  $fwd_costs->where('fwd_id', $request_data['company_id']);
            }

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['service_id'])) {
                    $fwd_costs =  $fwd_costs->where('fwd_costs.service_id', $filters['service_id']);
                }
                if (isset($filters['keywords'])) {
                    $fwd_costs =  $fwd_costs->where(function ($query) use ($filters) {
                        return $query->where('users.user_company_name', 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%');
                    });
                }
            }
            if (isset($request_data['html_kind']) && $request_data['html_kind'] == 'leader-link-cost') {
                $fwd_costs = $fwd_costs->whereNull('leader_id');
            }

            $fwd_costs = $fwd_costs->groupBy('fwd_costs.fwd_cost_id');


            $choosed_id = isset($request_data['choosed_id']) ? $request_data['choosed_id'] : [];
            $is_get_all_user = isset($request_data['is_get_all_user']) ? json_decode($request_data['is_get_all_user']) : false;
            if (isset($request_data['is_get_ids']) && json_decode($request_data['is_get_ids'])) {
                return response()->json([
                    "status" => HTTP_OK,
                    "fwd_costs" => $is_get_all_user ? $fwd_costs->pluck('fwd_cost_id') : [],
                ]);
            }

            $fwd_costs = $fwd_costs->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $fwd_costs->toArray()['data'],
                ]);
            } else {
                if (isset($request_data['html_kind']) && $request_data['html_kind'] == 'leader-link-cost') {

                    $html = view('manage.sale.ajax.fwd.table-costs', compact('fwd_costs', 'choosed_id', 'is_get_all_user'))->render();

                    return response()->json([
                        "status" => HTTP_OK,
                        "html" => $html,
                    ]);
                }

                $html = view('manage.sale.ajax.fwd.table-cost-service', compact('fwd_costs', 'is_company'))->render();
                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function showListUserFwd(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sale = $this->getPositionSale(SETTING_FWD_ACCOUNT);
            $users = User::isShow()->where('users.position_id', $sale['position_id']);

            if (isset($request_data['filters'])) {
                $users = $users->isFilters($request_data['filters']);
            }
            if (isset($request_data['service_id'])) {
                $user_not_in = FwdCostByService::isShow()->where('service_id', $request_data['service_id'])->pluck('fwd_id');
                $users = $users->whereNotIn('user_id', $user_not_in);
            }

            $choosed_id = isset($request_data['choosed_id']) ? $request_data['choosed_id'] : [];
            $is_get_all_user = isset($request_data['is_get_all_user']) ? json_decode($request_data['is_get_all_user']) : false;

            if (isset($request_data['is_get_ids']) && json_decode($request_data['is_get_ids'])) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $is_get_all_user ? $users->pluck('user_id') : [],
                ]);
            }

            $users = $users->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $users->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.ajax.fwd.table-users', compact('users', 'choosed_id', 'is_get_all_user'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleMiltipleFwdCostByServiceCreate(Request $request)
    {
        $this->validate($request, [
            'fwd_ids.*' => 'required|numeric',
            'service_id' => 'required|numeric',
            'leader_cost' => 'required|numeric|min:0',
            'member_cost' => 'required|numeric|min:0',
            'leader_member_cost' => 'required|numeric|min:0',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            foreach ($request_data['fwd_ids'] as $key => $value) {
                if (FwdCostByService::isShow()->where('fwd_id', $value)->where('service_id', $request_data['service_id'])->exists()) {
                    continue;
                }
                FwdCostByService::create([
                    'fwd_id' => $value,
                    'service_id' => $request_data['service_id'],
                    'leader_cost' => $request_data['leader_cost'],
                    'member_cost' => $request_data['member_cost'],
                    'leader_member_cost' => $request_data['leader_member_cost'],
                ]);
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Thêm thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleFwdCostByServiceDelete(Request $request)
    {
        $this->validate($request, [
            "fwd_cost_id" => "required|numeric"
        ]);

        try {

            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $fwd_cost = FwdCostByService::isShow()->where("fwd_cost_id", $request_data['fwd_cost_id'])->first();
            if (!$fwd_cost) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
            }
            DB::beginTransaction();
            $fwd_cost->update(['delete_flg' => DELETED]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa đối tượng thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleFwdCostByServiceUpdate(Request $request)
    {
        $this->validate($request, [
            "fwd_cost_id" => "required|numeric",
            'leader_cost' => 'required|numeric|min:0',
            'member_cost' => 'required|numeric|min:0',
            'leader_member_cost' => 'required|numeric|min:0',
        ]);

        try {

            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $fwd_cost = FwdCostByService::isShow()->where("fwd_cost_id", $request_data['fwd_cost_id'])->first();
            if (!$fwd_cost) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
            }

            DB::beginTransaction();
            $fwd_cost->update([
                'leader_cost' => $request_data['leader_cost'],
                'member_cost' => $request_data['member_cost'],
                'leader_member_cost' => $request_data['leader_member_cost'],
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật đối tượng thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    function displayLeaderLinkFwdCost()
    {
        if (auth()->user()['position_id'] == $this->getPositionSale()['position_id']) {
            abort('404');
        }
        try {
            $leaders = SaleMember::isShow()->where('member_kind', SALE_LEADER)
                ->leftJoin('users', 'sales_members.user_id', 'users.user_id')
                ->select('sale_member_id', 'users.user_id', 'user_contact_name')
                ->groupBy('sale_member_id')->get();

            $services = Service::isShow()->select('service_id', 'service_name')->get();

            return view('manage.sale.leader-link-fwd-cost', compact('leaders', 'services'));
        } catch (\Throwable $th) {
            abort('404');
        }
    }

    function handleLeaderLinkCostFwd(Request $request)
    {
        $this->validate($request, [
            'leader_id' => 'required|numeric',
            'choosed_id.*' => 'required|numeric'
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $leader = SaleMember::isShow()->where('user_id', $request_data['leader_id'])->where('member_kind', SALE_LEADER)->first();
            if (!$leader) {
                return returnResponse(HTTP_NOT_FOUND, 'Đối tượng liên kết phải là leader, vui lòng kiểm tra lại thông tin');
            }

            DB::beginTransaction();
            FwdCostByService::isShow()->whereIn('fwd_cost_id', $request_data['choosed_id'])->update([
                'leader_id' => $request_data['leader_id']
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Liên kết thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
    function handleRemoveLinkLeaderToFwdCost(Request $request)
    {
        if (!$request->fwd_cost_id) {
            $this->validate($request, [
                'leader_id' => 'required|numeric'
            ]);
        } else {
            $this->validate($request, [
                'fwd_cost_id' => 'required|numeric',
            ]);
        }
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            if (isset($request_data['leader_id'])) {
                FwdCostByService::isShow()->where('leader_id', $request_data['leader_id'])->update([
                    'leader_id' => null
                ]);
            } else {
                $fwd_cost = FwdCostByService::isShow()->where('fwd_cost_id',  $request_data['fwd_cost_id'])->first();
                if (!$fwd_cost) {
                    return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
                }
                $fwd_cost->update([
                    'leader_id' => null
                ]);
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa liên kết thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    function showListLeaderLinkCost(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            $fwd_costs = FwdCostByService::isShow()->whereNotNull('leader_id')
                ->leftJoin('users', 'fwd_costs.leader_id', 'users.user_id')
                ->select('users.user_contact_name', 'users.user_code', 'fwd_costs.leader_id')
                ->with(['fwdCosts']);
            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['service_id'])) {
                    $fwd_costs =  $fwd_costs->where('fwd_costs.service_id', $filters['service_id']);
                }
                if (isset($filters['keywords'])) {
                    $fwd_costs =  $fwd_costs->where(function ($query) use ($filters) {
                        return $query->where('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%');
                    });
                }
            }


            $fwd_costs = $fwd_costs->groupBy('fwd_costs.leader_id')
                ->orderBy('fwd_costs.created_at')
                ->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $fwd_costs->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.ajax.fwd.leader-link-cost', compact('fwd_costs'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }



    public function displayFwdCompany(Request $request)
    {
        // if (auth()->user()['position_id'] == $this->getPositionSale()['position_id']) {
        //     abort('404');
        // }
        $is_api = $request->is('api/*');
        $service = config('constans.address.service-stranfer');

        $company_code = $request->query->get('company_code');
        if (!$company_code) {
            abort('404');
        }

        $company = User::isShow()->where('user_code', $company_code)->first();
        if (!$company || !onCheckPositionAccount(SETTING_FWD_ACCOUNT, $company->position_id)) {
            abort('404');
        }
        $sale_manager = SaleLinkFwd::isShow()->where('fwd_id', $company->user_id)
            ->leftJoin('users', 'sale_link_fwds.sale_id', 'users.user_id')
            ->select('users.*')->first();

        if ($is_api) {
            return response()->json([
                "status" => HTTP_OK,
                "sale_manager" => $sale_manager,
                "company" => $company
            ]);
        }

        return view('manage.sale.fwd.company', compact('service', 'company', 'sale_manager'));
    }

    public function displayAll()
    {
        if (!isGetAllData()) {
            abort('404');
        }

        $currentMonthYear = Carbon::now()->format('Y-m');
        return view('manage.sale.all', compact('currentMonthYear'));
    }

    public function showAllSale(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        $filters = $request_data['filters'] ?? [];

        if (isset($filters['date_range'])) {
            $startDate = Carbon::parse($filters['date_range']['start_date'])->format('Y-m-d'); // Thay đổi ngày bắt đầu
            $endDate = Carbon::parse($filters['date_range']['end_date'])->format('Y-m-d');   // Thay đổi ngày kết thúc
            unset($filters['date_range']);
        } else {
            $startDate = Carbon::now()->startOfMonth()->toDateString();
            $endDate = Carbon::now()->endOfMonth()->toDateString();
        }
        $date_range = [
            'start_date' => $startDate,
            'end_date' => $endDate,
        ];
        try {
            $position_id = $this->getPositionSale()['position_id'];

            $sales = User::isShow()
                ->leftJoin('shipments', function ($join) use ($date_range) {
                    $join->on('shipments.user_id', '=', 'users.user_id')
                        ->where('shipments.delete_flg', NO_DELETE)
                        ->where('shipments.active_flg', ACTIVE)
                        ->whereNotIn('shipment_status', [SHIPMENT_STATUS_CREATE, SHIPMENT_STATUS_RETURNED])
                        ->whereBetween('shipments.created_at', [$date_range['start_date'], $date_range['end_date']]);
                })
                ->where(function ($query) use ($filters) {
                    if (!empty($filters['keywords'])) {
                        $query->where('users.user_name', 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere('users.user_code', 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere('users.user_contact_name', 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere('users.user_company_name', 'like', '%' . $filters['keywords'] . '%');
                    }
                })
                ->where('users.position_id', $position_id)
                ->select(
                    'users.*',
                    DB::raw(
                        "SUM(CASE 
                        WHEN shipments.shipment_payment_status = 0 THEN 0 
                        WHEN shipments.shipment_status IN (0, 3, 4) THEN 0 
                        ELSE shipments.shipment_amount_profit 
                        END) as total_profit"
                    ),
                    DB::raw(
                        "SUM(CASE 
                            WHEN shipments.shipment_payment_status = 0 THEN 0 
                            WHEN shipments.shipment_status IN (0, 3, 4) THEN 0 
                            ELSE shipments.shipment_final_amount 
                        END) as total_price"
                    ),
                    DB::raw(
                        "SUM(CASE 
                            WHEN shipments.shipment_payment_status = 0 THEN 0 
                            WHEN shipments.shipment_status IN (0, 3, 4) THEN 0 
                            ELSE shipments.shipment_amount_total_customer 
                        END) as total_amount_customer"
                    ),
                )
                ->groupBy('users.user_id')
                ->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "sales" => $sales->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.ajax.table-all', compact('sales'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }


    // Các chức vụ quản lý KPI
    public function displayManageKpi()
    {
        if (auth()->user()['position_id'] == $this->getPositionSale()['position_id']) {
            abort('404');
        }
        return view('manage.sale.manage-kpi');
    }

    public function showListSaleKpis(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        try {
            $sale_kpis = SaleKpi::isShow()->with(['range_kpis'])->whereNull('user_id');

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['keywords']) && $filters['keywords'] != '') {
                    $sale_kpis = $sale_kpis->where('kpi_name', 'like', '%' . $filters['keywords'] . '%');
                }
                if (isset($filters['time_kind']) && in_array($filters['time_kind'], [HAVE_TIME_RANGE, NO_TIME_RANGE])) {
                    $sale_kpis = $sale_kpis->where('time_kind', $filters['time_kind']);
                }
            }

            $sale_kpis = $sale_kpis->isPaginate($request->paginate);


            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "sale_kpis" => $sale_kpis->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.ajax.kpi.list-kpis', compact('sale_kpis'))->render();

                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function showDetailKpi(Request $request)
    {
        $this->validate($request, ['kpi_id' => 'required|numeric']);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sale_kpi = SaleKpi::where('sale_kpi_id', $request_data['kpi_id'])->isShow()->with(['range_kpis'])->first();
            return response()->json([
                "status" => HTTP_OK,
                "sale_kpi" => $sale_kpi,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleCreateOrUpdatekpi(Request $request)
    {
        $this->validate(
            $request,
            [
                'kpi_name' => 'required',
                'time_kind' => 'required|min:0|max:1',
                'time_start' => $request->time_kind == NO_TIME_RANGE  ? '' : 'required|date',
                'time_end' => $request->time_kind == NO_TIME_RANGE  ? '' : 'required|date|after:time_start',
                'range_kpis.*.salary' => 'required|numeric|min:0',
                'range_kpis.*.ratio_commission' => 'required|numeric|min:0|max:100',
                'range_kpis.*.profit_start' => 'required|numeric|min:0',
                'range_kpis.*.profit_end' => [
                    'required',
                    'numeric',
                    'min:0',
                    function ($attribute, $value, $fail) use ($request) {
                        $index = explode('.', $attribute)[1];
                        $range_start = $request->input("range_kpis.$index.profit_start");
                        if ($value <= $range_start) {
                            $fail("Giá trị kết thúc phải lớn hơn giá trị khởi đầu.");
                        }
                    },
                ],
                'range_kpis.*.leader_have_commission' => 'required|numeric|min:0|max:1',
            ]
        );

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $kpis_data = [
                'kpi_name' => $request_data['kpi_name'],
                'kpi_kind' => isset($request_data['kpi_kind']) ? $request_data['kpi_kind'] : 0,
                'time_kind' => $request_data['time_kind'],
                'time_start' => $request_data['time_start'],
                'time_end' => $request_data['time_end'],
                'user_id' => isset($request_data['user_id']) ? $request_data['user_id'] : null,
            ];
            DB::beginTransaction();
            if (isset($request_data['kpi_id'])) {
                $sale_kpi =  SaleKpi::findOrFail($request_data['kpi_id']);
                if (!$sale_kpi) {
                    return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
                }
                $sale_kpi->update($kpis_data);
            } else {
                $sale_kpi = SaleKpi::create($kpis_data);
            }

            $range_kpis = $request_data['range_kpis'];
            $arr_id_not_delete = [];
            foreach ($range_kpis as $key => $value) {
                $range = RangeKpi::updateOrCreate(
                    [
                        'range_kpi_id' => isset($value['range_kpi_id']) ? $value['range_kpi_id'] : null,
                    ],
                    [
                        'sale_kpi_id' => $sale_kpi->sale_kpi_id,
                        'range_kpi_name' => $value['range_kpi_name'],
                        'salary' => $value['salary'],
                        'profit_start' => $value['profit_start'],
                        'profit_end' => $value['profit_end'],
                        'ratio_commission' => $value['ratio_commission'],
                        'leader_have_commission' => $value['leader_have_commission'],
                    ]
                );
                array_push($arr_id_not_delete, $range->range_kpi_id);
            }
            RangeKpi::where('sale_kpi_id', $sale_kpi->sale_kpi_id)->whereNotIn('range_kpi_id', $arr_id_not_delete)->delete();
            DB::commit();
            return returnResponse(HTTP_OK, 'Lưu thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleDeleteKpi(Request $request)
    {
        $this->validate($request, ['kpi_id' => 'required|numeric']);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sale_kpi =  SaleKpi::findOrFail($request_data['kpi_id']);
            if (!$sale_kpi) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
            }
            DB::beginTransaction();
            $sale_kpi->update(['delete_flg' => DELETED]);
            RangeKpi::where('sale_kpi_id', $sale_kpi->sale_kpi_id)
                ->update(['delete_flg' => DELETED]);

            User::where('user_kpi_id', $sale_kpi->sale_kpi_id)->update(['user_kpi_id' => null]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa bảng kpi thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function showSaleWithKpi(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sale = $this->getPositionSale();
            $users = User::isShow()
                ->leftJoin('sale_kpis', function ($join) {
                    $join->on('users.user_kpi_id', 'sale_kpis.sale_kpi_id')
                        ->where('sale_kpis.delete_flg', NO_DELETE)
                        ->where('sale_kpis.active_flg', ACTIVE);
                })
                ->where('position_id', $sale['position_id']);
            // if (isset($request_data['filters']['key_kind']) && $request_data['filters']['key_kind'] == 0 && isset($request_data['filters']['keywords'])) {
            //     $users = $users->where('sale_kpis.kpi_name', 'like', '%' . $request_data['keywords'] . '%');
            // } else {
            //     $users = $users->isFilters(isset($request_data['filters']) ? $request_data['filters'] : null);
            // }

            if (isset($request_data['filters']['keywords'])) {
                $users = $users->isFilters(isset($request_data['filters']) ? $request_data['filters'] : null);
            }

            if (isset($request_data['filters']['show_kind']) && in_array($request_data['filters']['show_kind'], [0, 1])) {
                if ($request_data['filters']['show_kind'] == 1) {
                    $users = $users->whereNotNull('users.user_kpi_id');
                } else {
                    $users = $users->whereNull('users.user_kpi_id');
                }
            }
            $users = $users->leftJoin('sales_members', function ($join) {
                $join->on('users.user_id', 'sales_members.user_id')
                    ->where('sales_members.delete_flg', NO_DELETE)
                    ->where('sales_members.active_flg', ACTIVE);
                return $join;
            });

            if (isset($request_data['filters']['show_user_team']) && in_array($request_data['filters']['show_user_team'], [0, 1, 2])) {

                if ($request_data['filters']['show_user_team'] == 2) {
                    $users = $users->whereNull('sales_members.member_kind');
                } else if ($request_data['filters']['show_user_team'] == SALE_LEADER) {
                    $users = $users->where('member_kind', SALE_LEADER);
                } else if ($request_data['filters']['show_user_team'] == SALE_MEMBER) {
                    $users = $users->where('member_kind', SALE_MEMBER);
                }
            }

            $choosed_id = isset($request_data['choosed_id']) ? $request_data['choosed_id'] : [];
            $is_get_all_user = isset($request_data['is_get_all_user']) ? json_decode($request_data['is_get_all_user']) : false;
            if (isset($request_data['is_get_ids']) && json_decode($request_data['is_get_ids'])) {
                $users = $users->pluck('users.user_id')->toArray();
                return response()->json([
                    "status" => HTTP_OK,
                    "keys" => $is_get_all_user ? $users : [],
                ]);
            }

            $users = $users->select(
                'users.user_contact_name',
                'users.user_code',
                'users.user_name',
                'users.user_id',
                'sale_kpis.sale_kpi_id',
                'sale_kpis.kpi_name',
                'sale_kpis.kpi_kind',
                'sales_members.user_id as sale_id',
            )
                ->isPaginate(isset($request_data['paginate']) ? $request_data['paginate'] : null);

            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "users" => $users->toArray()['data'],
                ]);
            } else {
                $html = view('manage.sale.ajax.kpi.table-users-kpi', compact('users', 'choosed_id', 'is_get_all_user'))->render();

                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function selectKpis(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sale_kpis = SaleKpi::isShow()->whereNull('user_id');

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['keywords']) && $filters['keywords'] != '') {
                    $sale_kpis = $sale_kpis->where('kpi_name', 'like', '%' . $filters['keywords'] . '%');
                }
            }


            if (isset($request->paginate)) {
                $sale_kpis = $sale_kpis->isPaginate($request->paginate);

                return response()->json([
                    "status" => HTTP_OK,
                    "sale_kpis" => $sale_kpis->toArray()['data'],
                ]);
            }
            return response()->json([
                "status" => HTTP_OK,
                "kpis" => $sale_kpis->get(),
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function connectKpiToSale(Request $request)
    {
        $this->validate(
            $request,
            [
                'sale_ids' => 'required',
                'kind' => 'required|numeric|min:1|max:2',
                'kpi_id' =>  $request->kind == 'connect' ?  'required|numeric|exists:sale_kpis,sale_kpi_id' : "",
            ]
        );

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            if (empty($request_data['sale_ids'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Vui lòng hãy chọn sale để liên kết');
            }
            DB::beginTransaction();
            $user = User::whereIn('user_id', $request_data['sale_ids']);
            if ($request_data['kind'] == 2) {
                $user = $user->update(['user_kpi_id' => $request_data['kpi_id']]);
            } else {
                $user = $user->update(['user_kpi_id' => null]);
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Liên kết thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
