<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Branch;
use App\Models\OrderPickup;
use App\Models\SaleLinkFwd;
use App\Models\Shipment;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\ShipperLocation;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Storage;

class OrderPickupController extends Controller
{
    public function displayIndex()
    {

        $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->get()->toArray();
        $vehicles = config('constans.constans.vehicles');
        $order_pickup_status = config('constans.constans.order_pickup_status');
        $fwds = User::isShow()->where('position_id', getPositionAccountSetting(SETTING_FWD_ACCOUNT));
        if (onCheckPositionAccount(SETTING_SALE_ACCOUNT, auth()->user()['position_id'])) {
            $fwd_link = SaleLinkFwd::isShow()->where('sale_id', auth()->id())->pluck('fwd_id');
            $fwds = $fwds->whereIn('user_id', $fwd_link);
        }
        $fwds = $fwds->get();
        return view('order-pickup.index', compact('branchs', 'vehicles', 'order_pickup_status', 'fwds'));
    }



    public function getOrderPickup(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? auth()->user() : $request->user();
            $filters = isset($request_data['filters']) ? $request_data['filters'] : [];
            $filters = is_string($filters) ? json_decode($filters, true) : $filters;

            $is_shipper = onCheckPositionAccount(SETTING_DRIVER_ACCOUNT, auth()->user()['position_id']);
            $is_ops_leader = onCheckPositionAccount(SETTING_OPS_LEADER_ACCOUNT, auth()->user()['position_id']);

            $orders_pickup = OrderPickup::isActive(ACTIVE)->isDelete(NO_DELETE)->isFilters($filters);


            $orders_pickup = $orders_pickup->leftJoin('shipper_locations', 'orders_pickup.shipper_location_id', 'shipper_locations.shipper_location_id');

            if (isset($filters['pickup_ship_status'])) {
                if ($filters['pickup_ship_status'] == 1) {
                    $orders_pickup = $orders_pickup->whereIn('order_pickup_status', [ORDER_PICKUP_STATUS_PENDING_CONFIRM, ORDER_PICKUP_STATUS_GOING, ORDER_PICKUP_STATUS_GONE, ORDER_PICKUP_STATUS_CONFIRMED]);
                } else if ($filters['pickup_ship_status'] == 2) {
                    $orders_pickup = $orders_pickup->where('order_pickup_status', ORDER_PICKUP_STATUS_PICKED);
                } else {
                    $orders_pickup = $orders_pickup->where('order_pickup_status', ORDER_PICKUP_STATUS_CANCEL);
                }
            }

            if ($is_shipper) {
                $orders_pickup = $orders_pickup->where('shipper_locations.shipper_id', $user->user_id);
            }

            if (onCheckPositionAccount(SETTING_SALE_ACCOUNT, $user['position_id'])) {
                $fwd_link = SaleLinkFwd::isShow()->where('sale_id', $user->user_id)->pluck('fwd_id');
                $orders_pickup = $orders_pickup->where(function ($query) use ($fwd_link, $user) {
                    return $query->whereIn('user_id', $fwd_link)->orWhere('sale_id', $user->user_id);
                });
            }

            if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, auth()->user()['position_id'])) {
                $filters['fwd_id'] = $user->user_id;
            }

            if (isset($filters['fwd_id'])) {
                $orders_pickup = $orders_pickup->where('fwd_id', $filters['fwd_id']);
            }

            $orders_pickup = $orders_pickup->with(['branch', 'user', 'shipper'])
                ->orderByDesc('orders_pickup.created_at');

         

            $excels = $orders_pickup->get()->toArray();

            $orders_pickup = $orders_pickup
                ->groupBy('orders_pickup.order_pickup_id')
                ->select(
                    'orders_pickup.*',
                    DB::raw(
                        'CASE 
                    WHEN order_pickup_status = 0 THEN "Đang chờ duyệt"
                    WHEN order_pickup_status = 1 THEN "Đang chờ xác nhận"
                    WHEN order_pickup_status = 2 THEN "Đã xác nhận"
                    WHEN order_pickup_status = 3 THEN "Đang đi lấy"
                    WHEN order_pickup_status = 4 THEN "Đã lấy"
                    WHEN order_pickup_status = 5 THEN "Đã pickup"
                    WHEN order_pickup_status = 6 THEN "Đã hủy"
                    ELSE "Khác"
                    END as order_pickup_status_label'
                    )
                )->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            $path = 'public/config/system-settings.json';
            $is_edit = in_array($user['position_id'], collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_EDIT_ORDER_PICKUP)['position_ids']);
            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'orders_pickup' => $orders_pickup,
                    'is_edit' => $is_edit,
                ]);
            }


            $html = view('order-pickup.order-pickup-pagination', compact('orders_pickup', 'is_edit', 'is_shipper', 'is_ops_leader'))->render();
            return response()->json([
                'status' => HTTP_OK,
                'html' => $html,
                'orders_pickup' => $excels,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleUpdateOrderPickup(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
        $request_data['order_pickup_id'] = isset($request_data['order_pickup_id']) ? json_decode($request_data['order_pickup_id']) : null;
        $order_pickup_id = $request_data['order_pickup_id'];
        $request_data['order_pickup_status'] = isset($request_data['order_pickup_status']) ? $request_data['order_pickup_status'] : ORDER_PICKUP_STATUS_WAITING;
        $user = $is_api ? auth()->user() : $request->user();

        $validate = [
            'order_pickup_type' => 'required|integer|min:0|max:2',
            'branch_id' => 'required|exists:branchs,branch_id',
            'order_pickup_date_time' => 'required',
            'order_pickup_gross_weight' => 'required|numeric|min:0',
            // 'fwd_id' => 'required|numeric|min:0',
            'order_pickup_number_packages' => 'required|numeric|min:1',
            'order_pickup_name' => 'required',
            'order_pickup_phone' => 'required|regex:/^[0-9]{10,11}$/',
            'order_pickup_address' => 'required|min:1|max:255',
            // 'latitude' => 'required|numeric',
            // 'longitude' => 'required|numeric',
            'order_pickup_awb' => 'required|min:1|max:255',
        ];

        if (!isset($request_data['latitude']) || !isset($request_data['longitude'])) {
            return returnResponse(HTTP_BAD_REQUEST, 'Chưa xác định vị trí trên bản đồ, vui lòng hãy lại địa chỉ', null, 'warning');
        }

        if (isset($request_data['order_pickup_id'])) {
            $validate['order_pickup_status'] = 'required|integer|min:0|max:6';
            if (in_array($request_data['order_pickup_status'], [ORDER_PICKUP_STATUS_GONE, ORDER_PICKUP_STATUS_PICKED])) {
                $check_image = OrderPickup::findOrFail($request_data['order_pickup_id'])->order_pickup_image;
                if (!isset($check_image) || $check_image == '') {
                    $validate['order_pickup_image'] = 'required';
                }
            }

            if ($request_data['order_pickup_status'] == ORDER_PICKUP_STATUS_CANCEL) {
                $validate['order_pickup_cancel_des'] = 'required';
            } else {
                $request_data['order_pickup_cancel_des'] = "";
            }
        } else {
            if (onCheckPositionAccount(SETTING_DRIVER_ACCOUNT, $user['position_id'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Shipper không thể tạo đơn', null, 'warning');
            }
        }
        $request->validate($validate);
        try {
            $res = $this->onUpdateOrCreateOrderPickup($request_data, $user, $request);
            if ($res['status'] != HTTP_OK) {
                return returnResponse($res['status'], $res['message'], null, 'warning');
            }
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Cập nhật order pickup thành công.',
                    'icon' => 'success',
                ]
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onUpdateOrCreateOrderPickup($request_data, $user, $request)
    {
        try {
            DB::beginTransaction();
            if (!isset($request_data['order_pickup_id'])) {
                $lastOrderPickup = OrderPickup::latest('order_pickup_id')->first();
                if ($lastOrderPickup) {
                    $lastOrderPickupCode = $lastOrderPickup['order_pickup_code'];
                    $order_pickup_code = 'KSN' . str_pad((int) substr($lastOrderPickupCode, 3) + 1, 7, '0', STR_PAD_LEFT);
                } else {
                    $order_pickup_code = 'KSN0000001';
                }
            } else {
                $old_pickup = OrderPickup::findOrFail($request_data['order_pickup_id']);
                if (isset($old_pickup['shipper_location_id'])) {
                    $shipper = ShipperLocation::where('shipper_location_id', $old_pickup['shipper_location_id'])->first();
                    if ($shipper && $shipper['shipper_id'] != auth()->id()) {
                        if (!in_array($old_pickup['order_pickup_status'], [ORDER_PICKUP_STATUS_WAITING, ORDER_PICKUP_STATUS_PENDING_CONFIRM])) {
                            return [
                                'status' => HTTP_BAD_REQUEST,
                                'message' => "Không thể cập nhật trạng thái khi shipper đã nhận đơn hàng"
                            ];
                        } else if (!in_array($request_data['order_pickup_status'], [ORDER_PICKUP_STATUS_WAITING, ORDER_PICKUP_STATUS_PENDING_CONFIRM])) {
                            return [
                                'status' => HTTP_BAD_REQUEST,
                                'message' => "Bạn chỉ có thể cập nhật trạng thái chờ duyệt/chờ xác nhận"
                            ];
                        }
                    } elseif ($shipper && $shipper['shipper_id'] == auth()->id()) {
                        if ($request_data['order_pickup_status'] < $old_pickup['order_pickup_status']) {
                            return [
                                'status' => HTTP_BAD_REQUEST,
                                'message' => "Bạn không thể quay ngược lại trạng thái trước đó"
                            ];
                        }
                    }
                }
                $order_pickup_code = $old_pickup['order_pickup_code'];
            }
            $request_data['order_pickup_code'] = $order_pickup_code;

            $sale_id = null;
            if (onCheckPositionAccount(SETTING_FWD_ACCOUNT, $user['position_id'])) {
                $request_data['fwd_id'] = $user['user_id'];
            }
            if (onCheckPositionAccount(SETTING_SALE_ACCOUNT, $user['position_id'])) {
                $sale_id = $user['user_id'];
            }
            if (isset($request_data['fwd_id'])) {
                $sale_link = SaleLinkFwd::isShow()->where('fwd_id', $request_data['fwd_id'])->first();

                $sale_id = $sale_link ? $sale_link->sale_id : null;
            }

            $data_pickup = [
                'user_id' => $old_pickup['user_id'] ?? $user['user_id'] ,
                'branch_id' => $request_data['branch_id'],
                'fwd_id' => isset($request_data['fwd_id']) ? $request_data['fwd_id'] : null,
                'sale_id' => $sale_id,
                'order_pickup_code' => $request_data['order_pickup_code'],
                'order_pickup_awb' => $request_data['order_pickup_awb'],
                'order_pickup_address' => $request_data['order_pickup_address'],
                'latitude' => $request_data['latitude'],
                'longitude' => $request_data['longitude'],
                'order_pickup_phone' => $request_data['order_pickup_phone'],
                'order_pickup_name' => isset($request_data['order_pickup_name']) ? $request_data['order_pickup_name'] : null,
                'order_pickup_note' => $request_data['order_pickup_note'],
                'order_pickup_type' => $request_data['order_pickup_type'],
                'order_pickup_date_time' => Carbon::parse($request_data['order_pickup_date_time'])->format('Y-m-d H:i:s'),
                'order_pickup_number_packages' => $request_data['order_pickup_number_packages'],
                'order_pickup_gross_weight' => $request_data['order_pickup_gross_weight'],
                'order_pickup_status' => $request_data['order_pickup_status'],
                'order_pickup_cancel_des' => $request_data['order_pickup_cancel_des'] ?? null,
                'order_pickup_image' => !isset($check_image) || $check_image == '' ? '' : $check_image,
            ];
            if ($request_data['order_pickup_status'] == ORDER_PICKUP_STATUS_WAITING) {
                $data_pickup['shipper_location_id'] = null;
            }
            $order_pickup = OrderPickup::updateOrCreate([
                'order_pickup_id' => isset($request_data['order_pickup_id']) ? $request_data['order_pickup_id'] : null,
            ], $data_pickup);

            if ($request->hasFile('order_pickup_image')) {
                $upload_path = public_path(PUBLIC_PATH_ORDER_PICKUP_IMAGE);
                if (!File::isDirectory($upload_path)) {
                    File::makeDirectory($upload_path, 0777, true, true);
                }
                if (File::exists(public_path($order_pickup->order_pickup_image))) {
                    File::delete(public_path($order_pickup->order_pickup_image));
                }
                $file_name = $order_pickup->order_pickup_code . '-' . $request->file('order_pickup_image')->getClientOriginalName();
                $request->file('order_pickup_image')->move($upload_path, $file_name);
                $file_path = PUBLIC_PATH_ORDER_PICKUP_IMAGE . '/' . $file_name;
                $order_pickup->order_pickup_image = $file_path;
            } else if (isset($request_data['order_pickup_image']) && strlen($request_data['order_pickup_image']) > 200) {
                if (isset($request_data['order_pickup_image'])) {
                    $file_name = $order_pickup->order_pickup_code . '-' . Str::random(30);
                    $order_pickup->order_pickup_image = convertImagesBase64ToDirectory([$request_data['order_pickup_image']], PUBLIC_PATH_ORDER_PICKUP_IMAGE, $file_name)[0];
                }
            }

            if (isset($request_data['order_pickup_status']) && !in_array($request_data['order_pickup_status'], [ORDER_PICKUP_STATUS_GONE, ORDER_PICKUP_STATUS_PICKED])) {
                $order_pickup->user_receiver_id = $user['user_id'];

                $order_pickup->order_pickup_date_time_method = Carbon::now()->format('Y-m-d H:i:s');
            }

            $order_pickup->save();

            DB::commit();

            return [
                'status' => HTTP_OK,
                'data' => $order_pickup
            ];
        } catch (\Throwable $th) {
            return [
                'status' => HTTP_BAD_REQUEST,
                'message' => "Xảy ra lỗi trên serve"
            ];
        }
    }

    public function getDetailOrderPickup(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? auth()->user() : $request->user();

            $path = 'public/config/system-settings.json';
            $is_edit = in_array($user['position_id'], collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_EDIT_ORDER_PICKUP)['position_ids']);
            // if (!$is_edit) {
            //     return returnResponse(HTTP_BAD_REQUEST, 'Bạn không có quyền truy cập', null, 'warning');
            // }

            $order_pickup = OrderPickup::isShow()->where('order_pickup_id', $request_data['order_pickup_id'])->with(['branch', 'user', 'shipper', 'fwd', 'sale', 'user_created'])->first();

            // $order_pickup_status = config('constans.constans.order_pickup_status');
            $data_order_pickup_status = config('constans.constans.order_pickup_status');
            $order_pickup_status = [];
            foreach ($data_order_pickup_status as $key => $value) {
                array_push($order_pickup_status, $value);
            }
            // unset($order_pickup_status[ORDER_PICKUP_STATUS_CANCEL]);
            $show_status = true;

            if (onCheckPositionAccounts([SETTING_FWD_ACCOUNT, SETTING_SALE_ACCOUNT], $user['position_id'])) {
                $show_status = false;
            }

            return response()->json([
                'status' => HTTP_OK,
                'show_status' => $show_status,
                'data' => $order_pickup,
                'order_pickup_status' => $order_pickup_status,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }


    public function handleDeleteOrderPickup(Request $request) // Đổi sang thành hủy lệnh
    {
        $this->validate($request, [
            'order_pickup_id' => 'required|numeric',
            'order_pickup_cancel_des' => 'required|max:500',
        ]);

        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? auth()->user() : $request->user();
            $path = 'public/config/system-settings.json';

            $is_edit = in_array($user['position_id'], collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ACCOUNT_EDIT_ORDER_PICKUP)['position_ids']);
            // if (!$is_edit || !isGetAllData()) {
            //     return showMessageError();
            // }

            $pickup = OrderPickup::isActive(ACTIVE)->isDelete(NO_DELETE)->findOrFail($request_data['order_pickup_id']);

            // if ($pickup && $pickup->shipper_location_id) {
            //     return returnResponse(HTTP_BAD_REQUEST, "Không thể hủy đơn hàng khi shipper đã nhận");
            // }

            OrderPickup::where('order_pickup_id', $pickup->order_pickup_id)->update([
                'order_pickup_status' => ORDER_PICKUP_STATUS_CANCEL,
                'order_pickup_cancel_des' => $request_data['order_pickup_cancel_des'],
            ]);

            $data_logs = [
                'old_data' => $pickup,
                'new_data' => $request_data
            ];
            onCreateAccountActivityLogs('Quản lý hệ thống', 'Hủy đơn order pickup ' . $pickup['order_pickup_code'], $data_logs);

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Đã hủy order pickup này thành công.',
                    'icon' => 'success',
                ]
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function getDetailPrintOrderPickup(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            $user = $is_api ? auth()->user() : $request->user();

            $path = 'public/config/system-settings.json';
            $system_settings_kinds = [SETTING_COMPANY_NAME, SETTING_HEADQUARTERS_ADDRESS, SETTING_EMAIL_ADDRESS, SETTING_WEBSITE_ADDRESS, SETTING_HOTLINE];
            $system_settings = json_decode(Storage::get($path), true);
            $system_settings = collect($system_settings)->whereIn('kind', $system_settings_kinds)->toArray();

            $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->get()->toArray();
            $order_pickup = OrderPickup::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('order_pickup_code', $request_data['order_pickup_code'])
                ->with([
                    'user' => function ($query) {
                        return $query->with(['position']);
                    },
                    'userReceiver'
                ])->firstOrFail();

            return response()->json([
                'status' => HTTP_OK,
                'data' => $order_pickup,
                'branchs' => $branchs,
                'system_settings' => $system_settings,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function displayDetailOrderPickup($order_pickup_code)
    {
        try {
            $path = 'public/config/system-settings.json';
            $system_settings_kinds = [SETTING_COMPANY_NAME, SETTING_HEADQUARTERS_ADDRESS, SETTING_EMAIL_ADDRESS, SETTING_WEBSITE_ADDRESS, SETTING_HOTLINE];
            $system_settings = json_decode(Storage::get($path), true);
            $system_settings = collect($system_settings)->whereIn('kind', $system_settings_kinds)->keyBy('kind')->toArray();

            $branchs = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)->get()->toArray();
            $order_pickup = OrderPickup::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('order_pickup_code', $order_pickup_code)
                ->with([
                    'user' => function ($query) {
                        return $query->with(['position']);
                    },
                    'userReceiver',
                    'sale'
                ])->firstOrFail();

            return view('order-pickup.inpickup', compact('branchs', 'order_pickup', 'system_settings'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }


    // danh sách shipper rảnh rỗi 
    public function showListShipperFree(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            $filters = isset($request_data['filters']) ? $request_data['filters'] : [];
            $filters = is_string($filters) ? json_decode($filters, true) : $filters;

            $filters['position_id'] = getPositionAccountSetting(SETTING_DRIVER_ACCOUNT);
            $shippers = User::isFilters($filters)->isShow()
                ->orderByDesc('updated_at');

            if (isset($request_data['limit']) && isset($request_data['page'])) {
                $shippers = $shippers->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);
            } else {
                $shippers = $shippers->get();
            }

            return response()->json([
                "status" => HTTP_OK,
                "shippers" => $shippers
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }


    // Lấy đơn hàng
    public function handleshipperTakePackage(Request $request)
    {
        $request->validate([
            'shipper_id' => 'required',
            'order_pickup_id' => 'required',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $order_pickup = OrderPickup::isShow()->where('order_pickup_id', $request_data['order_pickup_id'])->first();

            if (!onCheckPositionAccounts([SETTING_OPS_LEADER_ACCOUNT, SETTING_ADMIN_ACCOUNT], auth()->user()['position_id'])) {
                return returnResponse(HTTP_BAD_REQUEST, 'Chỉ có OPS trưởng mới có thể chỉ định shipper');
            }
            if (!$order_pickup) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đơn hàng');
            }

            if ($order_pickup->shipper_location_id) {
                return returnResponse(HTTP_BAD_REQUEST, 'Đơn hàng này đã có người nhận');
            }

            if ($order_pickup->order_pickup_status != ORDER_PICKUP_STATUS_WAITING) {
                return returnResponse(HTTP_BAD_REQUEST, 'Trạng thái này không thể thêm shipper hãy trả về chờ duyệt');
            }

            $shipper = ShipperLocation::isShow()->where('shipper_id', $request_data['shipper_id'])
                ->where('ship_status', PENDING)->first();

            DB::beginTransaction();
            if ($shipper) {
                if (
                    isset($request_data['shipper_longitude'])
                    && isset($request_data['shipper_latitude'])
                    && isset($request_data['location_address'])
                ) {
                    $shipper->shipper_longitude = $request_data['shipper_longitude'];
                    $shipper->shipper_latitude = $request_data['shipper_latitude'];
                    $shipper->location_address = $request_data['location_address'];
                }

                $order_pickup->save();
            } else {
                $branch = User::find($request_data['shipper_id'])
                    ->leftJoin('branchs', 'users.branch_id', 'branchs.branch_id')
                    ->select('branch_latitude', 'branch_longitude', 'branch_description')->first();

                $shipper = ShipperLocation::create([
                    "shipper_id" => $request_data['shipper_id'],
                    "shipper_longitude" => $request_data['shipper_longitude'] ?? $branch['branch_longitude'],
                    "shipper_latitude" => $request_data['shipper_latitude'] ?? $branch['branch_latitude'],
                    "location_address" => $request_data['location_address'] ?? $branch['branch_description'],
                ]);
            }
            $order_pickup->shipper_location_id = $shipper->shipper_location_id;
            $order_pickup->order_pickup_status = ORDER_PICKUP_STATUS_PENDING_CONFIRM;
            $order_pickup->user_receiver_id = $request_data['shipper_id'];
            $order_pickup->save();
            DB::commit();
            return response()->json([
                "status" => HTTP_OK,
                "message" => [
                    "icon" => "success",
                    "title" => "Thông báo",
                    "text" => "Nhận đơn thành công",
                ],
                "order_pickup" => $order_pickup
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onCancelShipper(Request $request)
    {

        $request->validate([
            'order_pickup_id' => 'required',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            OrderPickup::findOrFail($request_data['order_pickup_id'])->update([
                "order_pickup_status" => ORDER_PICKUP_STATUS_WAITING,
                "shipper_location_id" => null
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Đã xóa đơn khỏi shipper', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function onShipperStartPickup(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $shipper = User::findOrFail($request_data['shipper_id']);
            if ($shipper) {
                if ($shipper->position_id != getPositionAccountSetting(SETTING_DRIVER_ACCOUNT)) {
                    return returnResponse(HTTP_BAD_REQUEST, 'Bạn không phải là shipper');
                }

                DB::beginTransaction();
                OrderPickup::isShow()
                    ->leftJoin('shipper_locations', 'orders_pickup.shipper_location_id', 'shipper_locations.shipper_location_id')
                    ->where('shipper_locations.shipper_id', $request_data['shipper_id'])
                    ->where('shipper_locations.ship_status', PENDING)
                    ->where('orders_pickup.order_pickup_status', ORDER_PICKUP_STATUS_PENDING_CONFIRM)
                    ->update(["order_pickup_status" => ORDER_PICKUP_STATUS_GOING]);

                DB::commit();
                return returnResponse(HTTP_OK, 'Bắt đầu đi lấy đơn', 'Thông báo', 'success');
            } else {
                return returnResponse(HTTP_BAD_REQUEST, 'Không tìm thấy shipper');
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    // Cập nhật vị trí của shipper
    public function handleUpdateShipper(Request $request)
    {
        $this->validate($request, [
            // 'shipper_location_id' => 'required|numeric',
            'shipper_longitude' => 'required|max:225',
            'shipper_latitude' => 'required|max:225',
            'location_address' => 'required|max:225',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $user = $is_api ? $request->user() : auth()->user();
            $shipper = ShipperLocation::isShow()
                ->where('shipper_id', $user->user_id)
                ->where('ship_status', 0)->first();

            // $shipper = ShipperLocation::isShow()->where('shipper_location_id', $request_data['shipper_location_id'])->first();

            if (!$shipper) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy hoạt động của shipper này');
            }

            DB::beginTransaction();
            $shipper->update([
                "shipper_longitude" => $request_data['shipper_longitude'],
                "shipper_latitude" => $request_data['shipper_latitude'],
                "location_address" => $request_data['location_address'],
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onfinishedshipper(Request $request)
    {

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $user = $is_api ? $request->user() : auth()->user();

            $shipper = ShipperLocation::isShow()
                ->where('shipper_id', $user->user_id)
                ->where('ship_status', 0)->first();

            if (!$shipper) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy hoạt động của shipper này');
            }

            $check_finish = OrderPickup::isShow()->where('shipper_location_id', $shipper->shipper_location_id)
                ->whereIn('order_pickup_status', [ORDER_PICKUP_STATUS_WAITING, ORDER_PICKUP_STATUS_PENDING_CONFIRM, ORDER_PICKUP_STATUS_GOING, ORDER_PICKUP_STATUS_GONE])
                ->exists();
            if ($check_finish) {
                return returnResponse(HTTP_NOT_FOUND, 'Bạn vẫn còn đơn hàng chưa hoàn thành');
            }

            if (empty($request_data['shipper_longitude']) || empty($request_data['shipper_latitude'] || empty($request_data['location_address']))) {
                $branch = Branch::isShow()->leftJoin('users', 'branchs.branch_id', 'users.branch_id')
                    ->where('user_id', $shipper->shipper_id)->first();

                $request_data['shipper_longitude'] = $branch->branch_longitude;
                $request_data['shipper_latitude'] = $branch->branch_latitude;
                $request_data['location_address'] = $branch->branch_description;
            }

            DB::beginTransaction();
            $shipper->update([
                "shipper_longitude" => $request_data['shipper_longitude'],
                "shipper_latitude" => $request_data['shipper_latitude'],
                "location_address" => $request_data['location_address'],
                "ship_status" => SUCCESS
            ]);

            DB::commit();
            return returnResponse(HTTP_OK, 'Xác nhận thành công', 'Thông báo', 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function showListHistoryshipper(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $shippers = OrderPickup::isShow()
                ->leftJoin('users', 'shipper_locations.shipper_id', 'users.user_id')
                ->select(
                    'shipper_id',
                    'user_contact_name',
                    'user_code',
                    'shipper_longitude',
                    'shipper_latitude',
                    'location_address',
                    'ship_status',
                    'shipper_locations.shipper_location_id',
                    'shipper_locations.created_at',
                    'shipper_locations.updated_at',
                    DB::raw(
                        'CASE 
                        WHEN ship_status = 0 THEN "Đang hoạt động"
                        WHEN ship_status = 1 THEN "Đã xong"
                        ELSE "Khác"
                        END as ship_status_label'
                    )
                );

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['keywords'])) {
                    $shippers = $shippers->where('user_contact_name', 'like', '%' . $filters['keywords'] . '%');
                }
            }

            if (isset($request_data['shipper_id'])) {
                $shippers = $shippers->where('shipper_id', $request_data['shipper_id']);
            }

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $shippers->isPaginate($request->paginate)->toArray()['data'],
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleUpdateStatusOrderPickup(Request $request)
    {
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $order_pickup = OrderPickup::findOrFail($request_data['order_pickup_id']);
            // where('order_pickup_status', $request_data['order_pickup_status'] - 1)

            if ($request->hasFile('order_pickup_image')) {
                $upload_path = public_path(PUBLIC_PATH_ORDER_PICKUP_IMAGE);
                if (!File::isDirectory($upload_path)) {
                    File::makeDirectory($upload_path, 0777, true, true);
                }
                if (File::exists(public_path($order_pickup->order_pickup_image))) {
                    File::delete(public_path($order_pickup->order_pickup_image));
                }
                $file_name = $order_pickup->order_pickup_code . '-' . $request->file('order_pickup_image')->getClientOriginalName();
                $request->file('order_pickup_image')->move($upload_path, $file_name);
                $file_path = PUBLIC_PATH_ORDER_PICKUP_IMAGE . '/' . $file_name;
                $order_pickup->order_pickup_image = $file_path;
            } else if (isset($request_data['order_pickup_image']) && strlen($request_data['order_pickup_image']) > 200) {
                if (isset($request_data['order_pickup_image'])) {
                    $file_name = $order_pickup->order_pickup_code . '-' . Str::random(30);
                    $order_pickup->order_pickup_image = convertImagesBase64ToDirectory([$request_data['order_pickup_image']], PUBLIC_PATH_ORDER_PICKUP_IMAGE, $file_name)[0];
                }
            }

            $order_pickup->order_pickup_status = $request_data['order_pickup_status'];
            $order_pickup->save();

            DB::commit();
            // return response()->json([
            //     'status' => HTTP_OK,
            //     'message' => [
            //         'title' => 'Thành công!',
            //         'text' => 'Cập nhật trạng thái order pickup thành công.',
            //         'icon' => 'success',
            //     ]
            // ], HTTP_OK);
            return returnResponse(HTTP_OK, 'Cập nhật trạng thái order pickup thành công.', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
