<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Branch;
use App\Models\BagCode;
use App\Models\Package;
use App\Models\Service;
use App\Models\Shipment;
use App\Utils\AfterShip;
use App\Models\PackageSg;
use App\Exports\ExportFile;
use App\Models\BagCodeItem;
use App\Models\HistoryScan;
use Illuminate\Http\Request;
use App\Models\PackageTracking;
use App\Models\ShipmentTracktry;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\ShipmentTrackingStatus;

class MawbManagerController extends Controller
{
    public function displayIndex()
    {
        $services = Service::isActive(ACTIVE)->isDelete(NO_DELETE)->get();
        $branchs = Branch::isShow()->get();
        return view('mawb-manager.index', compact('services', 'branchs'));
    }

    public function showDetailSmTracking(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $sm_tracking = ShipmentTracktry::find($request_data['sm_tracktry_id']);

            if ($sm_tracking) {
                return response()->json([
                    "status" => HTTP_OK,
                    'sm_tracking' => $sm_tracking
                ]);
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Đối tượng không tồn tại');
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function onShowHawb(Request $request)
    {

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : false;
            if (!isset($request_data['package_code'])) {
                return response()->json([
                    'status' => HTTP_BAD_REQUEST,
                    'message' => 'Không được để trống',
                ]);
            }
            $packages = Package::isShow()->where('package_code', 'like', '%' . $request_data['package_code'] . '%')
                ->whereNotNull('packages.sm_tracktry_id')
                ->leftJoin('shipments_tracktry', 'packages.sm_tracktry_id', 'shipments_tracktry.sm_tracktry_id')
                ->select('package_code', 'awb_code')->get();

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'packages' => $packages,
                ]);
            } else {
                $html = view('mawb-manager.ajax.modal-hawb', compact('packages'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => HTTP_BAD_REQUEST,
                'message' => 'Lỗi không xác định'  . $th->getMessage(),
            ]);
        }
    }

    public function handleCreateSMTracktry(Request $request)
    {
        $this->validate($request, [
            'awb_code' => 'required|max:224|unique:shipments_tracktry,awb_code',
            'service' => 'required|max:224',
            'branch_id' => 'required',
        ]);
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        try {
            DB::beginTransaction();

            ShipmentTracktry::create([
                "awb_code" => $request_data['awb_code'],
                "hawb_no" => isset($request_data['hawb_no']) ? $request_data['hawb_no'] : '',
                "service" => $request_data['service'],
                "service_ids" => $request_data['service_ids'],
                "branch_id" => $request_data['branch_id'],
                "airline" => isset($request_data['airline']) ? $request_data['airline'] : '',
                "dest" => isset($request_data['dest']) ? $request_data['dest'] : '',
                // Load, Flight, ETA, ETD'
                'load' => isset($request_data['load']) ? $request_data['load'] : '',
                'flight' => isset($request_data['flight']) ? $request_data['flight'] : '',
                'eta' => isset($request_data['eta']) ? $request_data['eta'] : '',
                'etd' => isset($request_data['etd']) ? $request_data['etd'] : '',
            ]);

            DB::commit();

            return returnResponse(HTTP_OK, 'Khởi tạo thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleUpdateSMTracktry(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        $this->validate($request, [
            'awb_code' => 'required|max:224|unique:shipments_tracktry,awb_code,' . $request_data['sm_tracktry_id'] . ',sm_tracktry_id',
            'service' => 'required|max:224',
            'branch_id' => 'required',
        ]);


        try {
            $sm_shipment = ShipmentTracktry::find($request_data['sm_tracktry_id']);
            if ($sm_shipment) {
                DB::beginTransaction();

                $sm_shipment->update([
                    "awb_code" => $request_data['awb_code'],
                    "hawb_no" => isset($request_data['hawb_no']) ? $request_data['hawb_no'] : '',
                    "service_ids" => $request_data['service_ids'],
                    "service" => $request_data['service'],
                    "branch_id" => $request_data['branch_id'],
                    "partner" => isset($request_data['partner']) ? $request_data['partner'] : '',
                    "airline" => isset($request_data['airline']) ? $request_data['airline'] : '',
                    "dest" => isset($request_data['dest']) ? $request_data['dest'] : '',
                    // Load, Flight, ETA, ETD'
                    'load' => isset($request_data['load']) ? $request_data['load'] : '',
                    'flight' => isset($request_data['flight']) ? $request_data['flight'] : '',
                    'eta' => isset($request_data['eta']) ? $request_data['eta'] : '',
                    'etd' => isset($request_data['etd']) ? $request_data['etd'] : '',

                ]);
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng cập nhật');
            }

            DB::commit();

            return returnResponse(HTTP_OK, 'Khởi tạo thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function showListSmTracktry(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        $filters = $request_data['filters'] ?? null;

        try {
            // Base query
            $shipments_tracktry = ShipmentTracktry::isShow()
                ->leftJoin('packages', 'shipments_tracktry.sm_tracktry_id', '=', 'packages.sm_tracktry_id')
                ->leftJoin('shipment_tracking_status', 'shipments_tracktry.sm_tracktry_id', '=', 'shipment_tracking_status.shipment_tracktry_id')
                ->leftJoin(DB::raw('(
                            SELECT 
                                shipment_tracktry_id, 
                                MAX(IF(tracking_status = 1, "Release", "")) AS tracking_status_label
                            FROM shipment_tracking_status
                            WHERE delete_flg = ' . NO_DELETE . ' AND active_flg = ' . ACTIVE . '
                            GROUP BY shipment_tracktry_id
                        ) as tracking_status_summary'), 'shipments_tracktry.sm_tracktry_id', '=', 'tracking_status_summary.shipment_tracktry_id')
                ->with(['branch']);

            // Apply filters if available
            if (!empty($filters['keywords'])) {
                $keywords = $filters['keywords'];
                $shipments_tracktry = $shipments_tracktry->where(function ($query) use ($keywords) {
                    $query->where("shipments_tracktry.awb_code", 'like', '%' . $keywords . '%')
                        ->orWhere("shipments_tracktry.dest", 'like', '%' . $keywords . '%')
                        ->orWhere("shipments_tracktry.partner", 'like', '%' . $keywords . '%')
                        ->orWhere("shipments_tracktry.service", 'like', '%' . $keywords . '%')
                        ->orWhere("packages.package_code", 'like', '%' . $keywords . '%')
                        ->orWhere("packages.package_tracking_code", 'like', '%' . $keywords . '%');
                });
            }

            if (!empty($filters['date_range'])) {
                $start_date = Carbon::parse($filters['date_range']['start_date'])->format('Y-m-d H:i:s');
                $end_date = Carbon::parse($filters['date_range']['end_date'])->endOfDay()->format('Y-m-d H:i:s');
                $shipments_tracktry = $shipments_tracktry->whereBetween('shipments_tracktry.created_at', [$start_date, $end_date]);
            }

            if (isset($filters['tracking_status']) && in_array($filters['tracking_status'], [0, 1])) {
                $shipments_tracktry = $shipments_tracktry->where('shipment_tracking_status.tracking_status', $filters['tracking_status']);
            }

            // Select columns
            $shipments_tracktry = $shipments_tracktry->select(
                'shipments_tracktry.*',
                // 'package_summary.count_package',
                // 'package_summary.count_bag',
                // 'package_summary.gross_weight',
                // 'package_summary.charged_weight',
                'tracking_status_summary.tracking_status_label'
            );

            // Group by and order
            $shipments_tracktry = $shipments_tracktry->groupBy('shipments_tracktry.sm_tracktry_id')
                ->orderBy('shipments_tracktry.created_at', 'desc');

            // Apply pagination if required
            $paginate = $request_data['paginate'] ?? null;
            $shipments_tracktry = $shipments_tracktry->isPaginate($paginate);

            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "shipments_tracktry" => $shipments_tracktry->toArray()['data'],
                ]);
            } else {
                if (isset($request_data['template'])) {
                    $html = view('mawb-manager.ajax.tables-nomal', compact('shipments_tracktry'))->render();
                } else {
                    $html = view('mawb-manager.ajax.tables-sm-tracktry', compact('shipments_tracktry'))->render();
                }
                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function exportRpPackage($sm_tracktry_id)
    {
        try {
            $data = [];
            $viewName = 'mawb-manager.excels.excel_rp';
            $others = null;
            $column_widths = [
                'A' => 10,
                'B' => 30,
                'C' => 30,
                'D' => 30,
                'E' => 30,
                'F' => 20,
                'G' => 20,
                'H' => 20,
                'I' => 20,
                'J' => 20,
                'K' => 20,
                'L' => 100,
                'M' => 40,
                'N' => 80,
                'O' => 10,
            ];
            $sm_tracking = ShipmentTracktry::find($sm_tracktry_id);
            $name_excel = 'report MAWB(' . $sm_tracking->awb_code . ').xlsx';
            $packages = Package::isShow()->where('sm_tracktry_id', $sm_tracktry_id)->with('packages_sg');
            $packages = $packages
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->leftJoin('branchs', 'shipments.shipment_branch_id', 'branchs.branch_id')
                ->select(
                    'shipments.shipment_code',
                    'packages.package_id',
                    'packages.package_code',
                    'packages.package_tracking_code',
                    'services.service_name',
                    'branchs.branch_name',
                    'packages.package_weight',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_weight_actual 
                    //     ELSE packages.package_weight 
                    // END as package_weight'),
                    'packages.package_length',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_length_actual 
                    //     ELSE packages.package_length
                    // END as package_length'),
                    'packages.package_width',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_width_actual 
                    //     ELSE packages.package_width
                    // END as package_width'),
                    'packages.package_height',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_height_actual 
                    //     ELSE packages.package_height
                    // END as package_height'),
                    'packages.package_charged_weight',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_charged_weight_actual 
                    //     ELSE packages.package_charged_weight
                    // END as package_charged_weight'),
                    // phụ thu
                    'packages.package_image',
                    'shipments.sender_company_name',
                    'shipments.receiver_company_name',
                    'shipments.receiver_contact_name',
                );
            $data = $packages->groupBy('packages.package_id')->get();
            return Excel::download(new ExportFile($data, $viewName, $others, $column_widths), $name_excel);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function exportPkgPackage($sm_tracktry_id)
    {
        try {
            $data = [];
            $viewName = 'mawb-manager.excels.excel_pkg';
            $others = null;
            $column_widths = [
                'A' => 10,
                'B' => 30,
                'C' => 30,
                'D' => 30,
                'E' => 30,
                'F' => 20,
                'G' => 20,
                'H' => 20,
                'I' => 20,
            ];
            $sm_tracking = ShipmentTracktry::find($sm_tracktry_id);
            $name_excel = 'PACKAGE LIST MAWB(' . $sm_tracking->awb_code . ').xlsx';
            $packages = Package::isShow()->where('sm_tracktry_id', $sm_tracktry_id)->with('packages_sg');
            $packages = $packages
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->select(
                    'shipments.shipment_code',
                    'packages.package_code',
                    'packages.package_tracking_code',
                    'packages.bag_code',
                    'packages.package_weight',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_weight_actual 
                    //     ELSE packages.package_weight
                    // END as package_weight'),
                    'packages.package_length',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_length_actual 
                    //     ELSE packages.package_length
                    // END as package_length'),
                    'packages.package_width',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_width_actual 
                    //     ELSE packages.package_width
                    // END as package_width'),
                    'packages.package_height',
                    // DB::raw('CASE 
                    //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                    //     THEN packages.package_height_actual 
                    //     ELSE packages.package_height
                    // END as package_height'),
                );
            $data = $packages->groupBy('packages.package_id')->get();
            return Excel::download(new ExportFile($data, $viewName, $others, $column_widths), $name_excel);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function hidenSMTracktry(Request $request)
    {

        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        try {
            DB::beginTransaction();

            $sm_shipment = ShipmentTracktry::find($request_data['sm_tracktry_id']);
            
            $data_logs = [
                'old_data' => null,
                'new_data' => $request_data
            ];
            onCreateAccountActivityLogs('Mawb manager', 'Ẩn mawb ' . $sm_shipment['awb_code'], $data_logs);

            if ($sm_shipment) {
                $sm_shipment->update(["active_flg" => INACTIVE]);
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng cập nhật');
            }

            DB::commit();
            return returnResponse(HTTP_OK, 'Ẩn đối tượng thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function printMawb($awb_code)
    {
        try {
            $shipment_tractry = ShipmentTracktry::isShow()->where('awb_code', $awb_code)
                ->leftJoin('branchs', 'shipments_tracktry.branch_id', 'branchs.branch_id')
                ->leftJoin('packages', 'shipments_tracktry.sm_tracktry_id', 'packages.sm_tracktry_id')
                ->select(
                    'shipments_tracktry.*',
                    'branch_name',
                    DB::raw('COUNT(packages.package_id) as count_package'),
                    DB::raw('SUM(packages.package_weight) as gross_weight'),
                    DB::raw('SUM(packages.package_charged_weight) as charged_weight'),
                    DB::raw('SUM(DISTINCT packages.bag_code) as bag_code')
                )->first();
            // $total_packages = Package::isShow()->where('sm_tracktry_id', $shipment_tractry->sm_tracktry_id)
            //     ->selectRaw('COUNT(*) as total_packages')->value('total_packages');

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            $barcode = $generator->getBarcode($shipment_tractry->awb_code, $generator::TYPE_CODE_128, 2, 70);

            return view('mawb-manager.print.mawb', compact('shipment_tractry', 'barcode'));
        } catch (\Throwable $th) {
            abort('404');
        }
    }

    public function addTracking($awb_code)
    {
        $shipment_tractry = ShipmentTracktry::isShow()->where('awb_code', $awb_code)->first();

        if ($shipment_tractry) {

            $histories = [];

            // $packages = Package::isShow()->where('sm_tracktry_id', $shipment_tractry->sm_tracktry_id)->get();
            // foreach ($packages as $key => $value) {
            //     if (isset($value->package_tracking_code) && isset($value->carrier_code)) {
            //         $track =  getTracktry($value->package_tracking_code, $value->carrier_code);
            //         if ($track['status'] && !empty($track['items'][0]['destination_info']['trackinfo'])) {
            //             $trackinfo = $track['items'][0]['destination_info']['trackinfo'];
            //             foreach ($trackinfo as $key => $value) {
            //                 $_data_tr = [
            //                     "date" => $value['Date'],
            //                     "address" => $value['Details'],
            //                     "status" => $value['checkpoint_status'],
            //                     "note" => $value['StatusDescription'],
            //                 ];
            //                 array_push($histories, $_data_tr);
            //             }
            //         }
            //     }
            // }
            return view('mawb-manager.add-tracking', compact('shipment_tractry', 'histories'));
        } else {
            abort('404');
        }
    }

    public function showListTrackingStatus(Request $request)
    {

        $this->validate($request, [
            "shipment_tracktry_id" => "required|numeric",
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $trackings_status = ShipmentTrackingStatus::isShow()
                ->where('shipment_tracktry_id', $request_data['shipment_tracktry_id'])
                ->orderBy('tracking_status_date', 'desc')->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $trackings_status->toArray()['data'],
                ]);
            } else {
                $html = view('mawb-manager.ajax.table-tracking-status', compact('trackings_status'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleShipmentStatusCreate(Request $request)
    {
        $this->validate($request, [
            'shipment_tracktry_id' => 'required|numeric',
            'tracking_status_date' => 'required|date',
            'tracking_status_address' => 'required|max:255',
            'tracking_status_note' => 'required|max:255',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            if (isset($request_data['shipment_tracktry_id'])) {
                $sm = ShipmentTracktry::where('sm_tracktry_id', $request_data['shipment_tracktry_id'])->isShow()->first();
                if (!$sm) {
                    return returnResponse(HTTP_NOT_FOUND, 'Mã mawb không tồn tại');
                }
            }
            DB::beginTransaction();

            $status = [
                'shipment_tracktry_id' => $request_data['shipment_tracktry_id'],
                'tracking_status_date' => Carbon::parse($request_data['tracking_status_date'])->format('Y-m-d H:i:s'),
                'tracking_status_address' => $request_data['tracking_status_address'],
                'tracking_status_note' => $request_data['tracking_status_note'],
            ];
            if (strtolower($request_data['tracking_status_note']) == strtolower("Destination Customs Released")) {
                $status['tracking_status'] = 1;
                // ShipmentTracktry::find($rquest_data['shipment_tracktry_id'])->update(['tracking_status' => 1]);
            }
            ShipmentTrackingStatus::create($status);
            DB::commit();
            return returnResponse(HTTP_OK, 'Thêm mới trạng thái thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function handleShipmentStatusDelete(Request $request)
    {
        $this->validate($request, [
            "tracking_status_id" => "required|numeric",
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            $tracking_status = ShipmentTrackingStatus::where('tracking_status_id', $request_data['tracking_status_id'])->isShow()->first();
            if (!$tracking_status) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy trạng thái này');
            }
            $package_ids = Package::isShow()
                ->where('sm_tracktry_id', $tracking_status['shipment_tracktry_id'])
                ->pluck('package_code');

            DB::beginTransaction();
            PackageTracking::isShow()
                ->whereIn('package_code', $package_ids)
                ->where('package_tracking_date', $tracking_status['tracking_status_date'])
                ->where('package_tracking_address', $tracking_status['tracking_status_address'])
                ->where('package_tracking_note', $tracking_status['tracking_status_note'])
                ->update(["delete_flg" => DELETED]);
            $tracking_status->update(["delete_flg" => DELETED]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa trạng thái tracking thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onUpdateStatusPackageByStatusTraking(Request $request)
    {
        try {
            $this->validate($request, [
                'shipment_tracktry_id' => 'required|numeric'
            ]);
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;


            $status_trackings = ShipmentTrackingStatus::isShow()
                ->where('shipment_tracktry_id', $request_data['shipment_tracktry_id'])
                ->select('tracking_status_date', 'tracking_status_note', 'tracking_status_address')->get()->toArray();
            if (empty($status_trackings)) {
                return returnResponse(HTTP_BAD_REQUEST, 'Trạng thái tracking của bạn đang trống', null, 'warning');
            }

            $packages = Package::isShow()->where('packages.sm_tracktry_id', $request_data['shipment_tracktry_id'])
                ->select('package_id', 'package_code')->get();

            DB::beginTransaction();

            foreach ($status_trackings as $key => $value) {
                // 
                $data_create = [
                    'package_tracking_date' => $value['tracking_status_date'],
                    'package_tracking_address' => $value['tracking_status_address'],
                    'package_tracking_note' => $value['tracking_status_note'],
                ];
                if (strtolower($value['tracking_status_note']) == strtolower("Destination Customs Released") || strtolower($value['tracking_status_note']) == strtolower("ORIGIN CUSTOMS RELEASED")) {
                    $data_create['package_tracking_status'] = 1;
                }
                // 
                foreach ($packages as $key => $package) {
                    $new_query =  null;
                    $new_query = PackageTracking::where('package_code', $package['package_code'])
                        // ->where('package_tracking_date', $value['tracking_status_date'])
                        // ->where('package_tracking_address', $value['tracking_status_address'])
                        ->where('package_tracking_note', $value['tracking_status_note'])
                        ->isShow()->exists();
                    if (!$new_query) {
                        $data_create['package_code'] = $package['package_code'];
                        PackageTracking::create($data_create);
                    }
                }
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật trạng thái cho hawb thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function updateStatusTrackingShipment(Request $request)
    {
        $this->validate($request, ['sm_tracktry_id' => 'required|numeric']);

        $request_data = $request->all();

        $locations = [];
        try {
            $packages = Package::isShow()->where('sm_tracktry_id', $request_data['sm_tracktry_id'])->first();
            if ($packages && $packages->tracking_code) {
                $tracking = getTrackingAfterShip($packages->tracking_code);
                if ($tracking['status'] == HTTP_OK) {
                    # code...
                }
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function billShipment($awb_code)
    {
        $shipment_tractry = ShipmentTracktry::isShow()->where('awb_code', $awb_code)->first();

        if ($shipment_tractry) {
            return view('mawb-manager.bill-shipment', compact('shipment_tractry'));
        } else {
            abort('404');
        }
    }

    public function showListPackageMawb(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            $packages = Package::isShow()->where('sm_tracktry_id', $request_data['sm_tracktry_id']);
            $packages = $packages
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                // ->leftJoin('history_scan', 'packages.package_id', 'history_scan.package_id')
                ->leftJoin('history_scan', function ($join) {
                    $join->on('packages.package_id', '=', 'history_scan.package_id')
                        ->where('packages.delete_flg', '=', NO_DELETE)
                        ->where('packages.active_flg', '=', ACTIVE);
                })
                ->leftJoin('users', 'history_scan.scan_by', 'users.user_id')
                ->select(
                    'packages.*',
                    'shipments.sender_company_name',
                    'services.promotion_flg',
                    'shipments.receiver_contact_name',
                    'users.user_contact_name as staff_scan'
                );

            if (isset($request_data['filters'])) {
                $filters = $request_data['filters'];
                if (isset($filters['filter_by'])) {
                    $packages = $packages->where($filters['filter_by'], 'like', '%' . $filters['keywords'] . '%');
                } else {
                    $packages = $packages->where(function ($query) use ($filters) {
                        $query->where("shipments.sender_company_name", 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere("shipments.receiver_contact_name", 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere("packages.package_hawb_code", 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere("packages.package_tracking_code", 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere("packages.bag_code", 'like', '%' . $filters['keywords'] . '%')
                            ->orWhere("users.user_contact_name", 'like', '%' . $filters['keywords'] . '%');
                    });
                }
                // if (isset($filters['filter_by'])) {
                //     $packages = $packages->where('', 'like', '%' . $filters['keywords'] . '%');
                // }
            }

            $packages = $packages->groupBy('packages.package_id');

            $packages = $packages->isPaginate(isset($request_data['paginate']) ? $request_data['paginate'] : null);

            if ($is_api) {
                return response()->json([
                    "status" => HTTP_OK,
                    "packages" => $packages->toArray()['data'],
                ]);
            } else {
                $html = view('mawb-manager.ajax.tables-package', compact('packages'))->render();
                return response()->json([
                    "status" => HTTP_OK,
                    "html" => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function printPackageBagCode(Request $request, $item_code)
    {
        try {

            $bag_codes = BagCodeItem::isShow()->where('item_code', $item_code)->get();
            if (count($bag_codes) < 1) {
                return abort('404');
            }
            $bag_code_id = $bag_codes[0]['bag_code_id'];
            $bag_code = BagCode::findOrFail($bag_code_id);
            if (!$bag_code) {
                return abort('404');
            }

            $generator = new \Picqer\Barcode\BarcodeGeneratorHTML();
            if ($bag_code['bag_code_kind'] == 0) {
                foreach ($bag_codes as $key => $value) {
                    $bag_codes[$key]['barcode'] = $generator->getBarcode($value['item_code'], $generator::TYPE_CODE_128, 2, 70);
                }
                return view('scan-system.ajax.print.bag-code', compact('bag_codes', 'bag_code'));
            } else {
                $bag_code['item_code'] = $bag_codes[0]['item_code'];
                $bag_code['barcode'] = $generator->getBarcode($bag_code['item_code'], $generator::TYPE_CODE_128, 2, 70);
                return view('scan-system.ajax.print.bag-code-uk', compact('bag_code'));
            }
        } catch (\Throwable $th) {
            return abort('404');
        }
    }


    public function updatePackagesMawbByTrackTry(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            $country_iso3 = config('constans.country_iso3');
            DB::beginTransaction();
            $excels = $request_data['excels'];
            $text = '';
            foreach ($excels as $key => $value) {
                $package = null;
                // $package = Package::where('package_hawb_code', $value['hawb'])
                //     ->where('sm_tracktry_id', $request_data['sm_tracktry_id'])
                //     ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')->first();
                $package = Package::where('package_hawb_code', $value['hawb'])->first();
                $not_find_packages = '';
                if ($package) {
                    $shipment = Shipment::with([
                        'branch',
                        'country',
                        'city',
                        'state',
                    ])->where('shipment_id', $package['shipment_id'])->first();
                    $package_data = [
                        "package_tracking_code" => $value['tracking_code'],
                    ];

                    if (isset($request_data['create_tracking']) && json_decode($request_data['create_tracking'])) {
                        // $data_tracking = [
                        //     'customer_name' => $package->receiver_contact_name,
                        //     'customer_phone' => $package->receiver_telephone,
                        //     'order_create_time' => Carbon::now()->toDateTimeString(),
                        // ];
                        // $result = createTracktry($value['tracking_code'], $data_tracking);
                        // if (!$result['status']) {
                        //     $text = $text . ' Khởi tạo mã hawb: ' . $value['hawb'] . ', tracking_code: ' . $value['tracking_code'] . ' thất bại do ' . $result['message'] . ' ';
                        //     $package_data['package_tracking_code'] = '';
                        // } else {
                        //     $package_data['carrier_code'] = $result['data']['carrier_code'];
                        //     $package_data['branch_connect'] = $result['data']['branch_connect'];
                        // };
                        $data_tracking = [
                            // Tracktry
                            "customer_name" => $shipment['receiver_contact_name'] ?? "",
                            "customer_email" => '',
                            "customer_phone" => $shipment['receiver_telephone'] ?? '',
                            "order_create_time" => $package['created_at'] ?? Carbon::now()->toDateTimeString(),
                            "tracking_postal_code" => $shipment['receiver_postal_code'] ?? '',

                            // Aftership
                            "slug" => "",
                            "tracking_number" => $value['tracking_code'],
                            "title" => $shipment->shipment_goods_name ?? 'product by achau',
                            "smses" => [
                                $shipment->receiver_telephone,
                                $shipment->sender_telephone,
                            ],
                            "emails" => [],
                            "order_id" => $package->package_code,
                            "order_number" => $package->package_id,
                            "order_id_path" => "",
                            "custom_fields" => [
                                "product_name" => "",
                                "product_price" => ""
                            ],
                            "language" => "en",
                            "order_promised_delivery_date" => Carbon::now()->format('Y-m-d'),
                            "delivery_type" => "pickup_at_store",
                            "pickup_location" => "achau Post",
                            "pickup_note" => "achau",
                            "origin_country_iso3" => "VNM",
                            "origin_state" => getAreaVietName($shipment->sender_city)['districts'][$shipment->sender_district] ?? '',
                            "origin_city" => getAreaVietName()['cities'][$shipment->sender_city] ?? '',
                            "origin_postal_code" => "",
                            "origin_raw_location" => $shipment->sender_address,
                            "destination_country_iso3" => $country_iso3[$shipment['country']['country_name']],
                            "destination_state" =>  $shipment['state']['state_name'] ?? $shipment['receiver_state_name'],
                            "destination_city" =>  $shipment['city']['city_name'] ?? '',
                            "destination_postal_code" => $package->receiver_postal_code,
                            "destination_raw_location" => $shipment->receiver_address_1
                        ];
                        // $result = createTrackingAfterShip($value['tracking_code'], $data_tracking);
                        $result = createTracktry($value['tracking_code'], $data_tracking);
                        if ($result['status'] != 200) {
                            // $text = $text . 'Mã hawb: ' . $value['hawb'] . ' khởi tạo tracking thất bại do ' . $result['message'];
                            // $package_data['package_tracking_code'] = '';
                        } else {
                            // $package_data['carrier_code'] = $result['data']['carrier_code'] ?? "";
                            $package_data['carrier_code'] = $result['carrier_code'] ?? "";
                            $package_data['branch_connect'] = $result['carrier_code'] ?? "";
                        };
                    } else {
                        $_package = Package::where('package_tracking_code', $request_data['create_tracking'])->whereNotNull('carrier_code', 'branch_connect')->first();
                        if ($_package) {
                            $package_data['carrier_code'] = $_package->carrier_code;
                            $package_data['branch_connect'] = $_package->branch_connect;
                        }
                    }
                    Package::find($package->package_id)->update($package_data);
                } else {
                    $text = $text .  ' Không tìm thấy mã hawb: ' . $value['hawb'] . '. ';
                }
            }

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'icon' => empty($text) ? 'success' : 'warning',
                    'title' => 'Thông báo',
                    'text' => empty($text) ? 'Cập nhật thành công' : $text
                ],
                'data' =>  $package = Package::where('sm_tracktry_id', $request_data['sm_tracktry_id'])->get()
            ]);
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }


    public function removePackagesMawbByTrackTry(Request $request)
    {
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
        try {
            DB::beginTransaction();
            $package = Package::isShow()->where('package_hawb_code', $request_data['hawb_code'])->first();
            if ($package) {
                if (
                    Package::isShow()->where('package_tracking_code', $package->package_tracking_code)
                    ->where('carrier_code', $package->carrier_code)->count() == 1
                    && !empty($package->package)
                    && !empty($package->package_tracking_code)
                ) {
                    deleteTracktry($package->package_tracking_code, $package->carrier_code);
                }
                Package::find($package->package_id)->update([
                    'package_tracking_code' => '',
                    'carrier_code' => '',
                    'branch_connect' => '',
                    'sm_tracktry_id' => '',
                    'package_status' => SHIPMENT_STATUS_IMPORTED
                ]);
                updateStatusShipment($package['shipment_id']);
                HistoryScan::where('package_id', $package->package_id)
                    ->where('status', SHIPMENT_STATUS_EXPORED)->update([
                        'delete_flg' => DELETED,
                    ]);
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy package');
            }

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'icon' => 'success',
                    'title' => 'Thông báo',
                    'text' => 'Đã xóa package ra khỏi mawb'
                ],
            ]);
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function billPackage($package_code)
    {
        // $package_trackings = PackageTracking::isShow()->where('package_code', $package_code)->first();
        $package = Package::isShow()->where('package_code', $package_code)->first();

        if ($package) {
            return view('mawb-manager.bill-package', compact('package'));
        } else {
            abort('404');
        }
    }



    public function handlePackageTrackingStatus(Request $request)
    {

        $this->validate($request, [
            "package_code" => "required|max:225",
            "date" => "required|date",
            "address" => "required|max:225",
            "note" => "required|max:225",
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            if (isset($request_data['package_code'])) {
                $package = Package::where('package_code', $request_data['package_code'])->isShow()->first();
                if (!$package) {
                    return returnResponse(HTTP_NOT_FOUND, 'Mã package không tồn tại');
                }
            }

            DB::beginTransaction();

            PackageTracking::updateOrCreate([
                'package_tracking_id'   => isset($request_data['package_tracking_id']) ? $request_data['package_tracking_id'] : null,
            ], [
                'package_code'     => $request_data['package_code'],
                'package_tracking_date'     => Carbon::parse($request_data['date'])->format('Y-m-d H:i:s'),
                'package_tracking_address' => $request_data['address'],
                'package_tracking_note'    => $request_data['note'],
                'package_tracking_status'    => isset($request_data['package_tracking_status']) ? $request_data['package_tracking_status'] : 0,
            ]);
            DB::commit();

            return returnResponse(HTTP_OK, 'Cập nhật Tracking cho package thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function listPackageTrackingStatus(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $pg_trackings = PackageTracking::where('package_code', $request_data['package_code'])->isShow()->orderBy('package_tracking_date', 'desc')->isPaginate($request->paginate);

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $pg_trackings->toArray()['data'],
                ]);
            } else {
                $html = view('mawb-manager.ajax.package-tracking-status', compact('pg_trackings'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function deletePackageTrackingStatus(Request $request)
    {

        $this->validate($request, [
            "package_tracking_id" => "required|numeric",
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            if (isset($request_data['package_code'])) {
            }
            $package = PackageTracking::where('package_tracking_id', $request_data['package_tracking_id'])->isShow()->first();
            if (!$package) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy trạng thái này');
            }

            DB::beginTransaction();
            $package->update(["delete_flg" => DELETED]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa trạng thái cho package thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onUpdatePackageTrackingCode(Request $request)
    {

        $this->validate($request, [
            'package_tracking_code' => 'required',
            'package_code' => 'required'
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;



            $package = Package::where('package_code', $request_data['package_code'])->isShow()->first();
            if (!$package) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy gói hàng này');
            }
            // $tracking =  new AfterShip();
            // $res_tracking = $tracking->getTrackings($request_data['package_tracking_code']);
            $res_tracking = getTracktry($request_data['package_tracking_code'],  $package['carrier_code']);
            if ($res_tracking['status'] != HTTP_OK) {
                return returnResponse(HTTP_BAD_REQUEST, $res_tracking['message']);
            }

            $checkpoints = $res_tracking['data']['trackings'][0]['checkpoints'] ?? [];
            $carrier_code = $res_tracking['data']['trackings'][0]['slug'] ?? '';

            DB::beginTransaction();
            Package::find($package->package_id)->update([
                'package_tracking_code' => $request_data['package_tracking_code'],
                'carrier_code' => $carrier_code
            ]);
            if (count($checkpoints) > 0) {
                PackageTracking::where('package_code', $request_data['package_code'])
                    ->where('create_kind', 1)
                    ->update([
                        'delete_flg' => DELETED
                    ]);
            }
            foreach ($checkpoints as $key => $point) {
                $date = Carbon::parse($point['checkpoint_time'])->format('Y-m-d H:i:s');
                // $check_status = PackageTracking::where('package_code', $request_data['package_code'])->isShow()
                // ->where()
                PackageTracking::create([
                    'package_code' => $request_data['package_code'],
                    'package_tracking_date' => $date,
                    'package_tracking_address' => $point['location'],
                    'package_tracking_note' => $point['message'],
                    'create_kind' => 1,
                    'package_tracking_status' => $point['tag'] == 'Delivered' ? 1 : 0,
                ]);
            }
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật tracking code thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }


    // xuất file excel
    public function onExportExcelBillByType(Request $request)
    {
        $this->validate($request, [
            'excel_type' => 'required',
            'sm_tracktry_id' => 'required|numeric'
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $sm_tracking = ShipmentTracktry::isShow()->where('sm_tracktry_id', $request_data['sm_tracktry_id'])->first();

            if (!$sm_tracking) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy tracking');
            }

            $data = []; // Mảng foreach
            $others = []; // dữ liệu khác
            $column_widths = null; // căn chỉnh chiều rộng col
            $viewName = ''; // vd: template.excels.shipments
            $name_excel = $request_data['name_excel'] ?? 'export.xlsx'; // 

            $others = $sm_tracking;


            $packages = Package::isShow()
                ->where('sm_tracktry_id', $request_data['sm_tracktry_id'])
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id') // Chỉ cần join một lần bảng shipments
                ->leftJoin('branchs', 'shipments.shipment_branch_id', 'branchs.branch_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->leftJoin('countries', 'shipments.receiver_country_id', 'countries.country_id')
                ->leftJoin('states', 'shipments.receiver_state_id', 'states.state_id')
                ->leftJoin('cities', 'shipments.receiver_city_id', 'cities.city_id')
                ->where('shipments.delete_flg', NO_DELETE)
                ->where('shipments.active_flg', ACTIVE);

            switch ($request_data['excel_type']) {
                case 'manifest_au_24c':
                    $packages = $packages->select(
                        'packages.package_tracking_code',
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        'shipments.receiver_telephone',
                        'packages.package_quantity',
                        // M3
                        'packages.package_weight',
                        // KG
                        'shipments.shipment_goods_name',
                        'shipments.shipment_value',
                        // AUD
                        // PO
                        // TRA
                        // AK1
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        // 
                        'shipments.sender_city',
                        'shipments.sender_district',
                        // 700000
                        // VN
                        // 
                        // 
                        // 
                        // awb_code
                        // 
                        // 
                        // dest
                        'packages.updated_at',
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_au_24c';
                    break;

                case 'manifest_au_auk':
                    $packages = $packages->select(
                        'packages.package_tracking_code',
                        'packages.package_code',
                        'packages.bag_code',
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        'shipments.receiver_telephone',
                        'packages.package_quantity',
                        // M3
                        'packages.package_weight',
                        // KG
                        'shipments.shipment_goods_name',
                        'shipments.shipment_value',
                        // AUD
                        // PO
                        // TRA
                        // AK1
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        // 
                        'shipments.sender_city',
                        'shipments.sender_district',
                        // 700000
                        // VN
                        // 
                        // 
                        // 
                        // awb_code
                        // 
                        // 
                        // dest
                        'packages.created_at',
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_au_auk';
                    break;

                case 'manifest_au_solid':
                    $packages = $packages->select(
                        'packages.package_code as hawb_no',
                        'shipments.shipment_code as awb_no',
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        'shipments.receiver_telephone',
                        DB::raw('SUM(packages.package_quantity) as package_quantity'),
                        // M3
                        DB::raw('SUM(packages.package_weight) as package_weight'),
                        // DB::raw('SUM(CASE 
                        //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                        //     THEN packages.package_weight_actual 
                        //     ELSE packages.package_weight 
                        // END) as package_weight'),
                        // KG
                        'shipments.shipment_goods_name',
                        'shipments.shipment_value',
                        // AUD
                        // PO
                        // TRA
                        // AK1
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        // 
                        'shipments.sender_city',
                        'shipments.sender_district',
                        // 700000
                        // VN
                        // 
                        // 
                        // 
                        // awb_code
                        // 
                        // 
                        // dest
                        'packages.created_at',
                        'packages.bag_code',
                    )->groupBy('shipments.shipment_code')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_au_solid';
                    break;

                case 'manifest_ca_fb_triple':
                    $packages = $packages->select(
                        'packages.package_tracking_code',
                        'shipments.shipment_reference_code',
                        'packages.package_code',
                        // 
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        // 
                        // 
                        'shipments.sender_city',
                        // vn
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'shipments.receiver_address_3',
                        'cities.city_name',
                        'states.state_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        //
                        'shipments.receiver_telephone',
                        'packages.package_quantity',
                        // DB::raw('CASE 
                        //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                        //     THEN packages.package_weight_actual 
                        //     ELSE packages.package_weight 
                        // END'),
                        'packages.package_weight',
                        // LBS
                        //
                        // CAD
                        // DPP
                        'shipments.shipment_goods_name',
                        //
                        // 
                        // VN
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_ca_fb_triple';
                    break;

                case 'manifest_elc_us':
                    $packages = $packages->select(
                        'packages.package_code',
                        'packages.bag_code',
                        'packages.package_tracking_code',
                        'packages.package_weight',
                        // DB::raw('CASE 
                        //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                        //     THEN packages.package_weight_actual 
                        //     ELSE packages.package_weight 
                        // END'),
                        'countries.country_code',
                        'shipments.sender_contact_name',
                        'shipments.user_id',
                        'shipments.sender_address',
                        'shipments.sender_city',
                        'shipments.sender_district',
                        // 
                        'shipments.receiver_contact_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'cities.city_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_telephone',
                        'shipments.receiver_postal_code',
                        'shipments.shipment_goods_name',
                        // 
                        // 
                        // 
                        // 
                        // 
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_elc_us';
                    break;

                case 'manifest_ibc_us':
                    $packages = $packages->select(
                        'branchs.branch_name', // branch_name
                        'packages.carrier_code',
                        'shipments.shipment_goods_name',
                        'packages.package_weight',
                        // DB::raw('CASE 
                        //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                        //     THEN packages.package_weight_actual 
                        //     ELSE packages.package_weight 
                        // END'),
                        'packages.package_quantity',
                        'shipments.shipment_value',
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        'shipments.sender_city',
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        'shipments.shipment_reference_code',
                        'packages.bag_code',
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_ibc_us';
                    break;

                case 'manifest_ubi_us':
                    $packages = $packages->select(
                        // awb-code
                        'branchs.branch_name', // branch_name
                        // dest
                        // dest
                        'packages.carrier_code',
                        // 
                        // 
                        // 
                        // hawb no
                        'shipments.shipment_goods_name',
                        'packages.package_weight',
                        // DB::raw('CASE 
                        //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                        //     THEN packages.package_weight_actual 
                        //     ELSE packages.package_weight 
                        // END'),
                        //  K
                        'packages.package_quantity',
                        'shipments.shipment_value',
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        // 
                        'shipments.sender_city',
                        // shipments.sender_city
                        // 
                        // VN
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        'shipments.shipment_reference_code',
                        'packages.bag_code',
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.manifest_ubi_us';
                    break;

                case 'universal_london_custom_manifest_lhr':
                    $packages = $packages->select(
                        'packages.package_code',
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        'shipments.sender_district',
                        'shipments.sender_city',
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        'states.state_name',
                        'countries.country_name',
                        'shipments.shipment_goods_name',
                        'packages.package_weight',
                        'packages.package_quantity',
                        // 
                        // dest
                        'packages.bag_code',
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.universal_london_custom_manifest_lhr';
                    $column_widths =  [
                        'A' => 40,
                        'B' => 30,
                        'C' => 100,
                        'D' => 100,
                        'E' => 50,
                        'F' => 40,
                        'G' => 20,
                        'H' => 20,
                        'I' => 20,
                        'J' => 20,
                        'K' => 40,
                    ];
                    break;

                case 'universal_uds_format_file_lhr':
                    $packages = $packages->select(
                        // awb-code
                        'packages.package_code',
                        'packages.package_tracking_code',
                        'shipments.sender_company_name',
                        'shipments.sender_contact_name',
                        'shipments.sender_address',  // && ward
                        'shipments.sender_district',
                        'shipments.sender_city',
                        'shipments.receiver_postal_code',
                        // 70000
                        // VIET NAM
                        'shipments.sender_telephone',
                        'shipments.user_id', // email
                        'shipments.receiver_company_name',
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'cities.city_name',
                        // 
                        'states.state_name',
                        'countries.country_name',
                        'shipments.receiver_telephone',
                        // 
                        'packages.package_quantity',
                        'packages.package_weight',
                        'shipments.shipment_goods_name',
                        'shipments.shipment_value',
                        'packages.carrier_code',
                        'packages.package_length',
                        'packages.package_width',
                        'packages.package_height',
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.universal_uds_format_file_lhr';
                    break;

                case 'xborder_bbv_b2c_ams':
                    $packages = $packages->select(
                        // 'shipments_tracktry.awb_code',
                        'packages.package_code',
                        'packages.bag_code',
                        'packages.package_tracking_code',
                        // 
                        'shipments.sender_contact_name',
                        'shipments.sender_address',
                        // 
                        // 
                        'shipments.sender_city',
                        'shipments.sender_district',
                        // 
                        // VN
                        'shipments.receiver_contact_name',
                        'shipments.receiver_address_1',
                        'shipments.receiver_address_2',
                        'shipments.receiver_address_3',
                        'cities.city_name',
                        'states.state_code',
                        'shipments.receiver_state_name',
                        'shipments.receiver_postal_code',
                        'countries.country_code',
                        //
                        //
                        'packages.package_quantity',
                        'packages.package_weight',
                        // DB::raw('CASE 
                        //     WHEN services.promotion_flg = 2 AND packages.package_status != 0 
                        //     THEN packages.package_weight_actual 
                        //     ELSE packages.package_weight 
                        // END'),
                        // KGS
                        'shipments.shipment_value',
                        // EUR
                        // DDP
                        // 
                        // OM
                        // 
                        'shipments.shipment_goods_name',
                        // 
                        // 
                        // 
                        // 
                        // 
                        // 
                        // shipments.shipment_goods_name
                    )->groupBy('packages.package_id')->get();
                    $data = $packages;
                    $viewName = 'mawb-manager.excels.xborder_bbv_b2c_ams';
                    break;

                default:

                    break;
            }
            return Excel::download(new ExportFile($data, $viewName, $others, $column_widths), $name_excel);
            // if (!empty($viewName)) {
            // }else{
            //     return returnResponse(HTTP_BAD_REQUEST);
            // }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getPackageTrackingStatus(Request $request)
    {
        $this->validate($request, ['code' => 'required']);
        try {
            $request_data = $request->all();
            $package_trackings = PackageTracking::isShow()
                ->where('package_code', $request_data['code'])
                ->where('create_kind', 0)->get();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'icon' => 'success',
                    'title' => 'Thông báo',
                    'text' => 'Thành công',
                ],
                'data' => $package_trackings,
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => HTTP_BAD_REQUEST,
                'message' => [
                    'icon' => 'error',
                    'title' => 'Thông báo',
                    'text' => 'Xảy ra lỗi không xác dịnh',
                ],
                'data' => [],
            ]);
        }
    }

    public function onCreateTrackingOnePackage(Request $request)
    {
        $this->validate($request, [
            'tracking_code' => 'required',
            'package_code' => 'required',
        ]);
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $country_iso3 = config('constans.country_iso3');

            $tracking_code = $request_data['tracking_code'];
            $package_code = $request_data['package_code'];

            $package = Package::isShow()->where('package_code', $package_code)->first();
            $shipment = Shipment::find($package['shipment_id']);

            $data_tracking = [
                // Tracktry
                "customer_name" => $shipment['receiver_contact_name'] ?? "",
                "customer_email" => '',
                "customer_phone" => $shipment['receiver_telephone'] ?? '',
                "order_create_time" => $package['created_at'] ?? Carbon::now()->toDateTimeString(),
                "tracking_postal_code" => $shipment['receiver_postal_code'] ?? '',

                // Aftership
                "slug" => "",
                "tracking_number" => $tracking_code,
                "title" => $shipment['shipment_goods_name'] ?? 'product by achau',
                "smses" => [
                    $shipment['receiver_sms_phone'],
                    $shipment['sender_telephone'],
                ],
                "emails" => [],
                "order_id" => $package['package_code'],
                "order_number" => $package['package_id'],
                "order_id_path" => "",
                "custom_fields" => [
                    "product_name" => "",
                    "product_price" => ""
                ],
                "language" => "en",
                "order_promised_delivery_date" => Carbon::now()->format('Y-m-d'),
                "delivery_type" => "pickup_at_store",
                "pickup_location" => "achau Post",
                "pickup_note" => "achau",
                "origin_country_iso3" => "VNM",
                "origin_state" => getAreaVietName($shipment['sender_city'])['districts'][$shipment['sender_district']] ?? '',
                "origin_city" => getAreaVietName()['cities'][$shipment['sender_city']] ?? '',
                "origin_postal_code" => "",
                "origin_raw_location" => $shipment['sender_address'],
                "destination_country_iso3" => $country_iso3[$shipment['country']['country_name']],
                "destination_state" =>  $shipment['state']['state_name'] ?? $shipment['receiver_state_name'],
                "destination_city" =>  $shipment['city']['city_name'] ?? '',
                "destination_postal_code" => $package['receiver_postal_code'],
                "destination_raw_location" => $shipment['receiver_address_1']
            ];
            // $result = createTrackingAfterShip($tracking_code, $data_tracking);
            $result = createTracktry($tracking_code, $data_tracking);
            if ($result['status'] != 200) {
                $text = 'Khởi tạo tracking thất bại do ' . $result['message'];
                // $package_data['package_tracking_code'] = '';
                if ($tracking_code == $package['package_tracking_code'] && !empty($result['data']['carrier_code'])) {
                    Package::isShow()->where('package_code', $package_code)
                        ->update([
                            'carrier_code' => $result['data']['carrier_code'],
                        ]);
                    DB::commit();
                    return returnResponse(HTTP_OK, "Cập nhật lại carrier_code/Lỗi: " . $text, null, 'success');
                }
                return returnResponse(HTTP_BAD_REQUEST, $text);
            } else {
                $package_data['carrier_code'] = $result['data']['carrier_code'] ?? "";
                $package_data['branch_connect'] = $result['data']['carrier_code'] ?? "";
                Package::isShow()->where('package_code', $package_code)
                    ->update([
                        'package_tracking_code' => $tracking_code,
                        'carrier_code' => $result['carrier_code'],
                    ]);
            };

            return returnResponse(HTTP_OK, 'Cập nhật mã tracking mới thành công', null, 'success');
            DB::commit();
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }
}
