<?php

namespace App\Http\Controllers;

use App\Models\Service;
use App\Models\LogoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class LogoServiceController extends Controller
{
    public function displayLogoService()
    {
        return view('logo-service.list');
    }

    public function getDataLogoService(Request $request)
    {
        try {
            $is_api = request()->is('api/*');

            $request_data = $request->all();
            $filters = $request_data['filters'] ?? [];
            $filters = is_array($filters) ? $filters : json_decode($filters, true);

            $services = Service::with(['logoService'])
                ->isActive(ACTIVE)->isDelete(NO_DELETE)
                ->isFilters($filters)
                ->orderBy('services.created_at', 'desc')
                ->paginate($request_data['limit'], ['*'], 'page', $request_data['page']);

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $services,
                ]);
            }

            $html = view('logo-service.pagination', compact('services'))->render();
            return response()->json([
                'status' => HTTP_OK,
                'html' => $html,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleUploadLogoService(Request $request)
    {
        $request->validate([
            'service_id' => 'required|integer|exists:services,service_id',
        ]);

        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $user_id = $request->user()->user_id;
            $user_code = $request->user()->user_code;
            $service_id = $request_data['service_id'];

            $logo_service = LogoService::where('service_id', $service_id)->first();
            $request_data['logo'] = $logo_service['logo'] ?? '';

            $public_path = PUBLIC_PATH_LOGO_SERVICE . '/' . $user_code;
            if ($request->hasFile('logo')) {
                if (!file_exists(public_path($public_path))) {
                    mkdir(public_path($public_path), 0777, true);
                }
                $logo = $request->file('logo');
                $logo_name = time() . '.' . $logo->getClientOriginalExtension();
                $logo->move(public_path($public_path), $logo_name);
                $request_data['logo'] = $public_path . '/' .$logo_name;

                if (!empty($logo_service['logo'])) {
                    // if (file_exists(public_path($public_path . '/' . $logo_service['logo']))) {
                    //     unlink(public_path($public_path . '/' . $logo_service['logo']));
                    // }
                    // if (File::exists(public_path($public_path . '/' . $logo_service['logo']))) {
                    //     File::delete(public_path($public_path . '/' . $logo_service['logo']));
                    // }
                }
            }

            LogoService::updateOrCreate(
                [
                    'logo_service_id' => $logo_service->logo_service_id ?? null,
                ],
                [
                    'service_id' => $service_id,
                    'user_id' => $user_id,
                    'logo' => $request_data['logo'],
                    'title' => $request_data['title'],
                    'description' => $request_data['description'],
                ]
            );

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Cập nhật logo dịch vụ thành công',
                    'icon' => 'success',
                ],
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleDeleteLogoService(Request $request)
    {
        DB::beginTransaction();
        try {
            $request_data = $request->all();
            $user_id = $request->user()->user_id;
            $user_code = $request->user()->user_code;
            $service_id = $request_data['service_id'];

            $public_path = PUBLIC_PATH_LOGO_SERVICE . '/' . $user_code;

            $logo_service = LogoService::where('service_id', $service_id)
                ->where('user_id', $user_id)
                ->first();

            if (isset($logo_service['logo'])) {
                if (file_exists(public_path($public_path . '/' . $logo_service['logo']))) {
                    unlink(public_path($public_path . '/' . $logo_service['logo']));
                }

                $logo_service->delete();
            }

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công',
                    'text' => 'Xóa logo dịch vụ thành công',
                    'icon' => 'success',
                ],
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
