<?php

namespace App\Http\Controllers;

use App\Exports\CreditExport;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Debit;
use App\Models\Credit;
use App\Models\Package;
use App\Models\Service;
use App\Models\Shipment;
use App\Models\CreditItem;
use App\Exports\ExportFile;
use App\Models\HistoryScan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Session;

class CreditController extends Controller
{
    public function indexCredit()
    {
        $fwds = User::isShow()->get();
        return view('accounting-management.manager-credit.index', compact('fwds'));
    }

    public function onShowListCredit(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $filters = $request_data['filters'] ?? null;
            $credits = Credit::isShow()
                ->leftJoin('users as customer', 'credits.customer_id', 'customer.user_id')
                ->leftJoin('users as creator', 'credits.creater_id', 'creator.user_id')
                ->leftJoin('credit_items', 'credits.credit_id', 'credit_items.credit_id');

            $credits = $credits->select(
                'credits.*',
                'customer.user_code as customer_code',
                'customer.user_contact_name as customer_contact_name',
                'customer.user_company_name as customer_company_name',
                'creator.user_contact_name as creater_contact_name',
                DB::raw('SUM(value_package + goods_compensation) as credit_amount'),
            );

            if ($filters) {
                $date_range = $filters['date_range'] ?? null;
                $start_date = isset($date_range['start_date']) ? Carbon::parse($date_range['start_date'])->format('y-m-d h:m:s') : null;
                $end_date = isset($date_range['end_date']) ? Carbon::parse($date_range['end_date'])->endOfDay()->format('y-m-d h:m:s') : null;

                if ($start_date && $end_date) {
                    $credits = $credits->whereBetween("credits.created_at", [$start_date, $end_date]);;
                }

                if (isset($filters['keywords']) && $filters['keywords'] != '') {
                    $keywords = $filters['keywords'];
                    $credits = $credits->where(function ($query) use ($keywords) {
                        $query = $query->where('customer.user_code', 'like', '%' . $keywords . '%')
                            ->orWhere('customer.user_contact_name', 'like', '%' . $keywords . '%')
                            ->orWhere('customer.user_company_name', 'like', '%' . $keywords . '%')
                            ->orWhere('creator.user_contact_name', 'like', '%' . $keywords . '%');
                    });
                }

                if (isset($filters['credit_status']) && $filters['credit_status'] != '') {
                    $credits = $credits->where('credits.credit_status', $filters['credit_status']);
                }
            }

            $credits = $credits->orderBy('credits.created_at', 'desc')->groupBy('credits.credit_id')
            ->isPaginate($request->paginate);

            if ($is_api) {
                # code...
            } else {
                $html = view('accounting-management.manager-credit.ajax.table-credit', compact('credits'))->render();

                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleCreateCredit(Request $request)
    {
        $this->validate($request, [
            'fwd_id' => 'required|exists:users,user_id',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            Credit::create([
                'credit_no' => initializationCode('credits', 'credit_no', 'CD'),
                'customer_id' => $request_data['fwd_id'],
                'creater_id' => auth()->id(),
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Khởi tạo credit thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onCancelCredit(Request $request)
    {
        $this->validate($request, [
            'credit_no' => 'required|exists:credits,credit_no',
        ]);

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            if (!isGetAllData()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn không có đủ quyền hạn');
            }
            $credit = Credit::isShow()->where('credit_no', $request_data['credit_no'])->first();
            if (!$credit) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy credit');
            }
            DB::beginTransaction();
            Credit::find($credit->credit_id)->update(['delete_flg' => DELETED]);

            $data_logs = [
                'old_data' => null,
                'new_data' => $request_data
            ];
            onCreateAccountActivityLogs('Credit manager', 'Hủy lệnh credit', $data_logs);
            DB::commit();
            return returnResponse(HTTP_OK, 'Hủy lệnh credit thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function displayCreditDetail($credit_no)
    {
        $credit = Credit::isShow()->where('credit_no', $credit_no)->first();
        if (!$credit) {
            return viewError404();
        }



        return view('accounting-management.manager-credit.detail', compact('credit'));
    }

    private function onUpdatePricePackageCredit($credit)
    {
        $package_id = CreditItem::isShow()->where('credit_id', $credit['credit_id'])->get();
        foreach ($package_id as $key => $value) {
            $price_package = getPriceOnePackage($value['package_id']);
            Package::find($value['package_id'])->update(['package_price' => $price_package]);
            CreditItem::find($value['package_id'])->update(['value_package' => $price_package]);
        }
    }

    public function showListPackageInCredit(Request $request)
    {

        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $filters = $request_data['filters'] ?? null;
            $routeName = Route::currentRouteName();
            if ($routeName == 'credit.detail.packages-export') {
                $request_data['credit_id'] = $request->query('credit_id');
            }

            $packages = CreditItem::isShow()->where('credit_id', $request_data['credit_id'])
                ->leftJoin('packages', 'credit_items.package_id', 'packages.package_id')
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->leftJoin('branchs', 'shipments.shipment_branch_id', 'branchs.branch_id')
                ->leftJoin('states', 'shipments.receiver_state_id', 'states.state_id')
                ->leftJoin('cities', 'shipments.receiver_city_id', 'cities.city_id')
                ->leftJoin('services', 'shipments.shipment_service_id', 'services.service_id')
                ->leftJoin('countries', 'shipments.receiver_country_id', 'countries.country_id');

            $packages = $packages->select(
                'shipments.shipment_code',
                'packages.package_tracking_code',
                'shipments.receiver_company_name',
                'shipments.receiver_contact_name',
                'services.service_name',
                'branchs.branch_name',
                'countries.country_name',
                'states.state_name',
                'cities.city_name',
                // 1
                'packages.package_code',
                DB::raw("
                    CASE
                        WHEN packages.package_type = 0 THEN 'Carton'
                        WHEN packages.package_type = 1 THEN 'Pallet'
                        WHEN packages.package_type = 2 THEN 'Túi(Phong bì)'
                        ELSE 'Khác'
                    END as package_type_name
                "),
                'packages.package_charged_weight',
                'packages.package_price',
                'credit_items.credit_item_id',
                'credit_items.created_at',
                'credit_items.value_package',
                'credit_items.goods_compensation',
                'credit_items.credit_item_note',
                DB::raw('(goods_compensation + packages.package_price) as total_value'),
            );

            if (isset($filters)) {
                if (isset($filters['keywords']) && $filters['keywords'] != '') {
                    $keywords = $filters['keywords'];
                    $packages = $packages->where(function ($query) use ($keywords) {
                        return $query->where('packages.package_tracking_code', 'like', '%' . $keywords . '%')
                            ->orWhere('packages.package_code', 'like', '%' . $keywords . '%')
                            ->orWhere('shipments.shipment_code', 'like', '%' . $keywords . '%')
                            ->orWhere('services.service_name', 'like', '%' . $keywords . '%')
                            ->orWhere('countries.country_name', 'like', '%' . $keywords . '%');
                    });
                }
            }
            $packages = $packages->groupBy('credit_items.credit_item_id');

            $credit = Credit::find($request_data['credit_id']);

            if ($routeName == 'credit.detail.packages-export') {

                $export_name = $credit['credit_no'] . '.xlsx';
                $data = [];

                $data['credit'] = $credit;
                $data['creater'] = User::find($credit['creater_id']);
                $data['customer'] = User::find($credit['customer_id']);
                $data['packages'] = $packages->get();
                return Excel::download(new CreditExport($data), $export_name);
            }

            $packages = $packages->isPaginate($request->paginate);
            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $packages->toArray()['data'],
                    'credit' => $credit,
                ]);
            } else {
                $html = view('accounting-management.manager-credit.ajax.table-packages', compact('packages', 'credit'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function handleAddPackageInCredit(Request $request)
    {
        $this->validate($request, [
            'credit_id' => 'required|numeric',
            'package_code' => 'required',
            'goods_compensation' => 'required|numeric|min:0',
            'credit_item_note' => 'required|max:225',
        ]);

        try {
            $request_data = $request->all();
            $credit = Credit::find($request_data['credit_id']);
            if (!$credit) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy credit', null, 'warning');
            }
            if ($credit['credit_status'] != 0) {
                return returnResponse(HTTP_BAD_REQUEST, 'Đơn này đã được thanh toán, không thể thao tác', null, 'warning');
            }
            $package = Package::isShow()->where('package_code', $request_data['package_code'])
                ->leftJoin('shipments', 'packages.shipment_id', 'shipments.shipment_id')
                ->where('shipments.delete_flg', NO_DELETE)->where('shipments.active_flg', ACTIVE)
                ->select('shipments.*', 'package_id', 'package_charged_weight')->first();

            if (!$package) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy gói hàng', null, 'warning');
            }

            if ($credit['customer_id'] != $package['user_id']) {
                return returnResponse(HTTP_BAD_REQUEST, 'Gói hàng này không phải của công ty được chọn', null, 'warning');
            }

            if (CreditItem::isShow()->where('credit_id', $credit->credit_id)->where('package_id', $package->package_id)->exists()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Bạn đã thêm gói hàng này vào credit rồi', null, 'warning');
            }

            $value_package = getPriceOnePackage($package['package_id']);

            DB::beginTransaction();
            Package::find($package['package_id'])->update(['package_price' => $value_package]);
            CreditItem::create([
                'credit_id' => $credit->credit_id,
                'package_id' => $package->package_id,
                'value_package' => $value_package,
                'goods_compensation' => $request_data['goods_compensation'],
                'credit_item_note' => $request_data['credit_item_note'] ?? '',
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Thêm gói hàng vào credit thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onRemovePackageInCredit(Request $request)
    {
        $this->validate($request, ['credit_item_id' => 'required']);
        try {
            $request_data = $request->all();
            $credit_item = CreditItem::isShow()->where('credit_item_id', $request_data['credit_item_id'])->first();
            if (!$credit_item) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy gói hàng trong credit');
            }
            $credit = Credit::find($credit_item->credit_id);
            if ($credit['credit_status'] != 0) {
                return returnResponse(HTTP_BAD_REQUEST, 'Đơn này đã được thanh toán, không thể thao tác tiếp', null, 'warning');
            }
            DB::beginTransaction();
            $credit_item->delete();
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
    public function onUpdateValuePackage(Request $request)
    {
        $this->validate($request, [
            'credit_item_id' => 'required',
            'value_package' => 'required|numeric|min:0',
        ]);
        try {
            $request_data = $request->all();
            $credit_item = CreditItem::isShow()->where('credit_item_id', $request_data['credit_item_id'])->first();
            if (!$credit_item) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy gói hàng trong credit');
            }
            $credit = Credit::find($credit_item->credit_id);
            if ($credit['credit_status'] != 0) {
                return returnResponse(HTTP_BAD_REQUEST, 'Đơn này đã được thanh toán, không thể thao tác tiếp', null, 'warning');
            }
            DB::beginTransaction();
            $credit_item->update([
                'value_package' => $request_data['value_package'],
            ]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Cập nhật giá trị gói hàng thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function onPaymentCredit(Request $request)
    {
        $this->validate($request, ['credit_id' => 'required']);
        try {
            $request_data = $request->all();
            $credit = Credit::isShow()->where(['credit_id' => $request_data['credit_id']])
            ->orWhere('credit_no', $request_data['credit_id'])->first();

            if (!$credit) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy credit');
            }

            if (!CreditItem::isShow()->where('credit_id', $credit['credit_id'])->exists()) {
                return returnResponse(HTTP_BAD_REQUEST, 'Credit đang trống!', null, 'warning');
            }
            $check_images = isset($request_data['credit_images']) && $request_data['credit_images'] != '';
            if ($check_images) {
                $imagesBase64 = json_decode($request_data['credit_images']);
                $credit_images = convertImagesBase64ToDirectory($imagesBase64, PUBLIC_PATH_CREDIT_IMAGE);
                $credit->credit_images = json_encode($credit_images);
            } 
            $check_note = isset($request_data['credit_note']) && $request_data['credit_note'] != '';
            if ($check_note) {
                $credit->credit_note = $request_data['credit_note'];
            } 
            if (!$check_images && !$check_note) {
                return returnResponse(HTTP_BAD_REQUEST, 'Dữ liệu thanh toán bị thiếu', 'warning');
            }

            DB::beginTransaction();
            $credit->credit_status = $request_data['credit_status'] ?? 1;
            $credit->save();
            DB::commit();
            Session::flash('success', 'Thanh toán thành công');
            return returnResponse(HTTP_OK, 'Thanh toán thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }
}
