<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Branch;
use App\Models\CodeOtp;
use App\Models\Position;
use App\Jobs\SendEmailJobs;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\MenuByPosition;
use Illuminate\Support\Facades\DB;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;

class AuthController extends Controller
{
    public function displayLogin()
    {
        onActivePackageTrackingPending();
        
        // onInActiveAccountSaleAndFwdThirtyDaysAgo();
        if (auth()->check()) {
            return redirect()->route('dashboard');
        }

        $path = 'public/config/system-settings.json';
        $system_settings = json_decode(Storage::get($path), true);
        $admin_position_id = [collect($system_settings)->firstWhere('kind', SETTING_ADMIN_ACCOUNT)['position_id']];
        $fwd_position_id = [collect($system_settings)->firstWhere('kind', SETTING_FWD_ACCOUNT)['position_id']];
        $not_in_position_ids = array_merge($admin_position_id, $fwd_position_id);
        $positions = Position::isActive(ACTIVE)->isDelete(NO_DELETE)
            ->select('position_id', 'position_name')
            ->whereNotIn('position_id', $not_in_position_ids)
            ->get();

        $branches = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)
            ->select('branch_id', 'branch_name')
            ->get();

        $address = address();
        return view('auth.login', compact('address', 'positions', 'branches'));
    }

    public function getTermsOfUse(Request $request)
    {
        try {
            $path = 'public/config/terms-interface.json';
            $content = Storage::get($path);
            $content = isset($content) ? $content : '';

            $path = 'public/config/system-settings.json';
            $system_settings = json_decode(Storage::get($path), true);
            $admin_position_id = [collect($system_settings)->firstWhere('kind', SETTING_ADMIN_ACCOUNT)['position_id']];
            $fwd_position_id = [collect($system_settings)->firstWhere('kind', SETTING_FWD_ACCOUNT)['position_id']];
            $not_in_position_ids = array_merge($admin_position_id, $fwd_position_id);
            $positions = Position::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->select('position_id', 'position_name')
                ->whereNotIn('position_id', $not_in_position_ids)
                ->get();

            $branches = Branch::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->select('branch_id', 'branch_name')
                ->get();

            $address = address();

            return response()->json([
                'status' => HTTP_OK,
                'data' => $content,
                'positions' => $positions,
                'branches' => $branches,
                'address' => $address,
            ], HTTP_OK);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function getMenuForUser(Request $request){
        try {
            $menu = getMenuUserPosiotion();
            return response()->json($menu, $menu['status']);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    private function onLogin($credentials, $is_api = false)
    {

        if (Auth::attempt($credentials)) {
            $user = User::where('user_name', $credentials['user_name'])->first();
            if ($user->delete_flg == DELETED || $user->active_flg == INACTIVE) {
                Auth::logout();
                Session::forget('user_id_' . auth()->id());
                Session::forget('is_show_modal_' . auth()->id());
                Session::flash('success', 'Tài khoản này không còn hoạt động!');
                return response()->json([
                    'status' => 500,
                    'message' => 'Tài khoản của bạn tạm thời không hoạt động'
                ]);
            }
            onCreateAccountActivityLogs('Đăng nhập', $is_api ? 'Đăng nhập qua API' : 'Đăng nhập qua web');
            if ($is_api == true) {
                // $token_expires_at = $remember ? null : Carbon::now()->addMinutes(env('SESSION_LIFETIME'));
                $token_expires_at = null;
                $tokenResult = $user->createToken('userToken', ['*'], $token_expires_at);


                $path = 'public/config/system-settings.json';
                $system_settings = json_decode(Storage::get($path), true);
                $sale = collect($system_settings)->whereIn('kind', [SETTING_SHIPPER_ACCOUNT, SETTING_OPS_LEADER_ACCOUNT, SETTING_DRIVER_ACCOUNT])
                    ->where('position_id', $user->position_id)->first();
                return response()->json([
                    'status' => HTTP_OK,
                    'message' => "Đăng nhập thành công",
                    'token' => $tokenResult->plainTextToken,
                    'token_expires_at' => $tokenResult->accessToken->expires_at,
                    'data' => $user->toArray(),
                    'is_shipper' => isset($sale),
                    'system_settings' =>  $system_settings
                ]);
            } else {
                Session::flash('success', 'Đăng nhập thành công!');
                Session::flash('is_show_modal_' . auth()->id(), true);


                $menu = MenuByPosition::where('position_id', auth()->user()->position_id)->first();
                return response()->json([
                    'status' => 200,
                    'route' => route(collect(config('constans.menus'))->pluck('menus')->flatten(1)->keyBy('primary')[$menu->menu_key]['route']),
                ]);
            }
        } else {
            return response()->json([
                'status' => 401,
                'message' => 'Mật khẩu của bạn không đúng'
            ]);
        }
    }

    public function handleLogin(Request $request)
    {
        $this->validate(
            $request,
            ['email' => 'required|email|exists:users,user_name', 'password' => 'required'],
            [
                'email.required' => 'Nội dung không được để trống',
                'password.required' => 'Nội dung không được để trống',
                'email.email' => 'Email sai định dạng',
                'email.exists' => 'Email không tồn tại',
            ]
        );
        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        try {
            $user = User::where('user_name', $request_data['email'])->first();
            if ($user->delete_flg == DELETED || $user->active_flg == INACTIVE) {
                Auth::logout();
                Session::forget('user_id_' . auth()->id());
                Session::forget('is_show_modal_' . auth()->id());
                Session::flash('success', 'Tài khoản này không còn hoạt động!');
                return response()->json([
                    'status' => 500,
                    'message' => 'Tài khoản của bạn tạm thời không hoạt động'
                ]);
            }
            $path = 'public/config/system-settings.json';
            $data = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_ALL_ACCOUNT)['data'];
            $is_2FA_CODE = collect($data)->firstWhere('name', 'authenticator_all_user_status')['active_flg'] == ACTIVE ? true : false;


            if (!$is_2FA_CODE) {
                $credentials = [
                    'user_name' => $request_data['email'],
                    'password' => $request_data['password'],
                ];
                return $this->onLogin($credentials, $is_api);
            } else {
                if (!Hash::check($request->password, $user->password)) {
                    return response()->json([
                        'status' => 501,
                        'message' => 'Mật khẩu của bạn không đúng'
                    ]);
                }

                $model = "App\Models\User";
                $details['email'] = $request_data['email'];
                $details['token'] = Str::random(50);
                $details['code'] = rand(1000, 9999);
                $details['template'] = "template.emails.accuracy";

                CodeOtp::where('email', $details['email'])->where('model', $model)->where('kind', ACCURACY)->delete();
                DB::beginTransaction();

                $code_otp = CodeOtp::create([
                    'email' => $details['email'],
                    'model' => $model,
                    'code' => $details['code'],
                    'token' => $details['token'],
                    'kind' => ACCURACY,
                    'other' => $request->password,
                ]);
                dispatch(new SendEmailJobs($details, $code_otp));

                DB::commit();
                return response()->json([
                    'status' => 200,
                    'route' => route('accuracy.show', ['email' => $code_otp['email'], 'token' => $details['token']]),
                    'otp' => [
                        'code_otp_id' => $code_otp['code_otp_id'],
                        'email' => $code_otp['email'],
                        'token' => $details['token'],
                        'created_at' => $code_otp['created_at']
                    ],
                    'message' => [
                        'title' => 'Thành công!',
                        'text' => 'Mã OTP đã được gửi qua email của bạn!',
                        'icon' => 'success',
                    ]
                ]);
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => 500,
                'message' => 'Xảy ra lỗi dưới serve',
                'error' => $th->getMessage(),
            ]);
        }
    }

    

    public function displayAccuracy(Request $request)
    {
        try {
            $is_api = isset($request->is_api) ? true : false;
            if (auth()->check() && $is_api) {
                return redirect()->route('dashboard');
            }

            $email = $request->query('email');
            $token = $request->query('token');


            $data = User::leftJoin('code_otps', 'users.user_name', 'code_otps.email')
                ->where('email', $email)->where('token', $token)
                ->select('user_name', 'user_contact_name', 'code', 'token', 'code_otps.created_at')->first();

            if (!$data) {
                return abort(HTTP_NOT_FOUND);
            }

            return view('auth.accuracy', compact('data'));
        } catch (\Throwable $th) {
            if ($is_api) {
                return response()->json([
                    'status' => 500,
                    'message' => 'Xảy ra lỗi dưới serve',
                    'error' => $th->getMessage(),
                ]);
            } else {
                return abort(HTTP_NOT_FOUND);
            }
        }
    }

    public function handleAccuracy(Request $request)
    {
        try {
            $is_api = isset($request->is_api) ? false : true;

            $data = CodeOtp::where('email', $request->email)->where('token', $request->token)
                ->where('code', $request->code)->select('code_otp_id', 'created_at', 'other')->first();
            if (!$data) {
                return response()->json([
                    'status' => 401,
                    'message' => 'Mã xác thực không đúng'
                ]);
            }
            $now = Carbon::now();
            $createdAt = Carbon::parse($data->created_at);
            $minutesDifference = $now->diffInMinutes($createdAt);
            if ($minutesDifference > 15) {
                return response()->json([
                    'status' => 202,
                    'message' => 'Mã xác thực đã hết hạn vui lòng đăng nhập lại'
                ]);
            }
            $credentials = [
                'user_name' => $request->email,
                'password' => $data->other,
            ];
            return $this->onLogin($credentials, $is_api);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => 500,
                'message' => 'Xảy ra lỗi dưới serve',
                'error' => $th->getMessage(),
            ]);
        }
    }

    public function handleLogout()
    {
        Auth::logout();
        Session::forget('user_id_' . auth()->id());
        Session::forget('is_show_modal_' . auth()->id());
        return redirect()->route('login.show')->with('success', 'Đăng xuất thành công');
    }

    public function handleRegister(Request $request)
    {
        $request_data = $request->all();
        $is_customer = isset($request_data['is_customer']) ? json_decode($request_data['is_customer']) : false;
        if ($is_customer) {
            $this->validate($request, [
                'user_name' => 'required|email|unique:users,user_name',
                'password' => 'required|min:8|max:255',
                'confirm_password' => 'required|same:password',
                'user_company_name' => 'required',
                'user_contact_name' => 'required',
                'user_accountant_key' => 'required|min:10',
                'user_tax_code' => 'required',
                'user_phone' => 'required',
                'user_address_1' => 'required',
                'user_address_2' => 'required',
                'user_address_3' => 'required',
                'user_address' => 'required',
                'user_logo' => 'required',
            ], [
                'user_name.required' => 'Email không được để trống',
                'user_name.email' => 'Email không đúng định dạng',
                'user_name.unique' => 'Email đã tồn tại',
                'password.required' => 'Mật khẩu không được để trống',
                'password.min' => 'Mật khẩu phải có ít nhất 8 ký tự',
                'password.max' => 'Mật khẩu không được vượt quá 255 ký tự',
                'confirm_password.required' => 'Xác nhận mật khẩu không được để trống',
                'confirm_password.same' => 'Xác nhận mật khẩu không khớp',
                'user_company_name.required' => 'Tên công ty không được để trống',
                'user_contact_name.required' => 'Tên liên hệ không được để trống',
                'user_accountant_key.required' => 'Mã kế toán không được để trống',
                'user_accountant_key.regex' => 'Mã kế toán phải chứa ít nhất một chữ cái viết hoa, một chữ cái viết thường và một chữ số, và phải có ít nhất 10 ký tự',
                'user_tax_code.required' => 'Mã số thuế không được để trống',
                'user_phone.required' => 'Số điện thoại không được để trống',
                'user_address_1.required' => 'Tỉnh/Thành phố không được để trống',
                'user_address_2.required' => 'Quận/Huyện không được để trống',
                'user_address_3.required' => 'Phường/Xã không được để trống',
                'user_address.required' => 'Địa chỉ không được để trống',
                'user_logo.required' => 'Logo không được để trống',
            ]);
        } else {
            $this->validate($request, [
                'user_name' => 'required|email|unique:users,user_name',
                'password' => 'required|min:8|max:255',
                'confirm_password' => 'required|same:password',
                'position_id' => 'required|exists:positions,position_id',
                'branch_id' => 'required|exists:branchs,branch_id',
                'user_contact_name' => 'required',
                'user_phone' => 'required',
                'user_address' => 'required',
            ], [
                'user_name.required' => 'Email không được để trống',
                'user_name.email' => 'Email không đúng định dạng',
                'user_name.unique' => 'Email đã tồn tại',
                'password.required' => 'Mật khẩu không được để trống',
                'password.min' => 'Mật khẩu phải có ít nhất 8 ký tự',
                'password.max' => 'Mật khẩu không được vượt quá 255 ký tự',
                'confirm_password.required' => 'Xác nhận mật khẩu không được để trống',
                'confirm_password.same' => 'Xác nhận mật khẩu không khớp',
                'position_id.required' => 'Chức vụ không được để trống',
                'position_id.exists' => 'Chức vụ không tồn tại',
                'branch_id.required' => 'Chi nhánh không được để trống',
                'branch_id.exists' => 'Chi nhánh không tồn tại',
                'user_contact_name.required' => 'Tên liên hệ không được để trống',
                'user_phone.required' => 'Số điện thoại không được để trống',
                'user_address.required' => 'Địa chỉ không được để trống',
            ]);
        }
        DB::beginTransaction();
        try {
            $user_code_old = User::orderBy('user_code', 'DESC')->first()['user_code'];
            $request_data['user_code'] = 'KG' . (int) str_replace('KG', '', $user_code_old) + 1;
            $request_data['user_api_key'] = Str::random(50);

            if ($is_customer) {
                $path = 'public/config/system-settings.json';
                $position_id = collect(json_decode(Storage::get($path), true))->firstWhere('kind', SETTING_FWD_ACCOUNT)['position_id'];

                if ($request->file('user_logo')) {
                    $upload_path = public_path(PUBLIC_PATH_USER_LOGO);
                    if (!File::isDirectory($upload_path)) {
                        File::makeDirectory($upload_path, 0777, true, true);
                    }
                    $file_name = $request_data['user_code'] . '-' . $request->file('user_logo')->getClientOriginalName();
                    $request->file('user_logo')->move($upload_path, $file_name);
                    $request_data['user_logo'] = PUBLIC_PATH_USER_LOGO . '/' . $file_name;
                } else {
                    $request_data['user_logo'] = convertImagesBase64ToDirectory([$request_data['user_logo']], PUBLIC_PATH_USER_LOGO)[0];
                }
            }

            User::create([
                'user_name' => $request_data['user_name'],
                'password' => $request_data['password'],
                'user_company_name' => $is_customer ? $request_data['user_company_name'] : 'KANGO EXPRESS',
                'user_accountant_key' => $is_customer ? $request_data['user_accountant_key'] : '',
                'user_contact_name' => $request_data['user_contact_name'],
                'branch_id' => $is_customer ? Branch::first()->branch_id : $request_data['branch_id'],
                'user_tax_code' => $is_customer ? $request_data['user_tax_code'] : '',
                'user_phone' => $request_data['user_phone'],
                'user_address_1' => $is_customer ? $request_data['user_address_1'] : '',
                'user_address_2' => $is_customer ? $request_data['user_address_2'] : '',
                'user_address_3' => $is_customer ? $request_data['user_address_3'] : '',
                'user_address' => $request_data['user_address'],
                'user_latitude' => $request_data['user_latitude'] ?? null,
                'user_longitude' => $request_data['user_longitude'] ?? null,
                'user_code' => $request_data['user_code'],
                'user_logo' => $is_customer ? $request_data['user_logo'] : LOGO,
                'user_api_key' => $request_data['user_api_key'],
                'position_id' => $is_customer ? $position_id : $request_data['position_id'],
                'user_price_list_main_type' => USER_TYPE_F1,
                'user_pending_approval' => PENDING,
            ]);

            Session::flash('success', 'Đăng ký tài khoản thành công, đang đợi admin xác nhận, chúng tôi sẽ gửi thông tin đến email của bạn sớm nhất');
            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'title' => 'Thành công!',
                    'text' => 'Đăng ký tài khoản thành công, đang đợi admin xác nhận, chúng tôi sẽ gửi thông tin đến email của bạn sớm nhất',
                    'icon' => 'success',
                ],
            ], HTTP_OK);
        } catch (\Throwable $th) {
            DB::rollBack();
            if (isset($request_data['user_logo']) && File::exists($request_data['user_logo'])) {
                File::delete($request_data['user_logo']);
            }
            return showMessageError();
        }
    }

    public function displayForgotPassword(Request $request)
    {
        try {
            if (auth()->check()) {
                return redirect()->route('dashboard');
            }
            $email = $request->query('email');
            $token = $request->query('token');

            $data = CodeOtp::where('email', $email)->where('token', $token)
                ->leftJoin('users', 'code_otps.email', 'users.user_name')
                ->select('user_name', 'user_contact_name', 'code', 'token', 'code_otps.created_at')->first();
            // $data = User::where('email', $email)->where('token', $token)
            //     ->leftJoin('code_otps', 'users.user_name', 'code_otps.email')
            //     ->select('user_name', 'user_contact_name', 'code', 'token', 'code_otps.created_at')->first();

            if (!$data) {
                return abort(HTTP_NOT_FOUND);
            }
            // Thời điểm bắt đầu (có thể là một thời gian lưu trong cơ sở dữ liệu)
            $startTime = Carbon::parse($data['created_at']); // Thay thế bằng thời gian thực tế của bạn

            // Thời gian hiện tại
            $currentTime = Carbon::now();

            if ($currentTime->diffInMinutes($startTime) >= 5) {
                $data['mess-err'] = "Đường dẫn đã hết hạn";
            }

            return view('auth.forgot-password', compact('data'));
        } catch (\Throwable $th) {
            return abort(HTTP_NOT_FOUND);
        }
    }

    public function handleForgotPassword(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,user_name'
        ]);

        try {
            $model = "App\Models\User";
            $details['email'] = $request->email;
            $details['token'] = Str::random(50);
            $details['code'] = rand(1000, 9999);
            $details['template'] = "template.emails.forgot-password";

            CodeOtp::where('email', $details['email'])->where('model', $model)->where('kind', FORGOT_PASSWORD)->delete();

            DB::beginTransaction();

            $code_otp = CodeOtp::create([
                'email' => $details['email'],
                'model' => $model,
                'code' => $details['code'],
                'token' => $details['token'],
                'kind' => FORGOT_PASSWORD,
            ]);

            $details['route'] = route('forgot.show', ['email' => $code_otp['email'], 'token' => $details['token']]);

            dispatch(new SendEmailJobs($details, $code_otp));

            DB::commit();
            return response()->json([
                'status' => 200,
                'route' => $details['route'],
                'otp' => [
                    'code_otp_id' => $code_otp['code_otp_id'],
                    'email' => $code_otp['email'],
                    'token' => $details['token'],
                    'created_at' => $code_otp['created_at']
                ],
                'message' => "Đường dẫn đổi mật khẩu đã được gửi đến email của bạn"
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'message' => "Lỗi không xác định dưới serve",
                'errors' => $th->getMessage(),
            ]);
        }
    }

    public function handleLogoutApi(Request $request)
    {
        try {
            $request->user()->currentAccessToken()->delete();

            return response()->json([
                'status' => HTTP_OK,
                'message' => "Đăng xuất thành công"
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'message' => "Đăng xuất thất bại",
                'errors' => $th->getMessage()
            ]);
        }
    }

    public function displayProfile(Request $request)
    {
        if ($request->query('uid')) {
            $user_id = $request->query('uid');
        } else {
            $user_id = auth()->id();
        }
        $tab = 'update';
        if ($request->query('tab')) {
            $tab = $request->query('tab');
        }


        $users = User::where('delete_flg', NO_DELETE)->where('active_flg', ACTIVE)
            ->select('user_id', 'user_name')->get();
        return view('profile.index', compact('user_id', 'users', 'tab'));
    }

    public function getInfo(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $user = $is_api ? $request->user() : User::find($request->user_id);
            if ($user) {
                if ($user->active_flg == INACTIVE || $user->delete_flg == DELETED) {
                    if ($is_api) {
                        $request->user()->currentAccessToken()->delete();
                    } else {
                        auth()->logout();
                    }

                    return response()->json([
                        'status' => HTTP_INTERNAL_SERVER_ERROR,
                        'message' => "Tài khoản này hiện tại không thể hoạt động"
                    ]);
                }
                $user->name_avatar = customTextAvatar($user->user_name);
                $user->user_address_1 = (int) $user->user_address_1;
                $user->user_address_2 = (int) $user->user_address_2;
                $user->user_address_3 = (int) $user->user_address_3;
                return response()->json([
                    'status' => HTTP_OK,
                    'data' => $user
                ]);
            } else {
                return response()->json([
                    'status' => HTTP_NOT_FOUND,
                    'message' => 'Không tìm thấy đối tượng'
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'message' => 'Không tìm thấy đối tượng'
            ]);
        }
    }

    public function handleEdit(Request $request)
    {
        $this->validate($request, [
            'user_company_name' => 'required',
            'user_contact_name' => 'required',
            // 'user_tax_code' => 'required',
            'user_phone' => 'required',
            'user_address_1' => 'required',
            // 'user_accountant_key' => 'required',
            'user_address' => 'required',
        ]);
        try {
            DB::beginTransaction();

            $request_data = $request->all();

            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $path = 'public/config/system-settings.json';
            $user = User::find($request_data['user_id']);
            $old_data = $user;
            if (!$user) {
                return showReturnNotFound();
            }

            $old_img = $user->user_logo;
            $new_img = '';
            $old_signature = $user->user_signature;
            $new_signature = '';

            if ($is_api) {
                $name_img = $user->user_code . '-' . time();
                if ($request_data['user_logo']) {
                    $imgs = convertImagesBase64ToDirectory([$request_data['user_logo']], PUBLIC_PATH_USER_LOGO, $name_img);
                    $new_img = $imgs[0];
                }

                if ($request_data['user_signature']) {
                    $imgs = convertImagesBase64ToDirectory([$request_data['user_signature']], PUBLIC_PATH_USER_LOGO, $name_img);
                    $new_signature = $imgs[0];
                }
            } else {
                if ($request->file('user_logo')) {
                    $logo = $request->file('user_logo');
                    $path = PUBLIC_PATH_USER_LOGO; // Đường dẫn đích tùy chỉnh
                    $fileName = time() . '-' . $logo->getClientOriginalName(); // Lấy tên gốc của tệp
                    $logo->move(public_path(PUBLIC_PATH_USER_LOGO), $fileName);
                    // Lấy đường dẫn tới tệp video sau khi đã lưu trữ
                    $new_img = $path . '/' . $fileName;
                }

                // if ($request->file('user_signature')) {
                //     $signature = $request->file('user_signature');
                //     $path = PUBLIC_PATH_USER_SIGNATURE;
                //     $fileName = time() . '-' . $signature->getClientOriginalName(); // Lấy tên gốc của tệp
                //     $signature->move(public_path(PUBLIC_PATH_USER_SIGNATURE), $fileName);
                //     $new_signature = $path .  '/' . $fileName;
                // }
                if (!empty($request_data['user_signature'])) {
                    $new_signature = convertImagesBase64ToDirectory([$request_data['user_signature']], PUBLIC_PATH_USER_SIGNATURE, $user->user_code)[0];
                }
            }
            $user_update = [
                'user_company_name' => $request_data['user_company_name'],
                'user_contact_name' => $request_data['user_contact_name'],
                'user_tax_code' => $request_data['user_tax_code'] ?? null,
                'user_phone' => $request_data['user_phone'],
                // 'user_accountant_key' => $request_data['user_accountant_key'],
                'user_address_1' => $request_data['user_address_1'],
                'user_address_2' => $request_data['user_address_2'],
                'user_address_3' => $request_data['user_address_3'],
                'user_address' => $request_data['user_address'],
                'user_latitude' => $request_data['user_latitude'] ?? null,
                'user_longitude' => $request_data['user_longitude'] ?? null,
                'user_logo' => $new_img == '' ? $user->user_logo : $new_img,
                'user_signature' => $new_signature == '' ? $user->user_signature : $new_signature,
            ];

            $user->update($user_update);
            if (!empty($new_img) && File::exists($old_img)) {
                File::delete($old_img);
            }
            if (!empty($new_signature) && File::exists($old_signature)) {
                File::delete($old_signature);
            }

            $data = [
                'old_data' => $old_data,
                'new_data' => $user_update,
            ];

            onCreateAccountActivityLogs('Cập nhật profile', '', $data);
            DB::commit();

            return response()->json([
                'status' => HTTP_OK,
                'message' => "Cập nhật thành công",
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            if (isset($new_img) && File::exists($new_img)) {
                File::delete($new_img);
            }
            if (isset($new_signature) && File::exists($new_signature)) {
                File::delete($new_signature);
            }
            return showMessageError();
        }
    }

    public function handleUpdateUserAccountantKey(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'user_accountant_key' => 'required',
        ], [
            'current_password.required' => 'Vui lòng nhập mật khẩu hợp lệ',
            'user_accountant_key.required' => 'Vui lòng nhập mã kế toán mới',
        ]);

        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        DB::beginTransaction();
        try {
            $user_id = $is_api ? $request->user()['user_id'] : auth()->id();

            $user = User::find($user_id);

            $old_data = $user['user_accountant_key'];
            $new_data = $request_data['user_accountant_key'];
            if (Hash::check($request_data['current_password'], $user->password)) {
                $user->update([
                    'user_accountant_key' => $request_data['user_accountant_key'],
                ]);
            } else {
                return response()->json([
                    "status" => HTTP_UNPROCESSABLE_ENTITY,
                    "message" => [
                        'icon' => 'warning',
                        'title' => 'Thông báo',
                        'text' => 'Mật khẩu không đúng'
                    ],
                    "error" => [
                        "current_password" => [
                            "Mật khẩu không đúng"
                        ]
                    ]
                ]);
            }

            $data = [
                'old_data' => $old_data,
                'new_data' => $new_data,
            ];
            onCreateAccountActivityLogs('Cập nhật mã kế toán', null, $data);
            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => [
                    'icon' => 'success',
                    'title' => 'Thông báo',
                    'text' => 'Cập nhật thành công'
                ],
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return showMessageError();
        }
    }

    public function handleNewPass(Request $request)
    {
        $request->validate([
            'password' => 'required|min:8',
            'confirm_password' => 'required|same:password',
        ], [
            'password.min' => 'Mật khẩu phải ít nhất 8 kí tự',
            'confirm_password.same' => 'Xác nhận mật khẩu không đúng',
        ]);

        DB::beginTransaction();
        try {
            $request_data = $request->all();

            $user = User::where('user_name', $request_data['email'])->first();

            $old_data = $user['password_show'];
            $new_data = $request_data['password'];

            if ($user) {
                $user->update([
                    'password' => $request_data['password'],
                    'password_show' => $request_data['password'],
                ]);
            } else {
                return response()->json([
                    "status" => HTTP_UNPROCESSABLE_ENTITY,
                    "message" => "Mật khẩu cũ không đúng",
                    "errors" => [
                        "old_password" => [
                            "Mật khẩu cũ không đúng"
                        ]
                    ]
                ]);
            }
            $user->tokens()->delete();

            $data = [
                'old_data' => $old_data,
                'new_data' => $new_data,
            ];
            onCreateAccountActivityLogs('Tạo mật khẩu mới', "Lấy lại mật khẩu mới do quên mật khẩu", $data);

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => 'Cập nhật mật khẩu mới thành công',
                'route' => route('login.show')
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'message' => "Xảy ra lỗi không xác đinh!",
                'errors' => $th->getMessage(),
            ]);
        }
    }

    // Cập nhật lại mật khẩu 
    public function handleNewPassword(Request $request)
    {

        $request->validate([
            'old_password' => 'required',
            'password' => 'required|min:8',
            'confirm_password' => 'required|same:password',
        ], [
            'password.min' => 'Mật khẩu phải ít nhất 8 kí tự',
            'confirm_password.same' => 'Xác nhận mật khẩu không đúng',
        ]);

        $request_data = $request->all();
        $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

        DB::beginTransaction();
        try {

            $user_id = $is_api ? $request->user()['user_id'] : auth()->id();
            $user = User::find($user_id);
            if (Hash::check($request_data['old_password'], $user->password)) {

                $data = [
                    'old_data' => $user['password_show'],
                    'new_data' =>  $request_data['password'],
                ];
                onCreateAccountActivityLogs('Cập nhật mật khẩu', null, $data);

                $user->update([
                    'password' => $request_data['password'],
                    'password_show' => $request_data['password'],
                ]);
            } else {
                return response()->json([
                    "status" => HTTP_UNPROCESSABLE_ENTITY,
                    "message" => "Mật khẩu cũ không đúng",
                    "error" => [
                        "old_password" => [
                            "Mật khẩu cũ không đúng"
                        ]
                    ]
                ]);
            }

            $user->tokens()->delete();

            DB::commit();
            return response()->json([
                'status' => HTTP_OK,
                'message' => 'Cập nhật mật khẩu mới thành công'
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => HTTP_INTERNAL_SERVER_ERROR,
                'message' => "Xảy ra lỗi không xác đinh!",
                'errors' => $th->getMessage(),
            ]);
        }
    }
}
