<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Branch;
use App\Models\Country;
use App\Models\Package;
use App\Models\Service;
use App\Models\ApiDocument;
use App\Models\City;
use Illuminate\Http\Request;
use App\Models\PackageTracking;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use App\Models\CountryAndServiceLink;
use App\Models\State;
use Illuminate\Support\Facades\Session;

class ApiDocumnetController extends Controller
{
    public function onCheckApiKey(Request $request)
    {
        try {
            // Lấy giá trị từ header
            $apiKey = $request->header('api-key');

            // Kiểm tra giá trị
            if (!$apiKey) {
                return [
                    'status' => HTTP_UNAUTHORIZED,
                    'message' => "'api-key' bị thiếu",
                ];
            }

            $user = User::where('user_api_key', $apiKey)->first();
            if (!$user) {
                return [
                    'status' => HTTP_UNAUTHORIZED,
                    'message' => "'api-key' không đúng",
                ];
            }

            if ($user['delete_flg'] == DELETED || $user['active_flg'] == INACTIVE) {
                return [
                    'status' => HTTP_UNAUTHORIZED,
                    'message' => "Tài khoản của của bạn không còn hoạt động",
                ];
            }

            return [
                'status' => HTTP_OK,
                'user' => $user,
            ];
        } catch (\Throwable $th) {
            return [
                'status' => HTTP_UNAUTHORIZED,
                'message' => $th->getMessage(),
            ];
        }
    }

    public function displayIndex()
    {

        return view('api-document.index');
    }

    public function showListApi(Request $request)
    {
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;
            $filters = $request_data['filters'] ?? null;
            $documents = ApiDocument::isShow()->isFilters($filters);

            $documents = $documents->isPaginate($request->paginate);

            if ($is_api) {
                return response()->json([
                    'status' => HTTP_OK,
                    'documents' => $documents->toArray()['data'],
                ]);
            } else {
                $html = view('api-document.ajax.list-document', compact('documents'))->render();
                return response()->json([
                    'status' => HTTP_OK,
                    'html' => $html,
                ]);
            }
        } catch (\Throwable $th) {
            return showMessageError();
        }
        return view('api-document.index');
    }

    public function displayDetail($slug)
    {
        $document = ApiDocument::isShow()->where('slug', $slug)->first();
        if (!$document) {
            return viewError404();
        }
        return view('api-document.detail', compact('document'));
    }

    public function displayCreate()
    {
        return view('api-document.handle');
    }

    public function displayUpdate($slug)
    {
        $document = ApiDocument::isShow()->where('slug', $slug)->first();
        if (!$document) {
            return viewError404();
        }
        return view('api-document.handle', compact('document'));
    }

    public function handleCreateOrUpdate(Request $request)
    {
        $this->validate($request, [
            'title' => 'required|max:225',
            'short_des' => 'required|max:325',
            'content' => 'required',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            $text_mess = 'thành công';
            DB::beginTransaction();
            if (isset($request_data['api_id'])) {
                $document = ApiDocument::isShow()->find($request_data['api_id']);
                if (!$document) {
                    return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
                }
                $path_content = writeFileText($request_data['content'], $document->path_content)['name'];
                $slug = setSlug($request_data['title']) . '-' . $document->api_document_id;
                $document->update([
                    'slug' => $slug,
                    'title' => $request_data['title'],
                    'short_des' => $request_data['short_des'],
                    'path_content' => $path_content,
                ]);
                $text_mess = 'Cập nhật document ' . $text_mess;
            } else {
                $document = ApiDocument::latest()->first();
                $slug = setSlug($request_data['title']) . '-' . (($document->api_document_id ?? 0) + 1);

                $path_name = PUBLIC_PATH_API_DOCUMENT_CONTENT . '/' . $slug . '.txt';
                $path_content = writeFileText($request_data['content'], $path_name)['name'];
                ApiDocument::create([
                    'slug' => $slug,
                    'title' => $request_data['title'],
                    'short_des' => $request_data['short_des'],
                    'path_content' => $path_content,
                ]);
                $text_mess = 'Tạo mới document ' . $text_mess;
            }
            DB::commit();
            Session::flash('success', $text_mess);
            return returnResponse(HTTP_OK, null, null, null, ['route' => route('api-document.show')]);
        } catch (\Throwable $th) {
            if (!isset($request_data['api_id']) && isset($path_content) && File::exists($path_content)) {
                File::delete($path_content);
            }
            DB::rollback();
            return showMessageError();
        }
    }

    public function deleteApiDocument(Request $request)
    {
        $this->validate($request, [
            'key' => 'required|numeric',
        ]);
        try {
            $request_data = $request->all();
            $is_api = isset($request_data['is_api']) ? json_decode($request_data['is_api']) : true;

            DB::beginTransaction();
            $document = ApiDocument::isShow()->find($request_data['key']);
            if (!$document) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy đối tượng');
            }
            $document->update(['delete_flg' => DELETED]);
            DB::commit();
            return returnResponse(HTTP_OK, 'Xóa document thành công', null, 'success');
        } catch (\Throwable $th) {
            DB::rollback();
            return showMessageError();
        }
    }

    public function onGetTracking(Request $request)
    {
        try {
            $check_auth = $this->onCheckApiKey($request);
            if ($check_auth['status'] != HTTP_OK) {
                return returnResponse($check_auth['status'], $check_auth['message']);
            }

            $code = $request->code;

            $package = Package::leftJoin('shipments', 'packages.shipment_id', '=', 'shipments.shipment_id')
                ->where(function ($query) use ($code) {
                    $query->where('packages.package_hawb_code', $code)
                        ->orWhere('shipments.shipment_code', $code);
                })
                ->select('packages.*')
                ->first();

            if (!isset($package)) {
                return response()->json([
                    'status' => 500,
                    'message' => 'Mã này không tồn tại.',
                ], 500);
            }

            $tracking_local = PackageTracking::isShow()->where('create_kind', 0)
                ->orderByDesc('package_tracking_date')
                ->where('package_code', $package['package_code'])
                ->select(
                    'package_tracking_date as time',
                    'package_tracking_address as location',
                    'package_tracking_note as title',
                )
                ->get()->toArray();

            $tracking_code = $package->package_tracking_code;
            $carrier_code = $package->carrier_code;
            $arr_labels = collect([
                "LABEL CREATED",
                "WE HAVE YOUR PARCEL",
                "Export Scan & Leave our warehouse",
                "Origin customs processin",
                "IN TRANSIT",
            ])->map(fn($label) => str($label)->lower())->toArray();
            if (empty($tracking_code)) {
                // Nếu chưa có tracking_code thì sẽ chỉ show các từ khóa có trong arr_labels
                foreach ($tracking_local as $key => $location) {
                    if (isset($location['tracking_status_note']) && !in_array(str($location['tracking_status_note'])->lower(), $arr_labels)) {
                        unset($tracking_local[$key]);
                    }
                }
            } else {
                // Check xem mawb đã Release chưa
                $hasStatusOne = DB::table('shipment_tracking_status')
                    ->where('shipment_tracktry_id', $package['sm_tracktry_id'])
                    ->where('delete_flg', 1)->where('active_flg', 1)
                    ->where(function ($db) {
                        $db->where('tracking_status', 1)
                            ->orWhere('tracking_status_note', 'like', '% CUSTOMS RELEASED');
                    })->exists();

                $tracking = [];
                if (!empty($tracking_code) && $hasStatusOne) {
                    // $tracking_code = '1Z2FE7250120281938';
                    // $carrier_code = 'ups';
                    $tracking = getTracktry($tracking_code, $carrier_code);

                    // $tracking = getTrackingAfterShip($tracking_code);
                }
                if (isset($tracking['status']) && $tracking['status'] != 200) {
                    // return response()->json([
                    //     'status' => 500,
                    //     'message' => 'Tải dữ liệu không thành công, vui lòng thử lại.',
                    // ], 500);
                }

                $tracking = $tracking['message']['trackings'][0] ?? [];
                $checkpoints = $tracking['checkpoints'] ?? [];

                usort($checkpoints, function ($a, $b) {
                    return strtotime($b['created_at']) - strtotime($a['created_at']);
                });

                foreach ($checkpoints as $checkpoint) {
                    $_checkpoint = [
                        'time' => Carbon::parse($checkpoint['created_at'])->format('Y-m-d H:i:s'),
                        'location' => $checkpoint['location'],
                        'title' => $checkpoint['subtag_message'],
                    ];
                    array_push($tracking_local, $_checkpoint);
                }
            }

            $data_convert = collect($tracking_local)->sortByDesc('time')->values()->toArray();
            return response()->json([
                'status' => HTTP_OK,
                'message' => 'Success',
                'trackings' => $data_convert,
            ]);
        } catch (\Throwable $th) {
            return showMessageError();
        }
    }

    public function onCreateShipment(Request $request)
    {
        $check_auth = $this->onCheckApiKey($request);
        if ($check_auth['status'] != HTTP_OK) {
            return returnResponse($check_auth['status'], $check_auth['message']);
        }
        $this->validate($request, [

            'shipment.receiver_country' => 'required|max:225',
            'shipment.receiver_state_name' => 'required|max:225',
            'shipment.receiver_city' => 'required|max:225',

            'shipment.shipment_service' => 'required|max:225',
            'shipment.shipment_branch' => 'required|max:225',
        ]);
        try {
            $shipment = $request->shipment;

            $user = $check_auth['user'];
            $shipment['sender_company_name'] = $user['user_company_name'];
            $shipment['sender_contact_name'] = $user['user_contact_name'];
            $shipment['sender_telephone'] = $user['user_phone'];
            $shipment['sender_city'] = $user['user_address_1'];
            $shipment['sender_district'] = $user['user_address_2'];
            $shipment['sender_ward'] = $user['user_address_3'];
            $shipment['sender_address'] = $user['user_address'];


            // Lấy thông tin dịch vụ
            $shipment_service = $shipment['shipment_service'];
            $service = Service::isShow()->where('service_name', 'like', $shipment_service)->first();
            if (empty($service)) {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy dịch vụ');
            }
            $shipment['shipment_service_id'] = $service['service_id'];
            $shipment_service_promotion_id = $service['promotion_flg'];

            // Lấy thông tin chi nhánh tạo
            $shipment_branch = $shipment['shipment_branch'];
            $branch = Branch::isShow()->where('branch_name', 'like', $shipment_branch)->first();
            // if (empty($branch)) {
            //     return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy chi nhánh');
            // }
            $shipment['shipment_branch_id'] = $branch['branch_id'] ?? $user['branch_id'];

            // Lấy thông thành phố người nhận
            $shipment_country = $shipment['receiver_country'];
            $country = Country::isActive(ACTIVE)->isDelete(NO_DELETE)->where('country_name', 'like', $shipment_country)->first();
            $shipment['receiver_country_id'] = $country['country_id'];

            $checkLinkService = CountryAndServiceLink::isCountry($country['country_id'])->isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('service_id', $shipment['shipment_service_id'])->exists();
            if (!$checkLinkService) {
                return returnResponse(HTTP_NOT_FOUND, "Dịch vụ " . $service['service_name'] . ' không được liên kết với quốc gia ' . $country['country_name']);
            }

            // Lấy khu vực người nhận
            $receiver_state_name = $shipment['receiver_state_name'];
            $state = State::isActive(ACTIVE)->isDelete(NO_DELETE)->where('state_name', 'like', $receiver_state_name)->first();
            if ($state) {
                $shipment['receiver_state_id'] = $state['state_id'];
            }

            // Lấy thành phố người nhận
            $receiver_city = $shipment['receiver_city'];
            $city = City::isActive(ACTIVE)->isDelete(NO_DELETE)
                ->where('country_id', $country['country_id'])
                ->where('city_name', 'like', '%' . $receiver_city . '%')->first();
            if ($city) {
                $shipment['receiver_city_id'] = $city['city_id'];
            } else {
                return returnResponse(HTTP_NOT_FOUND, 'Không tìm thấy thành phố người nhận');
            }

            $shipment['save_receiver_flg'] = false;
            $shipment['agree_terms_use_service'] = true;

            $request->merge([
                'api_document' => $user,
                'user' => $user,
                'shipment' => $shipment,
                'shipment_service_promotion_id' => $shipment_service_promotion_id,
            ]);

            $create_shipemnt = new ShipmentController();
            $result = $create_shipemnt->handleShipmentCreate($request);
            return $result;
        } catch (\Throwable $th) {
            return response()->json([
                'status' => '',
                'message' => $th->getMessage(),
            ]);
            return showMessageError();
        }
    }
}
