<?php

namespace App\Exports;

use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ExportStatement implements WithStyles, WithEvents
{
    protected $data;   // Dữ liệu được truyền vào

    public function __construct($data)
    {
        $this->data = $data;
    }

    public function styles(Worksheet $sheet)
    {
        $sheet->mergeCells('A1:D6');
        $sheet->mergeCells('E1:M1');
        $sheet->mergeCells('E2:M2');
        $sheet->mergeCells('E3:M3');
        $sheet->mergeCells('E4:M4');
        $sheet->mergeCells('E5:M5');
        $sheet->mergeCells('E6:M6');

        $sheet->mergeCells('B10:I10');
        $sheet->mergeCells('B11:I11');
        $sheet->mergeCells('B12:I12');
        $sheet->mergeCells('B13:I13');
        $sheet->mergeCells('B14:I14');

        $sheet->mergeCells('B16:C16');
        $sheet->mergeCells('D16:E16');
        $sheet->mergeCells('F16:G16');
        $sheet->mergeCells('H16:I16');
        $sheet->mergeCells('J16:K16');
        $sheet->mergeCells('L16:M16');


        return [
            'A1:D6' => ['alignment' => ['horizontal' => 'center', 'vertical' => 'center']],
            'M1:Q6' => [
                'font' => ['bold' => true, 'size' => 15],
                'alignment' => ['horizontal' => 'center', 'vertical' => 'center'],
            ],
            'E1' => [
                'bold' => true,
                'size' => 13,
                'color' => ['argb' => 'FFFF0000'], // Red color in ARGB format
            ],
            'E6' => [
                'font' => [
                    'bold' => true,
                    'size' => 12,
                    'color' => ['argb' => '000000'], // 
                ],
            ],
            'B9' => [
                'font' => [
                    'bold' => true,
                ],
            ],
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $borderStyle = [
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                            'color' => ['argb' => '000000'],
                        ],
                    ],
                ];

                $statement = $this->data['statement'];
                $creater = $this->data['creater'];
                $customer = $this->data['customer'];

                $path = 'public/config/info-debit.json';
                $info_debit = Storage::get($path);
                $info_debit = json_decode($info_debit, true);
                $info_debit = $info_debit['excels'];
                $header = $info_debit['header'];

                $sheet = $event->sheet->getDelegate();

                $drawing = new \PhpOffice\PhpSpreadsheet\Worksheet\Drawing();
                $drawing->setName('Logo');
                $drawing->setDescription('Company Logo');
                $drawing->setPath(public_path(LOGO));
                $drawing->setHeight(50);
                $drawing->setCoordinates('B2');
                $drawing->setOffsetX(10);
                $drawing->setOffsetY(10);
                $drawing->setWorksheet($sheet);;

                $sheet->setCellValue('E1', $header[0] ?? '');

                $sheet->setCellValue('E2', $header[1] ?? '');
                $sheet->setCellValue('E3', $header[2] ?? '');
                $sheet->setCellValue('E4', $header[3] ?? '');
                $sheet->setCellValue('E5', $header[4] ?? '');
                $sheet->setCellValue('E6', 'Prepared by: ' . $creater['user_contact_name']);

                $sheet->setCellValue('B9', 'Bill To');
                $sheet->setCellValue('B10', 'Customer: ' . $customer['user_company_name']);
                $sheet->setCellValue('B11', 'Adress: ' . $customer['user_address']);
                $sheet->setCellValue('B12', 'VAT/CODE: ' . $customer['user_phone']);
                $sheet->setCellValue('B13', 'Tel/Fax No: ' . $customer['user_phone']);
                $sheet->setCellValue('B14', 'Contact Name: ' . $customer['user_contact_name']);

                $debit_type_name = [
                    DEBIT_TYPE_DAY => 'DAY',
                    DEBIT_TYPE_WEEK1 => '1 WEEK',
                    DEBIT_TYPE_WEEK2 => '2 WEEK',
                    DEBIT_TYPE_MONTH => 'MONTH',
                ];
                $sheet->setCellValue('K10', 'No: ' . $statement['statement_code']);
                $sheet->setCellValue('K11', 'Date time: ' . formatFullDate($statement['created_at']));
                $sheet->setCellValue('K12', 'CREDIT TERM: ' . ($debit_type_name[$creater['user_debit_type']] ?? ''));
                $sheet->setCellValue('K13', 'CURRENCY: VND');

                // header
                $sheet->setCellValue('A16', 'STT');
                $sheet->setCellValue('B16', 'TRANSACTION');
                $sheet->setCellValue('D16', 'DATE');
                $sheet->setCellValue('F16', 'DESCRIPTION');
                $sheet->setCellValue('H16', 'AMOUNT');
                $sheet->setCellValue('J16', 'TOTAL');
                $sheet->setCellValue('L16', 'STATUS');
                $sheet->getStyle('A16:M16')->applyFromArray([
                    'font' => [
                        'bold' => true,
                        'color' => ['argb' => 'FFFFFFFF'], // White color for text in ARGB
                    ],
                    'fill' => [
                        'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                        'startColor' => ['argb' => 'FF007D88'],
                    ],
                ]);

                $debits = $this->data['debits'] ?? [];
                $num_start = 17;
                foreach ($debits as $key => $debit) {
                    $sheet->setCellValue('A' . $num_start, $key + 1);
                    $sheet->setCellValue('B' . $num_start, 'DEBIT');
                    $sheet->setCellValue('D' . $num_start, formatFullDate($debit['created_at']));
                    $sheet->setCellValue('F' . $num_start, $debit['debit_no']);
                    $sheet->setCellValue('H' . $num_start, formatNumber($debit['total_price']));
                    $sheet->setCellValue('J' . $num_start, formatNumber($debit['total_price']));
                    $sheet->setCellValue('L' . $num_start, $debit['status_label']);

                    $sheet->mergeCells("B$num_start:C$num_start");
                    $sheet->mergeCells("D$num_start:E$num_start");
                    $sheet->mergeCells("F$num_start:G$num_start");
                    $sheet->mergeCells("H$num_start:I$num_start");
                    $sheet->mergeCells("J$num_start:K$num_start");
                    $sheet->mergeCells("L$num_start:M$num_start");

                    $num_start++;
                }

                $sheet->setCellValue("A$num_start", 'PAYMENT REQUESTED');
                $sheet->mergeCells("A$num_start:I$num_start");

                $total_amount = collect($debits)->sum('total_price');
                $sheet->setCellValue("J$num_start", formatNumber($total_amount));
                $sheet->mergeCells("J$num_start:K$num_start");

                $sheet->getStyle("A$num_start:M$num_start")->applyFromArray([
                    'font' => [
                        'bold' => true,
                        'size' => 15,
                        'color' => ['argb' => '000000'], // 
                    ],
                ]);
                $sheet->getStyle("A16:M$num_start")->applyFromArray($borderStyle);


                // *Ghi chú
                $num_start = $num_start + 2;
                $sheet->setCellValue("A$num_start", '*Ghi chú');
                $sheet->getStyle("A$num_start")->applyFromArray([
                    'font' => [
                        'color' => ['argb' => 'FF0000'], // 
                    ],
                ]);
                $content_note = $info_debit['footer']['note'] ?? [];
                foreach ($content_note as $key => $note) {
                    $num_start = $num_start + 1;
                    $sheet->setCellValue("A$num_start", $note);
                    if (strpos($note, '*') === 0) {
                        $sheet->getStyle("A$num_start")->applyFromArray([
                            'font' => [
                                'color' => ['argb' => 'FF0000'], // 
                            ],
                        ]);
                    }
                }

                // * Thông Tin Tài Khoản chuyển khoản
                $num_start = $num_start + 2;
                onSetInfoDebitForExport($sheet, $customer, $num_start, $info_debit);

                $sheet->getStyle('A1:D6')->applyFromArray($borderStyle);
                $sheet->getStyle('E1:L6')->applyFromArray($borderStyle);
                $sheet->getStyle('M1:Q6')->applyFromArray($borderStyle);
                $sheet->getStyle('A16:M16')->applyFromArray($borderStyle);

                $sheet->getColumnDimension('A')->setWidth(9);
                $sheet->getColumnDimension('B')->setWidth(13);
                $sheet->getColumnDimension('C')->setWidth(20);
                $sheet->getColumnDimension('D')->setWidth(20);
                $sheet->getColumnDimension('E')->setWidth(30);
                $sheet->getColumnDimension('F')->setWidth(15);
                $sheet->getColumnDimension('G')->setWidth(15);
                $sheet->getColumnDimension('H')->setWidth(15);
                $sheet->getColumnDimension('I')->setWidth(15);
                $sheet->getColumnDimension('J')->setWidth(15);
                $sheet->getColumnDimension('K')->setWidth(15);
                $sheet->getColumnDimension('L')->setWidth(10);
                $sheet->getColumnDimension('M')->setWidth(15);
            },
        ];
    }
}
